<?php
/**
 *
 * @todo        Löschung (inklusive verknüpfter Daten) implementieren.
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  LivewireComponents
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 Wassilios Meletiadis <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Livewire\Core\EntityTypes;


use BplanBase\Globals\Livewire\Core\Global\BaseForm;
use BplanBase\Globals\Core\Registry;
use BplanBase\Globals\Repositories\EntityTypeRepository;
use BplanBase\Globals\Repositories\FieldDefinitionRepository;
use BplanBase\Globals\Repositories\FormSectionRepository;
use BplanBase\Globals\Repositories\LinkedEntityTypeRepository;
use BplanBase\Globals\Repositories\TenantLocaleRepository;
use BplanBase\Globals\Repositories\TenantRepository;
use BplanBase\Globals\Services\EntityTypeService;
use BplanBase\Globals\Services\LinkedEntityTypeService;
use Exception;
use Illuminate\Support\Str;


/**
 *
 * @version     1.0.0 / 2025-05-25
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Form extends BaseForm
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $dataStorages
     */
    public $dataStorages;


    /**
     * @var     array $dataStorageTypes
     */
    public $dataStorageTypes;


    /**
     * @var     array $deleteModeOptions
     */
    public $deleteModeOptions;


    /**
     * @var     string $entitytype_id
     */
    public $entitytype_id;


    /**
     * @var     array $fieldDefinitions
     */
    public $fieldDefinitions = [];


    /**
     * @var     array $formSections
     */
    public $formSections = [];

/*
    @todo   Ggf. wieder entfernen. Die Prüfung auf $this->linkID === null sollte ausreichen
*/
    /**
     * @var     bool $isLinked
     */
    public $isLinked = false;


    /**
     * @var     array $labels
     */
    public $labels;


    /**
     * @var     int|null $linkID
     */
    public $linkID;


    /**
     * @var     array $localeFields
     */
    public $localeFields = [];


    /**
     * @var         array   $main
     * @version     1.0.0 / 2025-05-25
     */
    public $main = [
        'active'          => true,
        'company'         => '',
        'customMask'      => '',
        'creatable'       => true,
        'deleteMode'      => 'soft',
        'editable'        => true,
        'editType'        => '',
        'historize'       => true,
        'htmlMask'        => '',
        'identifier'      => '',
        'internal'        => false,
        'journalize'      => true,
        'labels'          => '',
        'requireRelease'  => false,
        'resourceContext' => '',
        'resourceName'    => '',
        'storage'         => '',
        'storageType'     => '',
        'tenant_id'       => '',
        'textMask'        => '',
        'visible'         => true,
    ];


    /**
     * @var     array $resourceContexts
     */
    public $resourceContexts = [
        'core' => 'core',
        'data' => 'data',
        //'internal' => 'internal', // nur für Entwickler
    ];


    /**
     * @var    array $tabs
     */
    public $tabs = [
        'base',
        'status',
        'translations',
        'appearance',
        'form-sections',
        'field-definitions',
    ];


    /**
     * @var     array $tenant
     */
    public $tenant = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Überprüft den Status des EntityTypes
     *
     * Es wird überprüft ob der EntityType zum aktuellen oder ob er zu einem anderen Tenant gehört.
     * Wenn es sich um den EntityType eines anderen Tenants handelt wird zusätzlich noch geprüft ob
     * er bereits verknüpft ist.
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _checkState(): void
    {
        if ($this->exists === true) {
            $tenant_id = (int) Registry::get('tenantID');
            /*
            **  Prüfen ob es sich um einen eigenen EntityType handelt. */
            if ($this->main['tenant_id'] !== $tenant_id) {
                /*
                **  Bei fremden EntityTypes wird das Formular immer nur im Readonly-Modus angezeigt. */
                $this->readOnly = true;

                $this->linkID = LinkedEntityTypeRepository::getID($tenant_id, $this->mainID);

                if ($this->linkID === null) {
                    /*
                    **  Zusätzlichen Link-Button anzeigen, wenn es sich nicht um einen verknüpften
                    **  EntityType handelt. */
                    $this->additionalButtons[] = 'link';

                } else {
                    /*
                    **  Zusätzlichen Unlink-Button anzeigen, wenn es sich um einen verknüpften
                    **  EntityType handelt. */
                    $this->additionalButtons[] = 'unlink';
                    $this->isLinked = true;
                }
                /*
                **  Copy-Button anzeigen. */
                $this->additionalButtons[] = 'copy';
            }
        }
    } // _checkState()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initDataStorages(): void
    {
        $dataStorages = array_keys(config('dynamic-data-handling.data-storage'));

        $this->dataStorages = array_combine($dataStorages, $dataStorages);

    } // _initDataStorages()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initDataStorageTypes(): void
    {
        $dataStorageTypes = config('dynamic-data-handling.data-storage-types');

        $this->dataStorageTypes = array_combine($dataStorageTypes, $dataStorageTypes);

    } // _initDataStorageTypes()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initDeleteModeOptions(): void
    {
        $deleteModeOptions = config('dynamic-data-handling.delete-mode-options');

        $this->deleteModeOptions = array_combine($deleteModeOptions, $deleteModeOptions);

    } // _initDeleteModeOptions()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initFormSections(): void
    {
        $FormSections = FormSectionRepository::getByEntityTypeId($this->entitytype_id, orderBy: 'label');

        $this->formSections = array_combine($FormSections->pluck('id')->toArray(), $FormSections->toArray());

    } // _initFormSections()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initLocaleFields(): void
    {
        $tenant_id = Registry::get('tenantID');
        /*
            @todo   Ist das hier richtig, dass auch die inaktiven Locales ermittelt werden?
        */
        $tenantLocales = TenantLocaleRepository::determineTenantLocales($tenant_id, withInactive: true);

        $labels = json_decode($this->main['labels']);

        foreach ($tenantLocales as $keyIndex => $locale) {
            $this->labels[$locale] = isset($labels->{$locale}) ? $labels->{$locale} : null;

            $this->localeFields['labels'][] = [
                'label' => __('entity-type-form.label.main.label', ['locale' => $locale]),
                'name' => 'labels.'.$locale,
            ];
        }
    } // _initLocaleFields()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initTenant(): void
    {
        if ($this->exists === true) {
            /*
            **  Hier muss mit ignoreRestriction = TRUE gearbeitet werden, weil der User unter
            **  umständen keinen Zugriff auf den Tenant des EntityTypes hat.
            **
            **  @todo   Gegebenenfalls sollte die Methode TenantRepository::getWebModel() dahingehend
            **          geändert werden, dass bei ignoreRestriction = TRUE trotzdem die Tenants
            **          mit einbezogen werden, allerdings dann nicht mit den Tenants des Users
            **          sondern mit den Tenants aus dem ganzen Verbund.
            **          Alternativ könnte auch eine weitere Methode im TenantRepository angelegt
            **          werden, die diese Funktionalität implementiert. */
            $Tenant = TenantRepository::getWebModel($this->MainObject->tenant_id, ignoreRestriction: true);

            $this->tenant = [$this->MainObject->tenant_id => $Tenant->identifier];

        } else {
            $this->main['tenant_id'] = Registry::get('tenantID');

            $Tenant = TenantRepository::getWebModel($this->main['tenant_id']);

            $this->tenant = [$this->main['tenant_id'] => $Tenant->identifier];
        }
    } // _initTenant()


    /**
     *
     * @param       bool $saveWasCreate
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _onAfterSave(bool $saveWasCreate): void
    {
        parent::_onAfterSave($saveWasCreate);

        if ($saveWasCreate === true) {
            $this->tabs['field-definitions'] = [
                'active' => true,
                'identifier' => 'field-definitions',
                'label' => __($this->localeFile.'.tab-name.field-definitions'),
                'view' => 'field-definitions',
            ];
            $this->tabs['form-sections'] = [
                'active' => true,
                'identifier' => 'form-sections',
                'label' => __($this->localeFile.'.tab-name.form-sections'),
                'view' => 'form-sections',
            ];
        }
    } // _onAfterSave()


    /**
     *
     * @todo        Prüfen wie hier die Fehlermeldungen gesammelt und gemeinsam ausgegeben werden
     *              können. Am besten auch noch zusammen mit den Fehlern die der Validator findet.
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _onBeforeValidation(): void
    {
        parent::_onBeforeValidation();

        foreach ($this->labels as $locale => $label) {
            $label = trim($label);

            if ($label === '') {
                throw new Exception(__('validation.required', ['attribute' => __('entity-type-form.label.main.label', ['locale' => $locale])]));
            }
        }
        $this->main['labels'] = json_encode($this->labels);

    } // _onBeforeValidation()


    /**
     * Öffnet ein Modal-Fenster mit der Aufforderung zur Bestätigung einer Löschung
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function confirmCopy(): void
    {
        $this->showModal(self::MODAL_TYPE__CONFIRM, 'copy', __($this->localeFile.'.confirm.copy.message'), __($this->localeFile.'.confirm.copy.caption'));

    } // confirmCopy()


    /**
     * Öffnet ein Modal-Fenster mit der Aufforderung zur Bestätigung einer Löschung
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function confirmUnlink(): void
    {
        $this->showModal(self::MODAL_TYPE__CONFIRM, 'unlink', __($this->localeFile.'.confirm.unlink.message'), __($this->localeFile.'.confirm.unlink.caption'));

    } // confirmUnlink()


    /**
     * Verknüpft den EntityType mit dem aktuellen Tenant
     *
     * @throws      QueryException
     *
     * @throws      Exception
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function copy(): void
    {
        $this->_hideNotification();

        $currentIdentifier = $this->MainObject->identifier;
        /*
        **  Identifier aller verwendeten EntityTypes ermitteln um prüfen zu können ob es einen
        **  Konflikt mit dem Identifer des zu kopierenden EntityType gibt. */
        $AllEntityTypes = EntityTypeRepository::getOwnAndLinkedEntityTypes(withInactive: true);
        /*
        **  Für den Fall, dass der aktuelle EntityType ein verlinkter EntityType ist, wird das
        **  zugehörige Element aus der Collection entfernt. */
        $identifiers = array_flip($AllEntityTypes->where('id', '!=', $this->mainID)
            ->pluck('identifier')
            ->toArray());
//dd($identifiers);
        if (isset($identifiers[$currentIdentifier])) {
            $this->_showNotification(__('entity-type-form.error.identifier.unique', ['identifier' => $currentIdentifier]));

            return;
        }
        if ($this->isLinked === true) {
            $Service = new LinkedEntityTypeService();

            $Copy = $Service->copy($this->linkID);

        } else {
            $Service = new EntityTypeService();

            $Copy = $Service->copy($this->mainID);
        }
        $redirectRoute = $this->getRedirectRoute('Create');
        $redirectParams = $this->getRedirectParameters('Create', $Copy->id);

        $redirectParams['tab'] = 'base';

        redirect()->route($redirectRoute, $redirectParams);

    } // copy()


    /**
     * Verknüpft den EntityType mit dem aktuellen Tenant
     *
     * @throws      QueryException
     *
     * @throws      Exception
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function link(): void
    {
        $this->_hideNotification();

        $currentIdentifier = $this->MainObject->identifier;
        /*
        **  Identifier aller verwendeten EntityTypes ermitteln um prüfen zu können ob es einen
        **  Konflikt mit dem Identifer des zu kopierenden EntityType gibt. */
        $AllEntityTypes = EntityTypeRepository::getOwnAndLinkedEntityTypes(withInactive: true);
        /*
        **  Die Einschränkung aus der copy-Methode, dass der aktuelle EntityType für die Prüfung
        **  entfernt wird, muss hier nicht verwendet werden. */
        $usedIdentifiers = array_flip($AllEntityTypes->pluck('identifier')
            ->toArray());

        if (isset($usedIdentifiers[$currentIdentifier])) {
            $this->_showNotification(__('entity-type-form.error.identifier.unique', ['identifier' => $currentIdentifier]));

            return;
        }
        try {
            $Service = new LinkedEntityTypeService();

            $Service->create([
                'active' => 1,
                'entitytype_id' => $this->mainID,
            ]);
            $redirectRoute = $this->getRedirectRoute('Create');
            $redirectParams = $this->getRedirectParameters('Create', $this->mainID);

            redirect()->route($redirectRoute, $redirectParams);

        } catch (QueryException $Exception) {
            $message = $this->_getQueryExceptionMessage($Exception);

            $this->_showNotification($message);

        } catch (Exception $Exception) {
            $this->_showNotification($Exception->getMessage());
        }
    } // link()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function mount(): void
    {
        parent::mount();

        $this->_initDataStorages();
        $this->_initDataStorageTypes();
        $this->_initDeleteModeOptions();
        $this->_initLocaleFields();
        $this->_initTenant();

        if ($this->exists === true) {
            $this->entitytype_id = $this->MainObject->id;

            $this->fieldDefinitions = FieldDefinitionRepository::getByEntityTypeId($this->entitytype_id, orderBy: 'label')
                ->toArray();
// dd($this->fieldDefinitions);
            $this->_initFormSections();
            $this->_checkState();

        } else {
            $this->multiSaveButton = false;
            $this->redirectOnStandardSave = true;

            $this->tabs['field-definitions']['active'] = false;
            $this->tabs['form-sections']['active'] = false;
        }
    } // mount()


    /**
     * Verknüpft den EntityType mit dem aktuellen Tenant
     *
     * @throws      QueryException
     *
     * @throws      Exception
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function unlink(): void
    {
        $this->_hideNotification();

        try {
            $Service = new LinkedEntityTypeService();

            $Service->delete($this->linkID);

            $redirectRoute = $this->getRedirectRoute('Create');
            $redirectParams = $this->getRedirectParameters('Create', $this->mainID);
            /*
            **  Nach der Löschung wird auf den foreign-Tab in der Übersicht umgeleitet, weil der
            **  betroffene EntityType auf dem linked-Tab nicht mehr existiert. */
            $redirectParams['tab'] = 'foreign';

            redirect()->route($redirectRoute, $redirectParams);

        } catch (QueryException $Exception) {
            $message = $this->_getQueryExceptionMessage($Exception);

            $this->_showNotification($message);

        } catch (Exception $Exception) {
            $this->_showNotification($Exception->getMessage());
        }
    } // unlink()


    /**
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function updated($property): void
    {
        parent::updated($property);

        if ($property === 'main.identifier') {
            if ($this->main['resourceName'] === '') {
                $this->main['resourceName'] = Str::plural(strtolower($this->main['identifier']));
            }
        }
    } // updated()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class Form extends BaseForm {}
