<?php
/**
 * Migration Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Migrations
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */


use BplanBase\CodeGenerator\Loaders\CodeGenerator;
use BplanBase\Globals\Enums\DeleteMode;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;


/**
 * Migration Class
 *
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
return new class extends Migration
{


    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('entity_types');

    } // down()


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('entity_types', function (Blueprint $Table) {
            $Table->id();
            $Table->uuid('uuid');
            $Table->boolean('active')->default(1);

            $Table->timestamp('created_at');
            $Table->softDeletes('deleted_at');
            $Table->timestamp('updated_at');

            $Table->boolean('abstract')->default(0);
            $Table->boolean('creatable')->default(1)->comment('0 = Neuanlage nicht möglich, 1 = Neuanlage möglich');
            $Table->enum('delete_mode', array_column(DeleteMode::cases(), 'name'))->default(DeleteMode::SoftDelete->name);
            $Table->boolean('editable')->default(1)->comment('0 = nicht bearbeitbar, 1 = bearbeitbar');
            $Table->string('edit_type', 3)->default('111')->comment('Wie Binärwert: Position 1: Neuanlage, Position 2: Änderung, Position 3: Löschung. 0 = nicht erlaubt, 1 = erlaubt.');
            $Table->boolean('historize')->default(0);
            $Table->string('identifier', 65);
            $Table->boolean('internal')->default(1)->comment('internal = 1: Nur für interne Nutzung, können nicht abgerufen oder bearbeitet werden.');
            $Table->boolean('journalize')->default(0);
            $Table->json('labels')->nullable();
            $Table->string('main_type', 32);
            $Table->foreignId('master_id')->nullable()->constrained(table: 'entity_types', indexName: 'entity_types_FK_master_id');
            $Table->foreignId('model_id')->constrained(table: 'entity_type_models', indexName: 'entity_types_FK_model_id');
            $Table->foreignId('module_id')->constrained(table: 'modules', indexName: 'entity_types_FK_module_id');
            $Table->boolean('private')->default(1);
            $Table->boolean('require_release')->default(0)->comment('requireRelease = 1: EntityType-Daten müssen vor der Übertragung an Zielsysteme freigegeben werden.');
            $Table->string('resource_name', 50);
            $Table->foreignId('source_id')->nullable()->constrained(table: 'entity_types', indexName: 'entity_types_FK_source_id');
            $Table->string('sub_type', 32)->nullable();
            $Table->foreignId('tenant_id')->constrained(table: 'tenants', indexName: 'entity_types_FK_tenant_id');
            $Table->boolean('visible')->default(1);

            $Table->index('master_id',         'entity_types_FKI_master_id');
            $Table->index('model_id',          'entity_types_FKI_model_id');
            $Table->index('module_id',         'entity_types_FKI_module_id');
            $Table->index('source_id',         'entity_types_FKI_source_id');
            $Table->index('tenant_id',         'entity_types_FKI_tenant_id');
            $Table->unique('uuid',             'entity_types_UKI_uuid');
            $Table->index(['id', 'tenant_id'], 'entity_types_IDX_id_tenant');

            new CodeGenerator($Table, [
                'columns' => [
                    'active' => [
                        'sortable' => true,
                    ],
                    'identifier' => [
                        'sortable' => true,
                    ],
                    'master_id' => [
                        'sortable' => true,
                        'relationName' => 'master-entity-type',
                        'reverseRelationName' => 'sub-entity-types',
                    ],
                    'model_id' => [
                        'sortable' => true,
                    ],
                    'source_id' => [
                        'sortable' => true,
                        'relationName' => 'source-entity-type',
                        'reverseRelationName' => 'derived-entity-types',
                    ],
                    'tenant_id' => [
                        'guarded' => true,
                    ],
                ],
                'indexQueries' => [
                    'terminal' => '$Query->where(\'active\', \'=\', 1)',
                ],
                'routing' => [
                    'terminal' => 'readOnly',
                ],
                'traits' => [
                    'laravel.model' => [
                        'BplanBase\\Globals\Traits\\HasActiveScope',
                        'BplanBase\\Globals\Traits\\HasNoInternalScope',
                        'BplanBase\\Globals\Traits\\HasOrderedScope',
                    ],
                ],
                'uses' => [
                    'laravel.model' => [
                        'BplanBase\\Globals\\Scopes\\TenantScope'
                    ]
                ],

            ]);

        });
    } // up()


};
