<?php
/**
 * Package Service Provider Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Providers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals;


use BplanBase\Globals\Http\Middleware\CheckAccessLevelOrRoles;
use BplanBase\Globals\Http\Middleware\InitApiEnv;
use BplanBase\Globals\Http\Middleware\InitWebEnv;
use BplanBase\Globals\Http\Middleware\LogApiRequest;
use BplanBase\Globals\Registries\SeederRegistry;
use BplanBase\Globals\Services\HttpService;
use BplanBase\Globals\Support\LivewireComponentAutoDiscovery;
use Illuminate\Database\Events\QueryExecuted;
use Illuminate\Routing\Router;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;


/**
 * Package Service Provider Class
 *
 * @version     5.8.0 / 2025-05-29
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class PackageServiceProvider extends ServiceProvider
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_basePath
     */
    protected string $_basePath;


    /**
     * @var         string $_baseNamespace
     */
    protected string $_baseNamespace = 'BplanBase\\Globals';


    /**
     * Indicates if loading of the provider is deferred.
     *
     * @var         bool $defer
     */
    protected bool $defer = false;


    /**
     * @var         int $_packageKey
     */
    protected int $_packageKey = 1000;


    /**
     * @var         string $_packageName
     */
    protected string $_packageName = 'globals';


    /**
     * @var         string $_vendorPackageName
     */
    protected string $_vendorPackageName = 'bplan-base/globals';


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareSqlLogging(): void
    {
        if (Config::get('globals.db.sql-logging', false) === true) {
            /*
            **  Register log channel at runtime. */
            $this->_registerLogChannel();
            /*
            **  Log SQL queries. */
            DB::listen(function (QueryExecuted $Query) {
                Log::channel('sql')->info(
                    $Query->sql
                        ."\n".'  Bindings: '.((empty($Query->bindings)) ? '-' : print_r($Query->bindings, true))
                        ."\n".'  Runtime : '.$Query->time.'s'
                        ."\n"
                );
            });
        }
    } // _prepareSqlLogging()


    /**
     * Publish-Definitionen
     *
     * Aus der boot-Methode ausgelagert um den Code übersichtlich zu halten.
     *
     * @return 	    void
     *
     * @version     1.7.1 / 2025-05-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _publishes()
    {
        /*
        **  Dateien die bei jedem Setup neu gepublished werden. */
        $this->publishes([
            $this->_basePath.'/etc/setup/src/.gitignore'                                                              => base_path('.gitignore'),
            $this->_basePath.'/etc/setup/src/app/Http/Controllers/.gitignore'                                         => app_path('Http/Controllers/.gitignore'),
            $this->_basePath.'/etc/setup/src/composer.local.json.in'                                                  => base_path('composer.local.json.in'),
            $this->_basePath.'/etc/setup/src/config/jsonapi.php'                                                      => config_path('jsonapi.php'),
            $this->_basePath.'/etc/setup/src/database/migrations/0000_01_01_000100.globals.create_settings_table.php' => database_path('migrations/0000_01_01_000100.globals.create_settings_table.php'),
            $this->_basePath.'/etc/setup/src/database/migrations/0001_01_01_000000_create_users_table.php'            => database_path('migrations/0001_01_01_000000_create_users_table.php'),

        ], 'setup-forced');
        /*
        **  Dateien die nur einmalig gepublished werden sollen.
        **
        **  Alle Klassen zum Setting-Model würden eigentlich vom Code-Generator direkt im Projekt erzeugt
        **  werden. Da die zugehörige Logik aber individuell und bereits hier im Globals-Package definiert
        **  ist, werden im Projekt leere, abgeleitete und geschützte Klassen abgelegt.
        **  Weitere der hier gepublished Dateien ersetzen bereits vorhandene Dateien und dürfen aus diesem
        **  Grund nur einmalig gepublished werden. */
        $this->publishes([
            $this->_basePath.'/etc/setup/src/app/Exceptions/Handler.php'             => app_path('Exceptions/Handler.php'),
            $this->_basePath.'/etc/setup/src/app/Models/Setting.php'                 => app_path('Models/Setting.php'),
            $this->_basePath.'/etc/setup/src/app/Observers/SettingObserver.php'      => app_path('Observers/SettingObserver.php'),
            // $this->_basePath.'/etc/setup/src/app/Providers/AppServiceProvider.php'   => app_path('Providers/AppServiceProvider.php'),
            $this->_basePath.'/etc/setup/src/app/Repositories/SettingRepository.php' => app_path('Repositories/SettingRepository.php'),
            $this->_basePath.'/etc/setup/src/app/Services/SettingService.php'        => app_path('Services/SettingService.php'),
            $this->_basePath.'/etc/setup/src/bootstrap/app.php'                      => base_path('bootstrap/app.php'),
            $this->_basePath.'/etc/setup/src/config/project.php'                     => config_path('project.php'),
            $this->_basePath.'/etc/setup/src/database/seeders/DatabaseSeeder.php'    => database_path('seeders/DatabaseSeeder.php'),
            $this->_basePath.'/etc/setup/src/routes/api.php'                         => base_path('routes/api.php'),
            $this->_basePath.'/etc/setup/src/routes/web.php'                         => base_path('routes/web.php'),

        ], 'setup');

    } // _publishes()


    /**
     * Bootstrap the application events.
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-03-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerLogChannel()
    {
        /*
        **  Check whether the channel already exists to avoid duplicate entries. */
        if (!array_key_exists('sql', Config::get('logging.channels', []))) {
            Config::set('logging.channels.sql', [
                'driver' => 'daily',
                'path' => storage_path('logs/sql/sql.log'),
                'level' => 'info',
                'days' => 14,
                'replace_placeholders' => true,
            ]);
        }
    } // _registerLogChannel()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerMiddlewares(Router $Router)
    {
        $Router->aliasMiddleware('access-level.or.roles', CheckAccessLevelOrRoles::class);

    } // _registerMiddlewares()


    /**
     * Bootstrap the application events.
     *
     * @return      void
     *
     * @version     2.4.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function boot(Router $Router): void
    {

        $this->_prepareSqlLogging();
        /*
        **  Inform Laravel about translations. */
        $this->loadTranslationsFrom($this->_basePath.'/lang', $this->_packageName);

        if ($this->app->runningInConsole()) {
            /*
            **  Init publishes definitions. */
            $this->_publishes();
            /*
            **  Inform Laravel about migrations. */
            $this->loadMigrationsFrom($this->_basePath.'/database/migrations');

        } else {
            /*
            **  Inform Laravel about views. */
            $this->loadViewsFrom($this->_basePath.'/resources/views', $this->_packageName);
            /*
            **  Define a namespace for Blade components. */
            Blade::componentNamespace('BplanBase\\Globals\\View\\Components', $this->_packageName);
            LivewireComponentAutoDiscovery::register($this->_baseNamespace.'\\Livewire', $this->_basePath.'/src/Livewire', $this->_packageName);
        }
        $this->_registerMiddleWares($Router);
        /*
        **  Load package routes. */
        $this->loadRoutesFrom($this->_basePath.'/routes/api.php');
        $this->loadRoutesFrom($this->_basePath.'/routes/web.php');

    } // boot()


    /**
     * Register the service provider.
     *
     * @return      void
     *
     * @version     1.8.0 / 2025-05-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function register(): void
    {
        define('ENV_START', microtime(true));
        define('ENV_TIME', time());

        $this->_basePath = str_replace('\\', '/', realpath(__DIR__.'/..'));
        /*
        **  Load the package configuration. */
        $this->mergeConfigFrom($this->_basePath.'/config/'.$this->_packageName.'.php', $this->_packageName);
        $this->mergeConfigFrom($this->_basePath.'/config/dynamic-data-handling.php', 'dynamic-data-handling');
        // $this->mergeConfigFrom($this->_basePath.'/config/package-code-generator.php', 'code-generator');

        if ($this->app->runningInConsole()) {
            $this->commands([
                \BplanBase\Globals\Console\Commands\GlobalsSetup::class,
            ]);
            SeederRegistry::register($this->_packageKey, $this->_vendorPackageName, \BplanBase\Globals\Database\Seeders\DatabaseSeeder::class);
            SeederRegistry::register($this->_packageKey + 2000, $this->_vendorPackageName, \BplanBase\Globals\Database\Seeders\PostDatabaseSeeder::class);
        }
        $this->app->bind(HttpService::class, function() {
            return new HttpService();
        });

    } // register()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $group
     *              "api", "web"
     *
     * @param       string $getForUsage
     *              "append", "prepend"
     *
     * @param       array|string $append
     *
     * @param       array|string $prepend
     *
     * @return      array
     *
     * @version     1.1.0 / 2025-05-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getMiddlewares(string $group, string $getForUsage, array|string $append = [], array|string $prepend = []): array
    {
        $middlewares = [];

        switch ($group) {
            case 'api':
                if ($getForUsage === 'append') {
                    $middlewares = [
                        LogApiRequest::class,
                        InitApiEnv::class,
                    ];
                } else {
                    //
                }
                break;

            case 'web':
                if ($getForUsage === 'append') {
                    $middlewares = [
                        InitWebEnv::class,
                    ];
                } else {
                    //
                }
                break;

            default:
                //
        }
        if (!empty($append)) {
            $middlewares = array_merge($middlewares, Arr::wrap($append));
        }
        if (!empty($prepend)) {
            $middlewares = array_merge(Arr::wrap($prepend), $middlewares);
        }
        return $middlewares;

    } // getMiddlewares()


} // class PackageServiceProvider extends ServiceProvider {}
