<?php
/**
 * Migration Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Migrations
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */


use BplanBase\CodeGenerator\Loaders\CodeGenerator;
use BplanBase\Globals\Enums\ImportStatus;
use BplanBase\Globals\Enums\ImportTrigger;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;


/**
 * Migration Class
 *
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
return new class extends Migration
{


    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('import_logs');

    } // down()


    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('import_logs', function (Blueprint $Table) {
            $Table->id();

            $Table->timestamp('created_at');
            $Table->timestamp('updated_at');

            $Table->timestamp('begin')->nullable()->comment('Beginn der Verarbeitung.');
            $Table->string('context', 255);
            $Table->json('errors')->nullable();
            $Table->string('import_file_name', 255);
            $Table->string('import_file_path', 1000);
            $Table->foreignId('import_user_id')->nullable()->constrained(table: 'users', indexName: 'import_logs_FK_import_user_id');
            $Table->unsignedInteger('row_count')->comment('Anzahl verarbeiteter Datenzeilen (ohne Leer- und Kommentarzeilen).');
            $Table->decimal('runtime', 15, 10)->nullable()->comment('Laufzeit der Verarbeitung.');
            $Table->enum('status', [array_column(ImportStatus::cases(), 'name')]);
            $Table->enum('trigger', [array_column(ImportTrigger::cases(), 'name')]);
            $Table->foreignId('tenant_id')->constrained(table: 'tenants', indexName: 'import_logs_FK_tenant_id');

            $Table->index('import_user_id', 'import_logs_FKI_import_user_id');
            $Table->index('tenant_id',      'import_logs_FKI_tenant_id');

            new CodeGenerator($Table, [
                'columns' => [
                    'begin' => [
                        'sortable' => true,
                    ],
                    'context' => [
                        'sortable' => true,
                    ],
                    'errors' => [
                        'jsonType' => CodeGenerator::JSON_TYPE_ARRAY,
                    ],
                    'import_user_id' => [
                        'sortable' => true,
                    ],
                    'runtime' => [
                        'sortable' => true,
                    ],
                    'status' => [
                        'rules' => [
                            'required',
                            'Rule::enum(ImportStatus::class)',
                        ],
                        'sortable' => true,
                    ],
                    'trigger' => [
                        'rules' => [
                            'required',
                            'Rule::enum(ImportTrigger::class)',
                        ],
                        'sortable' => true,
                    ],
                ],
                'exclude' => [
                    'laravel.action.update',
                    'livewire.modal.create',
                    'livewire.modal.edit',
                ],
                'routing' => [
                    'data' => 'readOnly',
                    'terminal' => 'readOnly',
                ],
                'uses' => [
                    'jsonapi.request' => [
                        'BplanBase\\Globals\\Enums\\ImportStatus',
                        'BplanBase\\Globals\\Enums\\ImportTrigger',
                        'Illuminate\\Validation\\Rule',
                    ],
                    'laravel.form-request.store' => [
                        'BplanBase\\Globals\\Enums\\ImportStatus',
                        'BplanBase\\Globals\\Enums\\ImportTrigger',
                        'Illuminate\\Validation\\Rule',
                    ],
                    'laravel.form-request.update' => [
                        'BplanBase\\Globals\\Enums\\ImportStatus',
                        'BplanBase\\Globals\\Enums\\ImportTrigger',
                        'Illuminate\\Validation\\Rule',
                    ],
                ],
            ]);

        });
    } // up()


};
