<?php
/**
 * Middleware Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Middlewares
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Http\Middleware;


use App\Exceptions\BadRequestException;
use App\Models\User;
use BplanBase\Globals\Registries\Registry;
use BplanBase\Globals\Repositories\Core\TenantLocaleRepository;
use BplanBase\Globals\Repositories\Core\TenantRepository;
use BplanBase\Globals\Repositories\UserRepository;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Session;
use Symfony\Component\HttpFoundation\Response;


/**
 * Middleware Class
 *
 * @version     1.4.0 / 2025-06-30
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class InitWebEnv
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_configTenant
     */
    private $_configTenant;


    /**
     * @var     string $_singleTenantMode
     */
    private $_singleTenantMode = false;


    /**
     * @var     Request $_Request
     */
    private $_Request;


    /**
     * Die ID des Tenants zur UID aus dem Request-Header
     *
     * @var     string $_tenantId
     */
    protected $_tenantId;


    /**
     * Die UID des Tenants aus dem Request-Header
     *
     * @var     string $_tenantUid
     */
    protected $_tenantUid;


    /**
     * Die ID des Users, der über das Auth-Token aus dem Header ermittelt wurde
     *
     * @var     string $_userId
     */
    protected $_userId;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return
     *
     * @version     2.0.0 / 2025-05-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _changeTenant(Request $Request)
    {
        /*
        **  Die Übergabe des Parameters "change-tenant" initiiert einen Wechsel des aktuellen
        **  Tenants. Mit der übergebenen Tenant-ID werden die Daten in der Session neu
        **  initialisiert. */
        $tenantId = trim($Request->input('change-tenant'));

        $this->_initSession($tenantId);
        /*
        **  Abschließend werden die Query-Parameter ermittelt, der Parameter "change-tenant-id"
        **  wird entfernt und es wird ein Redirect auf die aktuelle Seite durchgeführt.
        **  Hintergrund dafür ist, dass die URL in der Adresszeile des Browsers hinterher
        **  wieder frei von dem Parameter ist. */
        $query = $Request->query();

        unset($query['change-tenant']);
        /*
            @todo   Das funktioniert leider nicht. Wenn die aktuelle View zum Beispiel ein
                    Formular ist, dann kommt es zu einem 404er, wenn der neue Tenant keinen
                    Zugriff auf das angezeigte Objekt hat.
                    Aus dem Grund wird erstmal auf das Dashboard umgeleitet.

                    Zur Lösung des Problems müsste erkannt werden ob die aktuelle Route sicher
                    für die Anzeige beim neuen Tenant ist.

        return redirect()
            ->route($Request->route()->getName(), $query);
        */
        return redirect()
            ->route('dashboard');

    } // _changeTenant()


    /**
     * Überträgt Session-Daten in die Registry
     *
     * @version     1.3.0 / 2025-06-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initRegistry(): void
    {
        Registry::set('masterTenantId', Session::get('masterTenantId'));
        Registry::set('tenantId', Session::get('tenantId'));
        Registry::set('tenantIds', Session::get('tenantIds'));
        Registry::set('tenantLocales', Session::get('tenantLocales'));
        Registry::set('tenants', Session::get('tenants'));
        Registry::set('User', Session::get('User'));
        Registry::set('userAccessLevel', Session::get('userAccessLevel'));
        Registry::set('userId', Session::get('userId'));

    } // _initRegistry()


    /**
     * Ermittelt Daten und legt diese zur späteren Verwendung in der Session ab
     *
     * @param       string $tenantId
     *              Die ID des Tenants dem der User direkt zugeordnet ist.
     *
     * @return      bool FALSE, wenn der User keine ihm zugeordneten Tenants oder keine aktiven
     *              UserTenantAccess-Verknüpfungen hat.
     *
     * @version     1.5.0 / 2025-06-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initSession(string $tenantId): bool
    {
        $AuthUser = auth()->user();
        /*
            @todo   Verwendung von $ignoreRestriction prüfen.
        */
        $tenants = TenantRepository::determineUserTenants($AuthUser);

        if (empty($tenants)) {
            /*
            **  Die Liste der Tenants kann leer sein, wenn der User zwar Verknüpfungen zu einem oder
            **  mehreren Tenants hat, diese aber alle inaktiv sind. */
            /*
                @todo   Das trifft nicht mehr zu. Es gibt keinen Aktivstatus bei Mandantenverknüpfungen.
            */
            return false;
        }
        /*
        **  Sicherstellen, dass die übergebene Tenant-ID auch in der Liste der Tenants des Users
        **  enthalten ist. */
        if (!isset($tenants[$tenantId])) {
            /*
            **  Grund für das Fehlen der übergebenen Tenant-ID ist, dass der User "restricted" ist
            **  und keine Verknüpfung zu seinem Tenant eingerichtet oder die Verknüpfung deaktiviert
            **  ist. In diesem Fall wird einfach der erste Tenant aus der Liste eingesetzt. */
            $tenantId = key($tenants);
        }
        $tenantLocales = TenantLocaleRepository::determineTenantLocales($tenantId);
        /*
        **  Der AuthUser ist vom Typ App\Models\User. Wenn dieser Typ in der Session abgelegt und
        **  anschließend in die Registry übertragen wird, dann erhält man beim Auslesen lediglich
        **  eine JSON-Repräsentation des Objekts (weil der Typ bei Deserialisieren nicht aufgelöst
        **  werden kann).
        **  Aus diesem Grund wird hier ein anderes User-Objekt erzeugt. */
        $User = UserRepository::getById($AuthUser->id, ignoreRestriction: true);

        Session::put('masterTenantId', empty($tenants[$tenantId]['master_id']) ? null : (int) $tenants[$tenantId]['master_id']);
        Session::put('tenantId', (int) $tenantId);
        Session::put('tenantIds', array_keys($tenants));
        Session::put('tenantLocales', $tenantLocales);
        Session::put('tenants', $tenants);
        Session::put('User', $User);
        Session::put('userAccessLevel', $User->accessLevel);
        Session::put('userId', (int) $User->id);

        return true;

    } // _initSession()


    /**
     * Handle an incoming request.
     *
     * @param       \Illuminate\Http\Request  $Request
     *
     * @param       Closure  $next
     *
     * @return      mixed
     *
     * @version     1.3.1 / 2025-06-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function handle(Request $Request, Closure $next): mixed
    {
        $appLocale = Cookie::get('app_locale') ?: config('app.locale');

        app()->setLocale($appLocale);

        Registry::set('appLocale', $appLocale);

        if (auth()->check()) {
            if ($Request->has('change-tenant')) {
                /*
                **  Die Übergabe des Parameters "change-tenant" initiiert einen Wechsel des aktuellen
                **  Tenants. Mit der übergebenen Tenant-ID werden die Daten in der Session neu
                **  initialisiert. */
                $this->_changeTenant($Request);
                $this->_initRegistry();

            } elseif (Session::has('tenantId')) {
                /*
                **  Registry mit Session-Daten initialisieren. */
                $this->_initRegistry();

            } else {
                $this->_configTenant = config('globals.global-tenant');

                if (empty($this->_configTenant)) {
                    /*
                    **  Tenant-ID des Users ermitteln und die Session mit Daten zu diesem Tenant
                    **  initialisieren. */
                    $AuthUser = auth()->user();
                    /*
                    **  Wenn der Active-Status des Users FALSE ist oder das Initialisieren der Session-
                    **  Daten fehlschlägt (weil der User keinen Tenant-Zugriff hat), dann wird der User
                    **  direkt wieder zum Login umgeleitet. */
                    if ((bool) $AuthUser->active === false || $this->_initSession($AuthUser->tenant_id) === false) {
                        Session::flush();

                        return redirect('login')->with('status', __('globals::global.error.auto-logout'));
                    }
                } else {
                    $this->_singleTenantMode = true;

                    $Tenant = TenantRepository::getByIdentifier($this->_configTenant, ignoreRestriction: true);

                    $this->_initSession($Tenant->id);
                }
                $this->_initRegistry();
            }
        }
        return $next($Request);

    } // handle()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class InitWebEnv {}
