<?php
/**
 * Service Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Services
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Services;


use Exception;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;


/**
 * Service Class
 *
 * @version     1.0.0 / 2025-05-18
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 */
class HttpService
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $configFileName
     */
    protected string $configFileName = 'httpservice';


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Response $response
     *
     * @param       string | null $cacheKey
     *
     * @return      mixed
     *
     * @version     1.0.0 / 2025-05-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     */
    private function _handleResponse(Response $response, string | null $cacheKey = null): mixed
    {
        if ($response->successful()) {
            return $response;

        } elseif ($response->failed()) {
            throw new Exception($response->getReasonPhrase());
        }
    } // _handleResponse()


    /**
     *
     * @param       $url
     *
     * @param       $data
     *
     * @return      mixed
     *
     * @version     1.0.0 / 2025-05-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     */
    public function delete($url, $data = null)
    {
        try {
            Log::info('DELETE ' . $url);
            Log::debug('DELETE ' . $url . ' DATA: ' . print_r($data, true));

            $response = Http::delete($url, $data);

            if (config("{$this->configFileName}.log_requests")) {
                Log::debug('RESPONSE FROM ' . $url . ' DELETE: ' . print_r($response->json(), true));
            }
            return $this->_handleResponse($response);

        } catch (ConnectionException $ex) {
            Log::emergency('ConnectionException on HTTPService delete: ' . print_r($ex->getMessage(), true));

            throw $ex;

        } catch (\Exception $ex) {
            Log::emergency('Exception on HTTPService delete: ' . print_r($ex->getMessage(), true));

            throw new Exception($ex->getMessage());
        }
    } // delete()


    /**
     *
     * @param       string $url
     *
     * @param       string $cacheKey
     *
     * @param       bool $localCacheEnabled
     *
     * @param       array $headers
     *
     * @param       int $requestTimeoutInSeconds
     *
     * @version     1.0.0 / 2025-05-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     */
    public function get(string $url, string $cacheKey = null, bool $localCacheEnabled = true, array $headers = [], int $requestTimeoutInSeconds = 10)
    {
        static $localCache = array();

        if ($localCacheEnabled) {
            $localCacheKey = md5(serialize(func_get_args()));

            Log::warning("localCacheKey for $url $localCacheKey");
            Log::warning(isset($localCache[$localCacheKey]) ? 'true' . ' - ' . $localCacheKey ?? 'cachekey leer' : 'false' . ' - ' . $localCacheKey ?? 'cachekey leer');

            if (isset($localCache[$localCacheKey])) {
                Log::warning("GET $url returning localCache content");

                return $localCache[$localCacheKey];
            }
            Log::warning(print_r($localCache, true));
        }
        Log::info('START GET ' . $url);

        try {
            $response = Http::withOptions([
                'headers' => $headers,
                'verify' => false
            ])->timeout($requestTimeoutInSeconds)->get($url);

            if (config("{$this->configFileName}.log_requests")) {
                Log::debug('RESPONSE FROM ' . $url . ' GET: ' . print_r($response->json(), true));
            }
            $handledResponse = $this->_handleResponse($response, $cacheKey);

            if ($localCacheEnabled) {
                $localCache[$localCacheKey] = $handledResponse;
            }
            return $handledResponse;

        } catch (ConnectionException $ex) {
            Log::emergency('ConnectionException on HTTPService get: ' . print_r($ex->getMessage(), true));

            throw $ex;

        } catch (\Throwable $th) {
            Cache::flush();

            $thArray = [
                'message' => $th->getMessage(),
                'file' => $th->getFile(),
                'line' => $th->getLine(),
            ];

            Log::emergency('Exception on HTTPService get: ' . print_r($thArray, true));

            throw new Exception($th->getMessage());
        }
    } // get()


    /**
     *
     * @param       string $url
     *
     * @param       array $data
     *
     * @param       $withBody
     *
     * @param       array $files
     *
     * @param       array $headers
     *
     * @param       int $requestTimeoutInSeconds
     *
     * @param       string $contentType
     *
     * @return      mixed
     *
     * @version     1.0.0 / 2025-05-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     */
    public function post(string $url, $data = [], $withBody = null, $files = [], $headers = [], int $requestTimeoutInSeconds = 10, string $contentType = 'application/json')
    {
        try {
            Log::info('POST ' . $url);
            Log::debug('POST ' . $url . ' DATA: ' . print_r($data, true));

            $httpClient = Http::withOptions([
                'headers' => $headers,
                'verify' => false
            ])->timeout($requestTimeoutInSeconds);

            foreach ($files as $file) {
                $name = $file['name'];
                $path = $file['path'];
                $filename = $file['filename'];

                $httpClient->attach($name, file_get_contents($path), $filename);
            }
            if (isset($withBody)) {
                $response = $httpClient->withBody($withBody, $contentType)->post($url);
            } else {
                $response = $httpClient->post($url, $data);
            }
            if (config("{$this->configFileName}.log_requests")) {
                Log::debug('RESPONSE FROM ' . $url . ' POST: ' . print_r($response->json(), true));
            }
            return $this->_handleResponse($response);

        } catch (ConnectionException $ex) {
            Log::emergency('ConnectionException on HTTPService post: ' . print_r($ex->getMessage(), true));

            throw $ex;

        } catch (\Exception $ex) {
            Log::emergency('Exception on HTTPService post: ' . print_r($ex->getMessage(), true));

            throw new Exception($ex->getMessage());
        }
    } // post()


    /**
     *
     * @param       string $url
     *
     * @param       array $data
     *
     * @param       array $files
     *
     * @param       array $headers
     *
     * @version     1.0.0 / 2025-05-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     */
    public function put($url, $data = [], $files = [], $headers = [])
    {
        try {
            Log::info('PUT '  . $url);
            Log::debug('PUT ' . $url . ' DATA: ' . print_r($data, true));

            $httpClient = Http::withOptions([
                'headers' => $headers
            ]);

            foreach ($files as $filename => $file) {
                $path = $file[0];
                $originalName = $file[1];

                $httpClient->attach($filename, file_get_contents($path), $originalName);
            }
            if (is_array($files) && count($files) > 0) {
                $data = ['data' => json_encode($data)];
            }
            $response = $httpClient->put($url, $data);

            if (config("{$this->configFileName}.log_requests")) {
                Log::debug('RESPONSE FROM ' . $url . ' PUT: ' . print_r($response->json(), true));
            }
            return $this->_handleResponse($response);

        } catch (ConnectionException $ex) {
            Log::emergency('ConnectionException on HTTPService put: ' . print_r($ex->getMessage(), true));

            throw $ex;

        } catch (\Exception $ex) {
            Log::emergency('Exception on HTTPService put: ' . print_r($ex->getMessage(), true));

            throw new Exception($ex->getMessage());
        }
    } // put()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class HttpService {}
