<?php
/**
 * String Helper Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/tool-box
 * @subpackage  Helpers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\ToolBox\Helpers;


/**
 * String Helper Class
 *
 * @version     1.0.0 / 2025-03-18
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class StringHelper
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Normalisiert die übergebene E-Mail-Adresse
     *
     * @param       null|string $email
     *
     * @param       mixed $returnOnEmpty
     *
     * @return 	    mixed
     *              Liefert die bereinigte E-Mail-Adresse in Kleinschreibung oder im Fehlerfall NULL
     *              beziehungsweise den übergebenen Return-Wert.
     *
     * @version     1.0.0 / 2025-03-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function normalizeEmail(null|string $email, mixed $returnOnEmpty = null): mixed
    {
        if ($email !== null) {
            $email = trim($email);
        }
        if (empty($email)) {
            return $returnOnEmpty;
        }
        return mb_strtolower($email);

    } // normalizeEmail()


    /**
     * Normalisiert das übergebene KFZ-Kennzeichen
     *
     * Entfernt alle nicht-alphanumerischen Zeichen aus der übergebenen Zeichenkette.
     *
     * @param       null|string $licensePlate
     *
     * @param       mixed $returnOnEmpty
     *
     * @return 	    mixed
     *              Liefert das bereinigte KFZ-Kennzeichen in Großschreibung oder im Fehlerfall NULL
     *              beziehungsweise den übergebenen Return-Wert.
     *
     * @version     1.0.0 / 2025-03-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function normalizeLicensePlate(null|string $licensePlate, mixed $returnOnEmpty = null): mixed
    {
        if ($licensePlate !== null) {
            $licensePlate = trim($licensePlate);
        }
        if (empty($licensePlate)) {
            return $returnOnEmpty;
        }
        /*
        **  Erklärung
        **  \p{L}   Erlaubt alle Buchstaben, einschließlich Umlauten und Buchstaben aus anderen Sprachen.
        **  \p{N}   Erlaubt alle Ziffern.
        **  ^       Negiert die Zeichenklasse, um alle Zeichen außer Buchstaben und Ziffern zu erfassen.
        **  +       Sorgt dafür, dass eine oder mehrere aufeinanderfolgende unzulässige Zeichen auf einmal entfernt werden.
        **  u       Aktiviert den UTF-8-Modus, damit auch Sonderzeichen korrekt verarbeitet werden. */
        return preg_replace('/[^\p{L}\p{N}]+/u', '', mb_strtoupper($licensePlate));

    } // normalizeLicensePlate()


    /**
     * Normalisiert die übergebene Telefonnummer
     *
     * Entfernt alle nicht-numerischen Zeichen außer dem führenden Pluszeichen (+).
     *
     * @param       null|string $phoneNumber
     *              Erwartet eine vollständige Telefonnummer mit führendem Pluszeichen (+) oder
     *              vorangestellten Doppelnullen (00).
     *
     * @param       mixed $returnOnEmpty
     *
     * @return      mixed
     *              Liefert die bereinigte Telefonnummer mit führendem Pluszeichen oder im Fehlerfall
     *              NULL beziehungsweise den übergebenen Return-Wert.
     *
     * @version     1.0.0 / 2025-03-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function normalizePhoneNumber(null|string $phoneNumber, mixed $returnOnEmpty = null): mixed
    {
        if ($phoneNumber !== null) {
            $phoneNumber = trim($phoneNumber);
        }
        if (empty($phoneNumber)) {
            return $returnOnEmpty;
        }
        /*
        **  Entfernt alle nicht-numerischen Zeichen außer dem führenden Pluszeichen (+). */
        $normalized = preg_replace('/[^\d+]/', '', $phoneNumber);
        /*
        **  Führende Doppel-Nullen (00) werden durch ein Plussymbol (+) ersetzt. */
        if (strpos($normalized, '00') === 0) {
            $normalized = '+'.substr($normalized, 2);
        }
        return $normalized;

    } // normalizePhoneNumber()


} // class StringHelper {}
