<?php
/**
 * Repository Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Repositories
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Repositories;


use BplanBase\Globals\Foundation\Repositories\BaseRepository;
use BplanBase\Globals\Models\Setting;
use BplanBase\Globals\ValueObjects\PaginateArgs;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;


/**
 * Repository Class
 *
 * @version     1.0.0 / 2025-05-03
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class SettingRepository extends BaseRepository
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_modelClass
     */
    protected static $_modelClass = Setting::class;


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Returns a collection of models
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       array|string|null $orderBy
     *              @see BplanBase\Globals\Helpers\QueryHelper::applyOrderBy() for
     *              details.
     *
     * @param       array|bool|int $paginate
     *              Can be either the number of elements, TRUE or an array. If
     *              TRUE is passed, the default number from the configuration is
     *              used. If an array is passed, then all known arguments for
     *              paginate() (columns, page, pageName, perPage, total) can be
     *              specified in it (all optional).
     *              @see https://api.laravel.com/docs/12.x/Illuminate/Database/Eloquent/Builder.html#method_paginate
     *
     * @param       Builder|null $Query
     *
     * @return      Collection|LengthAwarePaginator<Model>
     *
     * @version     2.0.0 / 2025-06-04
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getAll(bool $ignoreCache = false, array|string|null $orderBy = null, array|bool|int $paginate = false, Builder|null $Query = null): Collection|LengthAwarePaginator
    {
        static $cache = [];

        $cacheKey = md5(json_encode(func_get_args()));

        if ($paginate !== false) {
            $ignoreCache = true;
        }
        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = self::_resolveQuery($Query);

        $Query->ordered($orderBy);

        if ($paginate === false) {
            return $cache[$cacheKey] = $Query->get();
        }
        $PaginateArgs = PaginateArgs::from($paginate);

        return $Query->paginate(...$PaginateArgs->toArray());

    } // getAll()


    /**
     * Returns all settings of a context as a key => value array
     *
     * @param       string $context
     *
     * @return      array<string, mixed>
     *
     * @version     1.0.0 / 2025-05-01
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getAllByContext(string $context): array
    {
        return Setting::where('context', $context)
            ->pluck('value', 'key')
            ->toArray();

    } // getAllByContext()


    /**
     * Returns the model instance for the given context and key
     *
     * @param       string $context
     *
     * @param       string $key
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $returnOrFail
     *              Controls the return of the method.
     *              If no suitable entry can be found in the database, either NULL is returned (TRUE)
     *              or an exception is thrown (FALSE). This allows the result of the method call to
     *              be used directly in the Api controller.
     *
     * @throws      Illuminate\Database\Eloquent\ModelNotFoundException
     *
     * @return      mixed
     *
     * @version     1.0.0 / 2025-05-03
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getValue(string $context, string $key, bool $ignoreCache = false, bool $returnOrFail = true): mixed
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = Setting::where('context', '=', $context)
            ->where('key', '=', $key);

        if ($returnOrFail === true) {
            return $cache[$cacheKey] = $Query->first();
        }
        return $cache[$cacheKey] = $Query->firstOrFail();

    } // getValue()


} // class SettingRepository extends BaseRepository {}
