<?php
/**
 * Setup Command Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Commands
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Console\Commands;


use BplanBase\Globals\Console\Output;
use BplanBase\Globals\Console\Prompt;
use BplanBase\Globals\Helpers\StringHelper;
use BplanBase\Globals\Setup\FilePublisher;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Symfony\Component\Process\Process;


use function Laravel\Prompts\clear;


/**
 * Setup Command Class
 *
 * @version     6.2.0 / 2025-08-07
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class GlobalsSetup extends Command
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_apiPackage
     */
    private string $_apiPackage = 'laravel-json-api/laravel';


    /**
     * @var     string $_basePath
     */
    private string $_basePath;


    /**
     * @var     string $_codeGeneratorLoaderPackage
     */
    private string $_codeGeneratorLoaderPackage = 'bplan-base/laravel-code-generator-loader';


    /**
     * @var     string $_codeGeneratorPackage
     */
    private string $_codeGeneratorPackage = 'bplan-base/laravel-code-generator';


    /**
     * @var     array $_composerJson
     */
    private array $_composerJson;


    /**
     * @var     bool $_firstRun
     */
    private bool $_firstRun;


    /**
     * @var     bool $_hasApi
     */
    private bool $_hasApi;


    /**
     * @var     bool $_hasCodeGenerator
     */
    private bool $_hasCodeGenerator;


    /**
     * @var     bool $_hasCodeGeneratorLoader
     */
    private bool $_hasCodeGeneratorLoader;


    /**
     * @var     bool $_hasMergePlugin
     */
    private bool $_hasMergePlugin;


    /**
     * @var     bool $_hasPrivateGit
     */
    private bool $_hasPrivateGit;


    /**
     * @var     int $_installPackages
     */
    private int $_installPackages = 0;


    /**
     * @var     string $_mergePlugin
     */
    private string $_mergePlugin = 'wikimedia/composer-merge-plugin';


    /**
     * @var     Output $_Output
     */
    private Output $_Output;


    /**
     * @var     string $_privateGitName
     */
    private string $_privateGitName = 'bplan/composer-packages';


    /**
     * @var     string $_privateGitUrl
     */
    private string $_privateGitUrl = 'https://satis.bplan.solutions';


    /**
     * @var     Prompt $_Prompt
     */
    private Prompt $_Prompt;


    /**
     * @var     bool $_requireApi
     */
    private bool $_requireApi = false;


    /**
     * @var     bool $_requireCodeGenerator
     */
    private bool $_requireCodeGenerator;


    /**
     * @var     bool $_requireMergePlugin
     */
    private bool $_requireMergePlugin;


    /**
     * @var     string $description
     */
    protected $description = 'Prepares the Laravel project environment';


    /**
     * @var     string $signature
     */
    protected $signature = 'globals:setup';


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     int $_commandTimeout
     */
    private static int $_commandTimeout = 300;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-06-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _finishPackageSetup()
    {
        if ($this->_installPackages > 0) {
            $this->_Output->caption('Package installation');
            $this->_Output->print('Run `composer update`', bullet: true);

            $this->_runShellCommand([
                'composer',
                'update',
            ]);
            if ($this->_requireApi === true) {
                $Publisher = new FilePublisher($this->_basePath, ['etc/setup/src/config/jsonapi.php' => 'config/jsonapi.php'], [$this, 'publishMessage']);

                $Publisher->publish();
            }
        }
    } // _finishPackageSetup()


    /**
     *
     * @return 	    void
     *
     * @version     1.2.2 / 2025-07-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(): void
    {
        $this->_composerJson = json_decode(file_get_contents(base_path('composer.json')), true);

        $this->_basePath = str_replace('\\', '/', realpath(__DIR__.'/../../../'));
        $this->_firstRun = !file_exists(config_path('project.php'));
        $this->_hasApi = file_exists(base_path('vendor/'.$this->_apiPackage.'/core/composer.json'));
        $this->_hasCodeGenerator = file_exists(base_path('vendor/'.$this->_codeGeneratorPackage.'/composer.json'));
        $this->_hasCodeGeneratorLoader = file_exists(base_path('vendor/'.$this->_codeGeneratorLoaderPackage.'/composer.json'));
        $this->_hasMergePlugin = file_exists(base_path('vendor/'.$this->_mergePlugin.'/composer.json'));
        $this->_hasPrivateGit = isset($this->_composerJson['repositories'][$this->_privateGitName]);

        $this->_Output = new Output($this);
        $this->_Prompt = new Prompt($this);

        Output::registerStatusColor(FilePublisher::STATUS_DONE, 'green');
        Output::registerStatusColor(FilePublisher::STATUS_MISSING, 'red');
        Output::registerStatusColor(FilePublisher::STATUS_SKIPPED, 'gray');

    } // _init()


    /**
     *
     * @return 	    void
     *
     * @version     1.1.0 / 2025-06-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _linkStorage(): void
    {
        /*
        **  Storage-Link erzeugen. */
        $this->_Output->caption('Link storage directory');

        if (file_exists(public_path('storage'))) {
            $this->_Output->bulletMessage('Storage link already exists');

            return;
        }
        Artisan::call('storage:link');

        $this->_Output->bulletMessage(Artisan::output());

    } // _linkStorage()


    /**
     *
     * @return 	    void
     *
     * @version     1.2.0 / 2025-08-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareEnvFiles()
    {
        $envFile = base_path('.env');
        $inFile = base_path('.env.in');

        if (file_exists($inFile) === false) {
            $this->_Output->caption('Preparing configuration');
            $this->_Output->bulletMessage('Prepare file .env');

            $this->_Output->bulletMessage('Copy .env > .env.in');

            $envFilecontents = $this->_reorganizeEnvFilecontents();

            file_put_contents($envFile, rtrim($envFilecontents));

            copy($envFile, $inFile);

            $this->_Output->bulletMessage('Extend file .env');

            file_put_contents(
                $envFile,
                "\n\n\n".'# CUSTOM'
                ."\n"
                ."\n".'CODE_GENERATOR_ACTIVE = '.($this->_requireCodeGenerator ? 'true' : 'false')
                ."\n".'CODE_GENERATOR_CREATE_JSONAPI = '.($this->_requireApi ? 'true' : 'false')
                ."\n".'CODE_GENERATOR_SKIP_ROUTING = true'
                ."\n",
                FILE_APPEND
            );
            $this->_Output->bulletMessage('Extend file .env.in');

            file_put_contents(
                $inFile,
                "\n\n\n".'# CUSTOM'
                ."\n"
                ."\n".'CODE_GENERATOR_ACTIVE = false'
                ."\n".'CODE_GENERATOR_CREATE_JSONAPI = '.($this->_requireApi ? 'true' : 'false')
                ."\n".'CODE_GENERATOR_SKIP_ROUTING = true'
                ."\n",
                FILE_APPEND
            );
        }
    } // _prepareEnvFiles()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-06-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _publishFiles(): void
    {
        $this->_Output->caption('Publish files');

        $Publisher = new FilePublisher($this->_basePath, config('globals-setup.publish.always'), [$this, 'publishMessage']);

        $Publisher->force()->publish();

        $Publisher = new FilePublisher($this->_basePath, config('globals-setup.publish.once'), [$this, 'publishMessage']);

        if ($this->_firstRun === true) {
            $Publisher->force();
        }
        $Publisher->publish();

    } // _publishFiles()


    /**
     *
     * @param       null|string $localPath
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-06-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _registerPrivateGit(null|string $localPath = null): void
    {
        if ($this->_hasPrivateGit === true) {
            return;
        }
        $this->_Output->caption('Register private Git Repository');
        $this->_Output->bulletMessage('Update composer.json with private Git repository');

        $this->_runShellCommand([
            'composer',
            'config',
            'repositories.'.$this->_privateGitName,
            'composer',
            $this->_privateGitUrl
        ]);
    } // _registerPrivateGit()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-07-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _reorganizeEnvFilecontents(): string
    {
        /*
        **  Disassemble .env file. */
        $lines = file(base_path('.env'));

        $result = [];

        foreach ($lines as $line) {
            $line = trim($line);

            if ($line === '') {
                continue;
            }
            $comment = false;
            $quote = false;

            if ($line[0] === '#') {
                $comment = true;
                $line = trim(substr($line, 1));
            }
            /*
            **  Dismantle a single line. */
            $parts = explode('=', $line);

            $key = trim($parts[0]);

            if (!isset($parts[1])) {
                $value = false;

            } else {
                $value = trim($parts[1]);

                if ($value !== '' && $value[0] === '"') {
                    $quote = true;
                }
            }
            /*
            **  Get the key prefix. */
            $keyParts = explode('_', $key, 2);

            $prefix = $keyParts[0];

            if (!isset($result[$prefix])) {
                $result[$prefix] = [];
            }
            $result[$prefix][$key] = [
                'comment' => $comment,
                'key' => $key,
                'line' => $line,
                'quote' => $quote,
                'value' => $value,
            ];
        }
        /*
        **  Assemble the filecontents. */
        $filecontents = '';

        ksort($result);

        foreach ($result as $settings) {
            ksort($settings);

            $separator = '';

            foreach ($settings as $setting) {
                $comment = '';

                if ($setting['comment'] === true) {
                    $comment = '# ';
                }
                if ($setting['value'] === false) {
                    $filecontents .= $separator.$comment.$setting['line'];

                    continue;
                }
                $filecontents .= $separator.$comment.rtrim($setting['key'].' = '.$setting['value']);
                $separator = "\n";
            }
            $filecontents .= "\n\n";
        }
        return $filecontents;

    } // _reorganizeEnvFilecontents()


    /**
     *
     * @param       array $command
     *
     * @return      void
     *
     * @version     1.2.0 / 2025-06-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _runShellCommand(array $command): void
    {
        $Process = new Process($command);

        $Process->setTimeout(self::$_commandTimeout);

        $Process->run();

        if (!$Process->isSuccessful()) {
            $this->_Output->error('  Command failed: '.implode(' ', $command));
            $this->_Output->error($Process->getErrorOutput());

        } else {
            //$this->_Output->print($Process->getOutput());
        }
    } // _runShellCommand()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-06-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setupApiPackage(): void
    {
        if ($this->_hasApi === true) {
            return;
        }
        $this->_Output->bulletMessage('Update composer.json with api-package ['.$this->_apiPackage.']');

        $this->_runShellCommand([
            'composer',
            'require',
            $this->_apiPackage,
            '--no-update', // '--no-install',
            // '--no-scripts',  '--no-autoloader',
        ]);
        $this->_installPackages++;

    } // _setupApiPackage()


    /**
     *
     * @param       null|string $localPath
     *
     * @return 	    void
     *
     * @version     1.0.1 / 2025-06-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setupCodeGenerator(null|string $localPath = null): void
    {

        if ($this->_hasCodeGenerator === true) {
            return;
        }
        if ($localPath !== null) {
            $this->_Output->bulletMessage('Update composer.json with local repository');

            $package = $this->_codeGeneratorPackage;

            $this->_runShellCommand([
                'composer',
                'config',
                'repositories.'.$this->_codeGeneratorPackage,
                'path',
                str_replace('\\', '/', $localPath)
            ]);
            $package .= ':@dev';

            $this->_Output->bulletMessage('Update composer.json with code-generator ['.$this->_codeGeneratorPackage.']');

        } else {
            $this->_Output->bulletMessage('Download package ['.$this->_codeGeneratorPackage.']');
        }
        $this->_runShellCommand([
            'composer',
            'require',
            $package,
            '--dev',
            '--no-update', // '--no-install',
            // '--no-scripts',  '--no-autoloader',
        ]);
        $this->_installPackages++;

    } // _setupCodeGenerator()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-06-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setupCodeGeneratorLoader(): void
    {
        if ($this->_hasCodeGeneratorLoader === true) {
            return;
        }
        $this->_Output->caption('Update composer.json with CodeGenerator Loader');

        $this->_Output->bulletMessage('Download package ['.$this->_codeGeneratorLoaderPackage.']');

        $this->_runShellCommand([
            'composer',
            'require',
            $this->_codeGeneratorLoaderPackage.':*',
            '--no-update', // '--no-install',
            // '--no-scripts',  '--no-autoloader',
        ]);
        $this->_installPackages++;

    } // _setupCodeGeneratorLoader()


    /**
     *
     * @return 	    void
     *
     * @version     1.2.0 / 2025-06-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setupMergePlugin(): void
    {
        if ($this->_hasMergePlugin === true) {
            return;
        }
        $this->_Output->caption('Update composer.json with merge-plugin ['.$this->_mergePlugin.']');

        $this->_Output->bulletMessage('Configure "allow-plugins".');

        $this->_runShellCommand([
            'composer',
            'config',
            'allow-plugins.'.$this->_mergePlugin,
            'true',
        ]);
        $this->_Output->bulletMessage('Configure "extra.merge-plugin.include"');

        $this->_runShellCommand([
            'composer',
            'config',
            'extra.merge-plugin.include',
            '--json',
            '["composer.local.json"]',
        ]);
        $this->_Output->bulletMessage('Configure "extra.merge-plugin.merge-dev"');

        $this->_runShellCommand([
            'composer',
            'config',
            'extra.merge-plugin.merge-dev',
            'true',
        ]);
        $this->_Output->bulletMessage('Download package ['.$this->_mergePlugin.']');

        $this->_runShellCommand([
            'composer',
            'require',
            $this->_mergePlugin,
            '--dev',
             '--no-update', // '--no-install',
        ]);
        $this->_installPackages++;

    } // _setupMergePlugin()


    /**
     *
     * @return      void
     *
     * @version     1.4.0 / 2025-06-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function handle()
    {
        $this->_init();

        clear();

        $this->_Output->title('Globals Setup');

        $this->_registerPrivateGit();

        if ($this->_firstRun === true && $this->_Prompt->confirm('Does the current project require an API?', true)) {
            $this->_requireApi = true;

            $this->_setupApiPackage();
        }
        if ($this->_hasCodeGenerator === false) {
            $setupCodeGenerator = (int) $this->_Prompt->choice('Do you want to install the code generator?', ['no', 'yes (local)', 'yes (from Git)'], 0);

            $codeGeneratorPath = null;

            switch ($setupCodeGenerator) {
                case 1:
                    $codeGeneratorPath = $this->_Prompt->ask('Please enter the full path to the local repository', validator: function ($input) {
                        return file_exists($input.'/src/GeneratorServiceProvider.php');
                    });
                    // no break

                case 2:
                    $this->_setupCodeGenerator($codeGeneratorPath);
            }
        }
        $this->_setupCodeGeneratorLoader();
        $this->_setupMergePlugin();
        $this->_linkStorage();
        $this->_publishFiles();
        $this->_prepareEnvFiles();

        $this->_finishPackageSetup();

        $this->_Output->print('Setup completed ✅', color: 'title', indent: 3, prependNewline: 2);

    } // handle()


    /**
     *
     * @param       array $event
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-06-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function publishMessage(array $event)
    {
        $msg  = $event['message'];
        $source  = $event['source'];
        $status = $event['status'];
        $target  = $event['target'];

        $message = 'Publish '.$target;

        $this->_Output->statusMessage($message, $status);

    } // publishMessage()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class GlobalsSetup extends Command {}
