<?php
/**
 * Module Storage Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Support
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Support;


use BplanBase\Globals\Repositories\Core\TenantRepository;
use Illuminate\Contracts\Filesystem\Filesystem;
use Illuminate\Http\File;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\StreamedResponse;


/**
 * Module Storage Class
 *
 * @version     1.0.0 / 2025-08-12
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class ModuleStorage
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string GLOBAL_DIR
     */
    const GLOBAL_DIR = '.global';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_basePath
     */
    protected string $_basePath;


    /**
     * @var     string $_disk
     */
    protected string $_disk;


    /**
     * @var     string $_module
     */
    protected string $_module;


    /**
     * @var     bool $_public
     */
    protected string $_public;


    /**
     * @var     boolean|integer|string|null $_tenant
     */
    protected bool|int|string|null $_tenant;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       boolean $public
     *
     * @param       string $module
     *              Der Name des Moduls (~ Verzeichnisname).
     *
     * @param       boolean|integer|string|null $tenant
     *              Es kann die ID, die UUID oder der Identifier eines Tenants übergeben
     *              werden. Bei Übergabe von NULL (Default) wird der Tenant aus der Registry
     *              verwendet. Wenn FALSE übergeben wird, dann wird im globalen Kontext
     *              gearbeitet.
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(bool $public, string $module, bool|int|string|null $tenant = null)
    {
        $this->_disk = $public ? 'public' : 'local';
        $this->_module = $module;
        $this->_public = $public;
        $this->_tenant = $tenant;

        $this->_basePath = $this->_basePath();

    } // __construct()


    /**
     * Pfad relativ zum Disk ermitteln.
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _basePath(): string
    {
        $tenantFolder = $this->_resolveTenantFolder();

        return $this->_moduleBasePath().'/'.$tenantFolder;

    } // _basePath()


    /**
     * Gibt den Modul-Basis-Pfad (ohne Tenant-Verzeichnis) relativ zum Disk-Root zurück.
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _moduleBasePath(): string
    {
        return $this->_module;

    } // _moduleBasePath()


    /**
     *
     * @param       string $subdir
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _normalize($path): string
    {
        if ($path === '') {
            return $path;
        }
        return trim(str_replace('\\', '/', $path), '/');

    } // _normalize()


    /**
     * Interner Helper zum Speichern einer Datei im Modul-Storage.
     *
     * @param       string $destination
     *              Zielpfad relativ zum Modul-/Tenant-Root.
     *
     * @param       string|resource $contents
     *              Dateiinhalt oder Stream.
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _putWithDirectory(string $destination, $contents): bool
    {
        $toPath = $this->_basePath.'/'.$this->_normalize($destination);

        // Zielverzeichnis anlegen, falls nötig
        $targetDir = dirname($toPath);

        if (!Storage::disk($this->_disk)->exists($targetDir)) {
            Storage::disk($this->_disk)->makeDirectory($targetDir);
        }
        return Storage::disk($this->_disk)->put($toPath, $contents);

    } // _putWithDirectory()


    /**
     * Tenant-Ordner ermitteln.
     *
     * @return      string
     *              Liefert den Verzeichnisnamen des Tenant-Verzeichnisses oder ".global",
     *              wenn kein Tenant involviert ist. Der Name des Tenant-Verzeichnisses
     *              entspricht der MD5-Summe der UUID des Tenants. So lassen sich über
     *              den Verzeichnisnamen keine Rückschlüsse auf den Tenant ziehen.
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _resolveTenantFolder(): string
    {
        if ($this->_tenant === false) {
            return self::GLOBAL_DIR;
        }
        if ($this->_tenant === null) {
            $tenantId = \Registry::get('auth.tenantId');

            if (!$tenantId) {
                throw new \RuntimeException('No tenant found in context. Specify tenant explicitly or FALSE for global.');
            }
            $Tenant = TenantRepository::getById($tenantId, ignoreRestriction: true);

            return md5($Tenant->uuid);
        }
        $Tenant = TenantRepository::getByVarious($this->_tenant, ignoreRestriction: true);

        return md5($Tenant->uuid);

    } // _resolveTenantFolder()


    /**
     * Listet alle Unterverzeichnisse rekursiv im aktuellen Modul-/Tenant-Ordner.
     *
     * @param       string $subdir
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function allDirectories(string $subdir = ''): array
    {
        $path = $this->_basePath.'/'.$this->_normalize($subdir);

        return Storage::disk($this->_disk)->allDirectories($path);

    } // allDirectories()


    /**
     * Listet alle Dateien rekursiv im aktuellen Modul-/Tenant-Ordner.
     *
     * @param       string $subdir
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function allFiles(string $subdir = ''): array
    {
        $path = $this->_basePath.'/'.$this->_normalize($subdir);

        return Storage::disk($this->_disk)->allFiles($path);

    } // allFiles()


    /**
     * Kopiert eine Datei innerhalb desselben Modul-Storages.
     *
     * @param       string $from
     *              Relativer Pfad zur Quelldatei (innerhalb des Modul-/Tenant-Bereichs)
     *
     * @param       string $to
     *              Relativer Pfad zur Zieldatei (innerhalb des Modul-/Tenant-Bereichs)
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function copy(string $from, string $to): bool
    {
        $contents = Storage::disk($this->_disk)->get($from);

        return $this->_putWithDirectory($to, $contents);

    } // copy()


    /**
     * Kopiert eine Datei aus einem anderen Storage ins aktuelle Modul-Storage.
     *
     * @param       \Illuminate\Contracts\Filesystem\Filesystem $SourceDisk
     *              Quelle-Disk
     *
     * @param       string $sourcePath
     *              Pfad der Quelldatei im Quell-Storage
     *
     * @param       string $destination
     *              Relativer Pfad (innerhalb des Modul-/Tenant-Bereichs)
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function copyFrom(Filesystem $SourceDisk, string $sourcePath, string $destination): bool
    {
        $contents = $SourceDisk->get($sourcePath);

        return $this->_putWithDirectory($destination, $contents);

    } // copyFrom()


    /**
     * Kopiert eine Datei vom aktuellen Modul-Storage in ein anderes Storage.
     *
     * @param       string $source
     *              Relativer Pfad zur Quelldatei (innerhalb des Modul-/Tenant-Bereichs)
     *
     * @param       \Illuminate\Contracts\Filesystem\Filesystem $TargetDisk
     *              Das Ziel-Storage
     *
     * @param       string $targetPath
     *              Pfad der Zieldatei im Ziel-Storage
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-15
     */
    public function copyTo(string $source, Filesystem $TargetDisk, string $targetPath): bool
    {
        $sourcePath = $this->_basePath.'/'.$this->_normalize($source);

        if (!Storage::disk($this->_disk)->exists($sourcePath)) {
            return false;
        }
        $contents = Storage::disk($this->_disk)->get($sourcePath);

        // Zielverzeichnis anlegen, falls nötig
        $targetDir = dirname($targetPath);

        if ($targetDir !== '.' && !$TargetDisk->exists($targetDir)) {
            $TargetDisk->makeDirectory($targetDir);
        }
        return $TargetDisk->put($targetPath, $contents);

    } // copyTo()


    /**
     * Löscht eine Datei oder mehrere Dateien im aktuellen Modul-/Tenant-Verzeichnis.
     *
     * @param       string|array $files
     *              Relativer Dateiname oder Array von Dateinamen.
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function delete(string|array $files): bool
    {
        $files = (array) $files;

        $fullPaths = array_map(function ($file) {
            return $this->_basePath.'/'.$this->_normalize($file);

        }, $files);

        return Storage::disk($this->_disk)->delete($fullPaths);

    } // delete()


    /**
     * Löscht ein Verzeichnis im aktuellen Modul-/Tenant-Verzeichnis.
     *
     * Wird $subdir nicht angegeben, wird das gesamte Tenant-Verzeichnis des Moduls gelöscht.
     *
     * @param       string $subdir
     *              Relativer Verzeichnispfad innerhalb des Tenant-Ordners.
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function deleteDirectory(string $subdir = ''): bool
    {
        $path = $this->_basePath;

        if ($subdir !== '') {
            $path .= '/'.$this->_normalize($subdir);
        }
        return Storage::disk($this->_disk)->deleteDirectory($path);

    } // deleteDirectory()


    /**
     * Listet Unterverzeichnisse im aktuellen Modul-/Tenant-Ordner (nicht rekursiv).
     *
     * @param       string $subdir
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function directories(string $subdir = ''): array
    {
        $path = $this->_basePath.'/'.$this->_normalize($subdir);

        return Storage::disk($this->_disk)->directories($path);

    } // directories()


    /**
     * Datei als Download ausgeben.
     *
     * @param       string $file
     *
     * @param       string|null $name
     *
     * @param       array $headers
     *
     * @return      StreamedResponse
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function download(string $file, string|null $name = null, array $headers = []): StreamedResponse
    {
        $fullPath = $this->_basePath.'/'.$this->_normalize($file);

        if (!Storage::disk($this->_disk)->exists($fullPath)) {
            abort(404, 'File ['.$file.'] not found.');
        }
        return Storage::disk($this->_disk)->download($fullPath, $name ?? basename($file), $headers);

    } // download()


    /**
     * Existenz prüfen.
     *
     * @param       string $file
     *
     * @return      boolean
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function exists(string $file): bool
    {
        $fullPath = $this->_basePath.'/'.$this->_normalize($file);

        return Storage::disk($this->_disk)->exists($fullPath);

    } // exists()


    /**
     * Listet Dateien im aktuellen Modul-/Tenant-Ordner (nicht rekursiv).
     *
     * @param       string $subdir
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function files(string $subdir = ''): array
    {
        $path = $this->_basePath.'/'.$this->_normalize($subdir);

        return Storage::disk($this->_disk)->files($path);

    } // files()


    /**
     * Datei lesen.
     *
     * @param       string $file
     *
     * @return      string|null
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function get(string $file): string|null
    {
        $fullPath = $this->_basePath.'/'.$this->_normalize($file);
// dump($fullPath);
        return Storage::disk($this->_disk)->exists($fullPath)
            ? Storage::disk($this->_disk)->get($fullPath)
            : null;

    } // get()


    /**
     *
     * @param       string $subdir
     *
     * @return     string
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getBasePath(string $subdir = ''): string
    {
        return $this->_normalize($this->_basePath.'/'.$subdir);

    } // getBasePath()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getDisk(): string
    {
        return $this->_disk;

    } // getDisk()


    /**
     *
     *  @return     string
     *
     * @version     1.0.0 / 2025-08-14
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFullPath(string $subdir = ''): string
    {
        $fullPath = $this->_basePath.'/'.$this->_normalize($subdir);

        return $this->_normalize(Storage::disk($this->_disk)->path($fullPath));

    } // getFullPath()


    /**
     *
     *  @return     string
     *
     * @version     1.0.0 / 2025-08-14
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRootPath(): string
    {
        return $this->_normalize(Storage::disk($this->_disk)->path(''));

    } // getRootPath()


    /**
     *
     * @return      Filesystem
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getStorage(): Filesystem
    {
        return Storage::disk($this->_disk);

    } // getStorage()


    /**
     * Verschiebt (oder benennt um) eine Datei innerhalb desselben Modul-Storages.
     *
     * @param       string $from
     *              Relativer Pfad zur Quelldatei (innerhalb des Modul-/Tenant-Bereichs)
     *
     * @param       string $to
     *              Relativer Pfad zur Zieldatei (innerhalb des Modul-/Tenant-Bereichs)
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-12
     */
    public function move(string $from, string $to): bool
    {
        $fromPath = $this->_basePath.'/'.$this->_normalize($from);
        $toPath = $this->_basePath.'/'.$this->_normalize($to);
        /*
        **  Zielverzeichnis anlegen, falls erforderlich. */
        $targetDir = dirname($toPath);

        if (!Storage::disk($this->_disk)->exists($targetDir)) {
            Storage::disk($this->_disk)->makeDirectory($targetDir);
        }
        /*
        **  Atomare Verschiebung mit rename durchführen. */
        $Storage = Storage::disk($this->_disk);

        $fromFullPath = $Storage->path($fromPath);
        $toFullPath = $Storage->path($toPath);

        return rename($fromFullPath, $toFullPath);

    } // move()


    /**
     * Verschiebt eine Datei aus einem anderen Storage ins aktuelle Modul-Storage.
     *
     * @param       \Illuminate\Contracts\Filesystem\Filesystem $sourceDisk
     *              Die Quell-Disk.
     *
     * @param       string $sourcePath
     *              Pfad der Quelldatei im Quell-Storage.
     *
     * @param       string $destination
     *              Relativer Pfad (innerhalb des Modul-/Tenant-Bereichs).
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-12
     */
    public function moveFrom(Filesystem $SourceDisk, string $sourcePath, string $destination): bool
    {
        $toPath = $this->_basePath.'/'.$this->_normalize($destination);
        /*
        **  Zielverzeichnis anlegen, falls erforderlich. */
        $targetDir = dirname($toPath);

        if (!Storage::disk($this->_disk)->exists($targetDir)) {
            Storage::disk($this->_disk)->makeDirectory($targetDir);
        }
        /*
        **  Prüfen ob beide Disks auf demselben Filesystem liegen. */
        $sourceFullPath = $SourceDisk->path($sourcePath);
        $targetFullPath = Storage::disk($this->_disk)->path($toPath);
        /*
        **  Atomare Verschiebung mit rename versuchen. */
        if (@rename($sourceFullPath, $targetFullPath)) {
            return true;
        }
        /*
        **  Falls rename fehlschlägt (verschiedene Filesysteme), Fallback zu copy+delete. */
        $success = $this->copyFrom($SourceDisk, $sourcePath, $destination);

        if ($success === true) {
            $SourceDisk->delete($sourcePath);
        }
        return $success;

    } // moveFrom()


    /**
     * Verschiebt eine Datei vom aktuellen Modul-Storage in ein anderes Storage.
     *
     * @param       string $source
     *              Relativer Pfad zur Quelldatei (innerhalb des Modul-/Tenant-Bereichs)
     *
     * @param       \Illuminate\Contracts\Filesystem\Filesystem $TargetDisk
     *              Das Ziel-Storage
     *
     * @param       string $targetPath
     *              Pfad der Zieldatei im Ziel-Storage
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-08-15
     */
    public function moveTo(string $source, Filesystem $TargetDisk, string $targetPath): bool
    {
        $sourcePath = $this->_basePath.'/'.$this->_normalize($source);

        if (!Storage::disk($this->_disk)->exists($sourcePath)) {
            return false;
        }
        /*
        **  Zielverzeichnis anlegen, falls erforderlich. */
        $targetDir = dirname($targetPath);

        if ($targetDir !== '.' && !$TargetDisk->exists($targetDir)) {
            $TargetDisk->makeDirectory($targetDir);
        }
        /*
        **  Prüfen ob beide Disks auf demselben Filesystem liegen. */
        $sourceFullPath = Storage::disk($this->_disk)->path($sourcePath);
        $targetFullPath = $TargetDisk->path($targetPath);
        /*
        **  Atomare Verschiebung mit rename versuchen. */
        if (@rename($sourceFullPath, $targetFullPath)) {
            return true;
        }
        /*
        **  Falls rename fehlschlägt (verschiedene Filesysteme), Fallback zu copy+delete. */
        $success = $this->copyTo($source, $TargetDisk, $targetPath);

        if ($success === true) {
            Storage::disk($this->_disk)->delete($sourcePath);
        }
        return $success;

    } // moveTo()


    /**
     * Datei speichern (Verzeichnis wird komplett angelegt).
     *
     * @param       string $file
     *
     * @param       string $contents
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function put(string $file, string $contents): string
    {
        $dirPath = dirname($this->_basePath.'/'.$this->_normalize($file));
        $fullPath = $this->_basePath.'/'.$file;

        Storage::disk($this->_disk)->makeDirectory($dirPath);
        Storage::disk($this->_disk)->put($fullPath, $contents);

        return $fullPath;

    } // put()


    /**
     * Speichert eine Datei unter einem definierten oder generierten Namen im Modul-Storage.
     *
     * @param       UploadedFile|File|string $file
     *              Dateiobjekt oder Pfad zu einer existierenden Datei.
     *
     * @param       string $path
     *              Zielpfad relativ zum Modul-/Tenant-Root (ohne Dateinamen).
     *
     * @param       string|false|null $name
     *              FALSE = Originalname verwenden,
     *              NULL = eindeutiger Name mit Zeitstempel generieren,
     *              string = expliziter Name (Endung wird ggf. korrigiert).
     *
     * @throws      InvalidArgumentException
     *
     * @return      string
     *              Relativer Speicherpfad zur gespeicherten Datei.
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function putFileAs(UploadedFile|File|string $file, string $path, string|false|null $name = null): string
    {
        /*
        **  Extension ermitteln. */
        if ($file instanceof UploadedFile) {
            $extension     = strtolower($file->getClientOriginalExtension());
            $originalName  = $file->getClientOriginalName();

        } elseif ($file instanceof File) {
            $extension     = strtolower($file->extension());
            $originalName  = $file->getFilename();

        } elseif (is_string($file)) {
            $extension     = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            $originalName  = pathinfo($file, PATHINFO_BASENAME);

        } else {
            throw new \InvalidArgumentException('Invalid file type');
        }
        /*
        **  Ziel-Dateiname bestimmen. */
        if ($name === false) {
            /*
            **  Originalname verwenden. */
            $finalName = $originalName;

        } elseif ($name === null) {
            /*
            **  Eindeutiger Name mit Zeitstempel. */
            $finalName = now()->format('Ymd_His').'.'.uniqid().'.'.($extension ? '.' . $extension : '');

        } else {
            /*
            **  Entwickler gibt einen Namen vor – Endung ggf. korrigieren. */
            $givenExtension = strtolower(pathinfo($name, PATHINFO_EXTENSION));

            if (empty($givenExtension) || $givenExtension !== $extension) {
                $name .= $extension ? '.'.$extension : '';
            }
            $finalName = $name;
        }
        /*
        **  Vollständigen Zielpfad im Modul-Storage ermitteln. */
        $fullPath = $this->_basePath.'/'.$this->_normalize($path).'/'.$finalName;
        $directory = dirname($fullPath);

        if (!Storage::disk($this->_disk)->exists($directory)) {
            Storage::disk($this->_disk)->makeDirectory($directory);
        }
        /*
        **  Datei speichern. */
        if ($file instanceof UploadedFile || $file instanceof File) {
            Storage::disk($this->_disk)->putFileAs($directory, $file, $finalName);

        } elseif (is_string($file)) {
            Storage::disk($this->_disk)->put($fullPath, file_get_contents($file));
        }
        return $fullPath;

    } // putFileAs()


    /**
     * Gibt eine Datei als Download oder Inline-Response aus.
     *
     * @param       string $file
     *              Relativer Dateiname/Pfad innerhalb des Modul-/Tenant-Verzeichnisses
     *
     * @param       bool   $download
     *              TRUE = Download erzwingen, FALSE = Inline-Anzeige (z. B. Bild im Browser)
     *
     * @param       string|null $name
     *              Optionaler Dateiname für den Download
     *
     * @param       array  $headers
     *              Zusätzliche HTTP-Header
     *
     * @throws      \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
     *
     * @return      StreamedResponse
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function serve(string $file, bool $download = true, ?string $name = null, array $headers = [])
    {
        $path = $this->_basePath.'/'.$this->_normalize($file);

        if (!Storage::disk($this->_disk)->exists($path)) {
            abort(404, 'File ['.$file.'] not found.');
        }
        if ($download === true) {
            /*
            **  Für Downloads. */
            return Storage::disk($this->_disk)->download($path, $name, $headers);
        }
        /*
        **  Für Inline-Anzige. */
        $mimeType = Storage::disk($this->disk)->mimeType($path);

        if (!empty($mimeType)) {
            $headers['Content-Type'] = $mimeType;
        }
        return Storage::disk($this->_disk)->response($path, $name, $headers);

    } // serve()


    /**
     * Gibt eine Datei aus dem Tenant- oder dem .global-Verzeichnis zurück.
     *
     * Falls im Tenant nichts gefunden wird, wird aus ".global" geladen.
     *
     * @param       string $file
     *              Relativer Dateiname/Pfad innerhalb des Modul-Verzeichnisses.
     *
     * @param       bool $download
     *              TRUE = Download erzwingen, FALSE = Inline-Anzeige.
     *
     * @param       string|null $name
     *              Optionaler Dateiname für den Download.
     *
     * @param       array $headers
     *              Zusätzliche HTTP-Header.
     *
     * @return      StreamedResponse
     *
     * @throws      \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function serveGlobalFallback(string $file, bool $download = true, ?string $name = null, array $headers = [])
    {
        $pathTenant = $this->_basePath.'/'.$this->_normalize($file);

        if (Storage::disk($this->_disk)->exists($pathTenant)) {
            return $this->serve($file, $download, $name, $headers);
        }
        /*
        **  Prüfen, ob die Datei als Global-Version existiert. */
        $pathGlobal = $this->_moduleBasePath().'/'.self::GLOBAL_DIR.'/'.$file;

        if (Storage::disk($this->_disk)->exists($pathGlobal)) {
            /*
            **  Temporäres Storage-Objekt für das globale Verzeichnis. */
            $globalStorage = new self($this->_public, $this->_module, false);

            return $globalStorage->serve(self::GLOBAL_DIR.'/'.$file, $download, $name, $headers);
        }
        abort(404, 'File ['.$file.'] not found in either the tenant or global directory.');

    } // serveGlobalFallback()


    /**
     *
     * @return      Filesystem
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function storage(): Filesystem
    {
        return Storage::disk($this->_disk);

    } // storage()


    /**
     * Datei als Stream öffnen.
     *
     * @param       string $file
     *
     * @version     1.0.0 / 2025-08-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function stream(string $file)
    {
        $fullPath = $this->_basePath.'/'.$this->_normalize($file);

        if (!Storage::disk($this->_disk)->exists($fullPath)) {
            throw new \RuntimeException('File ['.$file.'] not found.');
        }
        return Storage::disk($this->_disk)->readStream($fullPath);

    } // stream()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class ModuleStorage {}
