<?php
/**
 * Middleware Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Middlewares
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Http\Middleware;


use App\Models\User;
use BplanBase\Globals\Enums\UserAccessLevel;
use BplanBase\Globals\Registries\Registry;
use BplanBase\Globals\Repositories\Core\TenantLocaleRepository;
use BplanBase\Globals\Repositories\Core\TenantRepository;
use BplanBase\Globals\Repositories\ModuleRepository;
use BplanBase\Globals\Repositories\UserRepository;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Session\Store;
use InvalidArgumentException;


/**
 * Middleware Class
 *
 * @version     2.3.0 / 2025-08-28
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class InitWebEnv
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     bool $_authCheck
     */
    private bool $_authCheck;


    /**
     * @var     string $_configTenant
     */
    private null|string $_configTenant;


    /**
     * @var     Store $_Session
     */
    private Store $_Session;


    /**
     * @var     bool $_singleTenantMode
     */
    private bool $_singleTenantMode = false;


    /**
     * @var     int|string $_tenantId
     */
    protected int|string $_tenantId;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Request $Request
     *
     * @param       Model $User
     *
     * @version     3.0.0 / 2025-07-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _changeTenant(Request $Request, Model $User)
    {
        /*
        **  Die Übergabe des Parameters "change-tenant" initiiert einen Wechsel des aktuellen
        **  Tenants. Mit der übergebenen Tenant-ID werden die Daten in der Session neu
        **  initialisiert. */
        $tenantId = trim($Request->input('change-tenant'));

        if ($this->_initSession($tenantId, $User, $status) === false) {
            return $this->_performRedirect('login', status: $status);
        }
        /*
        **  Abschließend werden die Query-Parameter ermittelt, der Parameter "change-tenant"
        **  wird entfernt und es wird ein Redirect auf die aktuelle Seite durchgeführt.
        **  Hintergrund dafür ist, dass die URL in der Adresszeile des Browsers hinterher
        **  wieder frei von dem Parameter ist. */
        $query = $Request->query();

        unset($query['change-tenant']);
        /*
            @todo   Das funktioniert leider nicht. Wenn die aktuelle View zum Beispiel ein
                    Formular ist, dann kommt es zu einem 404er, wenn der neue Tenant keinen
                    Zugriff auf das angezeigte Objekt hat.
                    Aus dem Grund wird erstmal auf das Dashboard umgeleitet.

                    Zur Lösung des Problems müsste erkannt werden können ob die aktuelle Route
                    sicher für die Anzeige beim neuen Tenant ist.

        return $this->_performRedirect($Request->route()->getName(), query: $query);
        */
        return $this->_performRedirect('dashboard', query: $query);

    } // _changeTenant()


    /**
     * handle-Methode für angemeldete Benutzer
     *
     * @param       Request $Request
     *
     * @param       Closure $next
     *
     * @return      mixed
     *
     * @version     1.2.0 / 2025-08-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _handleAuthenticated(Request $Request, Closure $next): mixed
    {
        $AuthUser = auth()->user();
        /*
        **  Reihenfolge der Abarbeitung (angemeldete Benutzer)
        **
        **  1. Der Mandantenwechsel muss vor allem anderen abgearbeitet werden, weil dieser Fall
        **     ansonsten nicht geprüft werden kann.
        **
        **  2. Prüfung der Session auf einen gespeicherten Mandanten.
        **     Das ist der Normalfall, wenn bereits einmal eine Initialisierung stattgefunden hat.
        **
        **  3. Das ist der Normalfall beim allerersten Request nach dem Login. Hier wird die initiale
        **     Ermittlung des Mandanten vorgenommen.
        **     a) Zunächst wird eine Prüfung auf den Einzel-Mandanten-Modus durchgeführt. In einem
        **        Einzel-Mandanten-System wird der aktuelle Mandant auf Basis des in der Konfiguration
        **        hinterlegten Mandaten initialisiert.
        **
        **     b) In einem Mehr-Mandanten-System wird der aktuelle Mandant auf Basis des angemeldeten
        **        Benutzers ermittelt. Daraus ergibt sich, dass die Anwendung immer im Kontext des
        **        direkt zugeordneten Mandant startet.
        */
        if ($Request->has('change-tenant')) {
            /*
            **  Aufruf mit einem Mandanten im Query-Parameter.
            **
            **  Die Übergabe des Parameters "change-tenant" initiiert einen Wechsel des aktuellen
            **  Tenants. Mit der übergebenen Tenant-ID werden die Daten in der Session neu
            **  initialisiert. */
            return $this->_changeTenant($Request, $AuthUser);

        } elseif ($this->_Session->has('auth.tenantId') && $this->_Session->get('auth.check') === true) {
            /*
            **  Arbeitsweise für alle Requests nach dem initialen Aufruf.
            **
            **  Registry mit Session-Daten initialisieren. */
            $this->_initRegistry($Request);

        } else {
            /*
            **  Standardverfahren beim initialen Aufruf ohne übergebenen Mandanten. */
            if ($this->_singleTenantMode === true) {
                $Tenant = TenantRepository::getByVarious($this->_configTenant, ignoreRestriction: true);

                if ($Tenant === null) {
                    throw new InvalidArgumentException('Configured tenant ['.$this->_configTenant.'] not found.');
                }
                if ($this->_initSession($Tenant->id, $AuthUser, $status) === false) {
                    return $this->_performRedirect('logout', status: $status);
                }
            } else {
                /*
                **  Wenn der Active-Status des Users FALSE ist oder das Initialisieren der Session-
                **  Daten fehlschlägt (weil der User keinen Tenant-Zugriff hat), dann wird der User
                **  direkt wieder ausgeloggt. */
                if ((bool) $AuthUser->active === false) {
                    return $this->_performRedirect('logout', __('globals::global.error.auto-logout.account-inactive'));
                }
                if ($this->_initSession($AuthUser->tenant_id, $AuthUser, $status) === false) {
                    return $this->_performRedirect('logout', status: $status);
                }
            }
            $this->_initRegistry($Request);
        }
        return $next($Request);

    } // _handleAuthenticated()


    /**
     * handle-Methode für nicht angemeldete Benutzer
     *
     * @param       Request $Request
     *
     * @param       Closure $next
     *
     * @return      mixed
     *
     * @version     1.3.0 / 2025-08-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _handleNotAuthenticated(Request $Request, Closure $next): mixed
    {
        /*
        **
        **  Reihenfolge der Abarbeitung (nicht angemeldet)
        **
        **  1. Wechsel des aktuell aktiven Mandanten.
        **     Prüfung, ob ein Mandant im Query-Parameter übergeben wurde.
        **
        **  2. Prüfung der Session auf einen gespeicherten Mandanten.
        **     Das ist der Normalfall, wenn bereits einmal eine Initialisierung stattgefunden hat.
        **
        **  3. Prüfung auf Einzel-Mandanten-Modus
        **     In einem Einzel-Mandanten-System wird der aktuelle Mandant auf Basis des in der
        **     Konfiguration hinterlegten Mandaten initialisiert.
        */
        $queryTenant = $Request->query('tenant');

        if ($queryTenant !== null) {
            /*
            **  Aufruf mit einem Mandanten im Query-Parameter. */
            $Tenant = TenantRepository::getByVarious($queryTenant, ignoreRestriction: true);
            $GuestUser = UserRepository::getByName('guest', ignoreRestriction: true);

            if ($this->_Session->has('auth.tenantId') && $this->_Session->get('auth.tenantId') === (int) $Tenant->id && $this->_Session->get('auth.check') === false) {
                /*
                **  Wenn die aktuelle Tenant-ID bereits in der Session existiert, dann muss die
                **  Initialisierung nicht erneut durchgeführt werden und die Registry kann direkt
                **  mit den Session-Daten initialisiert werden.
                **  Diese Prüfung wird durchgeführt weil die ohne Login zugänglichen Seiten eventuell
                **  immer mit einem Mandanten-Query-Parameter aufgerufen werden (z.B. CallInMonitor). */
                $this->_initRegistry($Request);

            } else {
                if ($this->_initSession($Tenant->id, $GuestUser, $status) === false) {
                    return $this->_performRedirect('login', status: $status);
                }
                $this->_initRegistry($Request);
            }
        } elseif ($this->_Session->has('auth.tenantId')) {
            /*
            **  Arbeitsweise für alle Requests nach dem initialen Aufruf.
            **
            **  Registry mit Session-Daten initialisieren. */
            $this->_initRegistry($Request);

        } else {
            /*
            **  Standardverfahren beim initialen Aufruf ohne übergebenen Mandanten. */
            if ($this->_singleTenantMode === true) {
                $tenant = $this->_configTenant;
            } else {
                $tenant = 'System';
            }
            $Tenant = TenantRepository::getByVarious($tenant, ignoreRestriction: true);

            if ($Tenant === null) {
                throw new InvalidArgumentException('Tenant ['.$tenant.'] not found.');
            }
            $GuestUser = UserRepository::getByName('guest', ignoreRestriction: true);

            if ($this->_initSession($Tenant->id, $GuestUser, $status) === false) {
                return $this->_performRedirect('login', status: $status);
            }
            $this->_initRegistry($Request);
        }
        return $next($Request);

    } // _handleNotAuthenticated()


    /**
     * Überträgt Session-Daten in die Registry
     *
     * @param       Request $Request
     *
     * @return      void
     *
     * @version     2.0.0 / 2025-08-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initRegistry(Request $Request): void
    {
        Registry::put('auth.check', $this->_Session->get('auth.check'), true);
        Registry::put('auth.masterTenantId', $this->_Session->get('auth.masterTenantId'), true);
        Registry::put('auth.tenantId', $this->_Session->get('auth.tenantId'), true);
        Registry::put('auth.tenantIds', $this->_Session->get('auth.tenantIds'), true);
        Registry::put('auth.tenantLocales', $this->_Session->get('auth.tenantLocales', [1 => config('app.fallback_locale')]), true);
        Registry::put('auth.tenants', $this->_Session->get('auth.tenants'), true);
        Registry::put('auth.userAccessLevel', $this->_Session->get('auth.userAccessLevel'), true);
        Registry::put('auth.userId', $this->_Session->get('auth.userId'), true);
        /*
        **  Get the route prefix from the request object. */
        $routePrefix = $Request->route()->getPrefix();

        if (empty($routePrefix)) {
            /*
            **  Reset module information in the registry. */
            Registry::remove('env');
            Registry::put('env.currentModule', null);

        } else {
            /*
            **  Determine the module via the route prefix and put the module
            **  information into the registry. */
            $Module = ModuleRepository::getByRoutePrefix($routePrefix);

            Registry::put('env.currentModule', $Module->route_prefix);
            Registry::put('env.moduleName', $Module->package_name);
        }
    } // _initRegistry()


    /**
     * Ermittelt Daten und legt diese zur späteren Verwendung in der Session ab
     *
     * @param       string $tenantId
     *              Die ID des Tenants dem der User direkt zugeordnet ist.
     *
     * @param       User $User
     *
     * @param       null|string & $statusMessage
     *
     * @return      bool FALSE, wenn der User keine ihm zugeordneten Tenants oder keine
     *              UserTenantAccess-Verknüpfungen hat.
     *
     * @version     1.7.0 / 2025-07-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initSession(string $tenantId, Model $User, null|string & $statusMessage = null): bool
    {
        /*
        **  Die Tenant-ID muss hier frühzeitig gesetzt werden, damit sie in den nachfolgend
        **  verwendeten Repository-Methoden aus der Registry gelesen werden kann. */
        $this->_Session->put('auth.tenantId', (int) $tenantId);
        Registry::put('auth.tenantId', (int) $tenantId);

        $tenants = TenantRepository::determineUserTenants($User, ignoreRestriction: true);

        if (empty($tenants)) {
            /*
            **  Dieser Fall dürfte nie eintreten.
            **  Ein User hat immer den Mandanten dem er direkt zugeordnet ist (auch wenn er selbst
            **  als "restricted" gekennzeichnet ist). Wenn er "restricted" ist und Zuordnungen zu
            **  anderen Mandanten existieren, dann besteht die Möglichkeit, dass er den Zugriff auf
            **  seinen direkt zugeordneten Mandanten verloren hat, dafür hat er dann aber Zugriff
            **  auf den/die extern zugeordneten Mandanten. */
            $this->_reset();

            $statusMessage = __('error.auto-logout.no-tenant');

            return false;
        }
        $TenantCollection = collect($tenants);

        if ($this->_authCheck === true) {
            /*
            **  Sicherstellen, dass die übergebene Tenant-ID auch in der Liste der Tenants des Users
            **  enthalten ist. */
            if (!isset($tenants[$tenantId])) {
                $ActiveTenants = $TenantCollection->where('active', '=', 1);
                /*
                **  Prüfen ob es aktive Mandanten zum User gibt.
                **  Wenn alle verfügbaren Mandanten inaktiv sind, dann wird der User direkt wieder
                **  ausgeloggt. Andernfalls wird der erste gültige Mandant als aktiver Mandant
                **  eingesetzt. */
                if ($ActiveTenants->isEmpty()) {
                    $this->_reset();

                    $statusMessage = __('error.auto-logout.no-active-tenant');

                    return false;
                }
                /*
                **  Bei einem angemeldeten User kann dieser Fall eigentlich nur dann eintreten, wenn
                **  der Request zum Wechsel des Mandanten manipuliert wurde.
                **  In diesem Fall wird der Kontext der Anwendung einfach auf einen der zugelassenen
                **  Mandaten gesetzt. */
                $tenant = $ActiveTenants->first();
                $tenantId = $tenant['id'];
            }
        } else {
            /*
                @todo   Derzeit noch undefiniert.
                        Wenn kein angemeldeter User existiert, dann wird der Guest-User verwendet.
                        Dieser User ist direkt dem System-Tenant zugeordnet und ist als nicht restricted
                        gekennzeichnet. Somit kann er in jedem Mandantenkontext agieren, so dass der
                        übergebene Mandant verwendet werden kann.
            */
        }
        $tenant = $tenants[$tenantId];
        $tenantLocales = TenantLocaleRepository::determineTenantLocales($tenantId);

        $this->_Session->put('auth.check', $this->_authCheck);
        $this->_Session->put('auth.masterTenantId', empty($tenant['master_id']) ? null : (int) $tenant['master_id']);
        $this->_Session->put('auth.tenantId', (int) $tenantId);
        $this->_Session->put('auth.tenantIds', array_keys($tenants));
        $this->_Session->put('auth.tenantLocales', $tenantLocales);
        $this->_Session->put('auth.tenants', $TenantCollection);
        $this->_Session->put('auth.userAccessLevel', $User->access_level ?? UserAccessLevel::Guest->name);
        $this->_Session->put('auth.userId', $User->id);

        return true;

    } // _initSession()


    /**
     *
     * @param       string $route
     *
     * @param       array $query
     *
     * @param       null|string $status
     *
     * @version     1.0.0 / 2025-07-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _performRedirect(string $route, array $query = [], null|string $status = null)
    {
        if ($route === 'login') {
            $this->_reset();

            if ($status === null) {
                $status = __('globals::global.error.auto-logout');
            }
            return redirect('login')->with('status', $status);

        } elseif ($route === 'logout') {
            if ($status === null) {
                $status = __('globals::global.error.auto-logout');
            }
            auth()->guard('web')->logout();

            $this->_Session->invalidate();
            $this->_Session->regenerateToken();

            return $this->_performRedirect('login', $query, $status);
        }
        return redirect()->route($route, $query);

    } // _performRedirect()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-07-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _reset(): void
    {
        Registry::remove('auth');
        $this->_Session->remove('auth');

    } // _reset()


    /**
     * Handle an incoming request.
     *
     * @param       Request $Request
     *
     * @param       Closure $next
     *
     * @return      mixed
     *
     * @version     1.5.0 / 2025-07-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function handle(Request $Request, Closure $next): mixed
    {
        $appLocale = $Request->cookie('app_locale') ?: config('app.locale');

        app()->setLocale($appLocale);

        Registry::put('appLocale', $appLocale);

        $this->_authCheck = auth()->check();
        $this->_configTenant = config('globals.global-tenant');
        $this->_Session = $Request->session();

        if (!empty($this->_configTenant)) {
            $this->_singleTenantMode = true;
        }
        if ($this->_authCheck === true) {
            return $this->_handleAuthenticated($Request, $next);
        }
        return $this->_handleNotAuthenticated($Request, $next);

    } // handle()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class InitWebEnv {}
