<?php
/**
 * Base User Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @generated   2025-05-12 09:02:42
 * @package     bplan-base/globals
 * @subpackage  Models
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Models;


use BplanBase\Globals\Enums\UserAccessLevel;
use BplanBase\Globals\Models\Role;
use BplanBase\Globals\Observers\UserObserver;
use BplanBase\Globals\Scopes\TenantScope;
use BplanBase\Globals\Traits\HasActiveScope;
use BplanBase\Globals\Traits\HasNoInternalScope;
use BplanBase\Globals\Traits\HasOrderedScope;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Str;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Laravel\Jetstream\HasProfilePhoto;
use Laravel\Sanctum\HasApiTokens;


/**
 * Base User Class
 *
 * @version     6.1.0 / 2025-09-01
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
#[ObservedBy([UserObserver::class])]
class User extends Authenticatable
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use HasActiveScope;
    use HasApiTokens;
    use HasNoInternalScope;
    use HasOrderedScope;
    use HasProfilePhoto;
    use Notifiable;
    use SoftDeletes;
    use TwoFactorAuthenticatable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_randomPassword
     */
    private $_randomPassword;


    /**
     * The accessors to append to the model's array form.
     *
     * @var         array<int, string> $appends
     *
     * @version     2.0.0 / 2025-06-20
     */
    protected $appends = [
        'full_name',
        'full_name_reverted',
        'profile_photo_url',
    ];


    /**
     * @var         array<string, string> $casts
     *
     * @version     2.0.0 / 2025-01-13
     */
    protected $casts = [
        'active'            => 'boolean',
        'created_at'        => 'datetime',
        'deleted_at'        => 'datetime',
        'email_verified_at' => 'datetime',
        'internal'          => 'boolean',
        'password'          => 'hashed',
        'restricted'        => 'boolean',
        'updated_at'        => 'datetime',
    ];


    /**
     * The attributes that are mass assignable.
     *
     * @var         array<int, string> $fillable
     *
     * @version     5.0.0 / 2025-05-20
     */
    protected $fillable = [
        'access_level',
        'active',
        'current_team_id',
        'email',
        'email_verified_at',
        'internal',
        'name',
        'password',
        'profile_photo_path',
        'restricted',
    ];


    /**
     * The attributes that aren't mass assignable.
     *
     * @var         array<int, string> $guarded
     *
     * @version     3.0.0 / 2025-05-20
     */
    protected $guarded = [
        'created_at',
        'deleted_at',
        'remember_token',
        'tenant_id',
        'updated_at',
        'uuid',
    ];


    /**
     * The attributes that should be hidden for serialization.
     *
     * @var         array<int, string> $hidden
     *
     * @version     1.0.0 / 2024-11-03
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];


    /**
     * The table associated with the model.
     *
     * @var         string $table
     */
    protected $table = 'users';


    /**
     * Indicates if the model should be timestamped.
     *
     * @var         bool $timestamps
     */
    public $timestamps = true;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       bool $reverted
     *              Der Parameter $reverted hat beim User keine Auswirkung, weil es keine Vor- und
     *              Nachnamen gibt, die unterschiedlich zusammengestellt werden können. Er ist nur
     *              aus Gründen der Kompatibilität zu den gleichnamigen Methoden zum Beispiel bei
     *              VisitEmployees und VisitVisitors vorhanden.
     *
     * @return 	    string
     *
     * @version     2.0.0 / 2025-02-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFullName(bool $reverted = false): string
    {
        return $this->name;

    } // getFullName()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-06-19
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFullNameAttribute(): string
    {
        return $this->name;

    } // getFullNameAttribute()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-06-19
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFullNameRevertedAttribute(): string
    {
        return $this->name;

    } // getFullNameRevertedAttribute()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-02-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRandomPassword(): string
    {
        if ($this->_randomPassword === null) {
            $this->_randomPassword = Str::random(12);
        }
        return $this->_randomPassword;

    } // getRandomPassword()


    /**
     *
     * @param       array<int, string>|string $roles
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-04-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasAnyRole(array|string $roles): bool
    {
        if (is_string($roles)) {
            $roles = (array) $roles;
        }
        return $this->roles()->whereIn('identifier', $roles)->exists();

    } // hasAnyRole()


    /**
     *
     * @param       int|string|UserAccessLevel $requiredLevel
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-04-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasMinAccessLevel(int|string|UserAccessLevel $requiredLevel): bool
    {
        $userAccessLevel = UserAccessLevel::getValue($this->access_level);

        if ($requiredLevel instanceof UserAccessLevel) {
            return $userAccessLevel >= $requiredLevel->value;

        } elseif (is_int($requiredLevel)) {
            return $userAccessLevel >= $requiredLevel;
        }
        return $userAccessLevel >= UserAccessLevel::getValue($requiredLevel);

    } // hasMinAccessLevel()


    /**
     *
     * @param       string $identifier
     *
     * @return      bool
     *
     * @deprecated  use hasAnyRole() instead.
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     */
    public function hasRole(string $identifier): bool
    {
        $isAdmin = $this->roles->contains(function (Role $role, int $key) use ($identifier) {
            return $role->identifier === 'Admin';
        });
        if($isAdmin === true) {
            return true;
        }
        return $this->roles->contains(function (Role $role, int $key) use ($identifier) {
            return $role->identifier === $identifier;
        });
    } // hasRole()


    /**
     *
     * @return      HasMany
     *
     * @version     1.0.0 / 2025-06-10
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function importLogs(): HasMany
    {
        return $this->hasMany(ImportLog::class, 'import_user_id');

    } // importLogs()


    /**
     *
     * @return      HasOne
     *
     * @version     1.0.0 / 2025-05-12
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function individual(): HasOne
    {
        return $this->hasOne(Individual::class, 'user_id');

    } // individual()


    /**
     *
     * @return      BelongsToMany
     *
     * @version     1.2.0 / 2025-09-01
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function roles(): BelongsToMany
    {
        /*
        **  @todo   Übergangslösung: Es werden immer alle Rollen des Users ermittelt, unabhängig
        **          davon, bei welchem Tenant diese Rollen definiert sind. Im Normalfall können
        **          einem User eh nur Rollen bei seinem direkten Tenant zugeordnet werden.
        **          So sind die Rollen des Users in jedem Kontext verfügbar (insbesondere wichtig
        **          für die Admin-Rolle in der Demo).
        **
        **          Da geplant ist, dass die gleiche Arbeitsweise für Berechtigungen implementiert
        **          werden soll (Berechtigungen des Users werden bei seinem direkten Tenant
        **          definiert und gelten bei allen anderen Tenants, auf die der User Zugriff hat)
        **          könnte es sein, dass diese Ausnahme hier zur Regel wird und bestehen bleibt. */
        return $this->belongsToMany(Role::class, 'role_user', 'user_id', 'role_id')
            ->withPivot('active', 'created_at', 'uuid')
            ->withoutGlobalScope(TenantScope::class);

    } // roles()


    /**
     * Route notifications for the mail channel.
     *
     * @param       Notification $Notification
     *
     * @return      array<string, string>|string
     *
     * @version     1.1.0 / 2025-01-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function routeNotificationForMail(Notification $Notification): array|string
    {
        return [$this->email => $this->getFullName()];

    } // routeNotificationForMail()


    /**
     *
     * @return      BelongsTo
     *
     * @version     1.0.0 / 2025-05-03
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class, 'tenant_id');

    } // tenant()


    /**
     *
     * @return      BelongsToMany
     *
     * @version     1.0.0 / 2025-06-01
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function tenants(): BelongsToMany
    {
        return $this->belongsToMany(Tenant::class, 'tenant_user', 'user_id', 'tenant_id')->withPivot('access_level', 'created_at', 'updated_at', 'uuid');

    } // tenants()


    /**
     *
     * @return      HasOne
     *
     * @version     1.0.0 / 2025-04-16
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function userRecord(): HasOne
    {
        return $this->hasOne(UserRecord::class, 'id');

    } // userRecord()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-05-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected static function booted(): void
    {
        static::addGlobalScope(new TenantScope());

    } // booted()


} // class User extends Authenticatable {}
