<?php
/**
 * Package Service Provider Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Providers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals;


use BplanBase\Globals\Http\Middleware\CheckAccessLevelOrRoles;
use BplanBase\Globals\Http\Middleware\InitApiEnv;
use BplanBase\Globals\Http\Middleware\InitWebEnv;
use BplanBase\Globals\Http\Middleware\LogApiRequest;
use BplanBase\Globals\Providers\EventServiceProvider;
use BplanBase\Globals\Registries\SeederRegistry;
use BplanBase\Globals\Services\HttpService;
use BplanBase\Globals\Support\LivewireComponentAutoDiscovery;
use Illuminate\Database\Events\QueryExecuted;
use Illuminate\Routing\Router;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Blade;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;


/**
 * Package Service Provider Class
 *
 * @version     8.0.0 / 2025-09-01
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class PackageServiceProvider extends ServiceProvider
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_basePath
     */
    protected string $_basePath;


    /**
     * @var         string $_baseNamespace
     */
    protected string $_baseNamespace = 'BplanBase\\Globals';


    /**
     * Indicates if loading of the provider is deferred.
     *
     * @var         bool $defer
     */
    protected bool $defer = false;


    /**
     * @var         int $_packageKey
     */
    protected int $_packageKey = 1000;


    /**
     * @var         string $_packageName
     */
    protected string $_packageName = 'globals';


    /**
     * @var         string $_vendorPackageName
     */
    protected string $_vendorPackageName = 'bplan-base/globals';


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.2.0 / 2025-08-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareSqlLogging(): void
    {
        if (Config::get('globals.db.sql-logging', false) === true) {
            /*
            **  Register log channel at runtime. */
            $this->_registerSqlLogChannel();
            /*
            **  Log SQL queries. */
            DB::listen(function (QueryExecuted $Query) {
                Log::channel('sql')->info(
                    "\n".'  '.$Query->sql
                        ."\n".'  Bindings: '.((empty($Query->bindings)) ? '-' : print_r($Query->bindings, true))
                        ."\n".'  Runtime : '.$Query->time.'s'
                        ."\n"
                );
            });
        }
    } // _prepareSqlLogging()


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-08-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerAppLogChannel(): void
    {
        $logging = Config::get('logging');
        $logLevel = Config::get('logging.channels.slack.level');

        $logging['channels']['applog'] = [
            'driver' => 'daily',
            'path' => storage_path('logs/app/app.log'),
            'level' => $logLevel,
            'days' => 14,
            'replace_placeholders' => true,
        ];
        $logging['channels']['applog-composite'] = [
            'driver' => 'stack',
            'channels' => ['applog', $logging['default']],
            'ignore_exceptions' => false,
        ];
        Config::set('logging', $logging);

    } // _registerAppLogChannel()


    /**
     *
     * @return      void
     *
     * @version     1.1.0 / 2025-08-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerFacadeAlias()
    {
        if (!class_exists('AppLog')) {
            class_alias(\BplanBase\Globals\Facades\AppLog::class, 'AppLog');
            class_alias(\BplanBase\Globals\Facades\Registry::class, 'Registry');
        }
    } // _registerFacadeAlias()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerMiddlewares(Router $Router): void
    {
        $Router->aliasMiddleware('access-level.or.roles', CheckAccessLevelOrRoles::class);

    } // _registerMiddlewares()


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-08-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerSqlLogChannel(): void
    {
        /*
        **  Check whether the channel already exists to avoid duplicate entries. */
        if (!array_key_exists('sql', Config::get('logging.channels', []))) {
            Config::set('logging.channels.sql', [
                'driver' => 'daily',
                'path' => storage_path('logs/sql/sql.log'),
                'level' => 'info',
                'days' => 14,
                'replace_placeholders' => true,
            ]);
        }
    } // _registerSqlLogChannel()


    /**
     * Bootstrap the application events.
     *
     * @return      void
     *
     * @version     2.9.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function boot(Router $Router): void
    {
        /*
        **  Inform Laravel about translations. */
        $this->loadTranslationsFrom($this->_basePath.'/lang', $this->_packageName);

        if ($this->app->runningInConsole()) {
            $this->commands([
                \BplanBase\Globals\Console\Commands\GlobalsSetup::class,
                // \BplanBase\Globals\Console\Commands\HealthCheck::class,
                // \BplanBase\Globals\Console\Commands\SelfCheck::class,
                \BplanBase\Globals\Console\Commands\WarmUpApp::class,
            ]);
            require_once $this->_basePath.'/helpers/seeder-helpers.php';

            SeederRegistry::register($this->_packageKey, $this->_vendorPackageName, \BplanBase\Globals\Database\Seeders\DatabaseSeeder::class);
            SeederRegistry::register($this->_packageKey + 2000, $this->_vendorPackageName, \BplanBase\Globals\Database\Seeders\PostDatabaseSeeder::class);
            SeederRegistry::register($this->_packageKey + 3000, $this->_vendorPackageName, \BplanBase\Globals\Database\Seeders\FinishSeeder::class);
        }
        $this->_registerAppLogChannel();
        $this->_prepareSqlLogging();
        /*
        **  Inform Laravel about views, define a namespace for Blade components and register Livewire components. */
        $this->loadViewsFrom($this->_basePath.'/resources/views', $this->_packageName);

        Blade::componentNamespace('BplanBase\\Globals\\View\\Components', $this->_packageName);
        LivewireComponentAutoDiscovery::register($this->_baseNamespace.'\\Livewire', $this->_basePath.'/src/Livewire', $this->_packageName);

        if ($this->app->runningInConsole()) {
            /*
            **  Inform Laravel about migrations. */
            $this->loadMigrationsFrom($this->_basePath.'/database/migrations');
        }
        $this->_registerMiddleWares($Router);
        /*
        **  Load package routes. */
        $this->loadRoutesFrom($this->_basePath.'/routes/api.php');
        $this->loadRoutesFrom($this->_basePath.'/routes/web.php');

    } // boot()


    /**
     * Register the service provider.
     *
     * @return      void
     *
     * @version     1.15.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function register()
    {
        if (!defined('ENV_START')) {
            define('ENV_START', microtime(true));
            define('ENV_TIME', time());
        }
        $this->_basePath = str_replace('\\', '/', realpath(__DIR__.'/..'));
        /*
        **  Funktionsbibliotheken laden.
        **  Hier werden nur die Helper geladen werden, die frühzeitig im Bootstrapping zur
        **  Verfügung stehen müssen. Alle anderen können regulär in composer.json eingebunden
        **  werden. */
        require_once $this->_basePath.'/helpers/env-helpers.php';
        /*
        **  Load the package configuration. */
        $this->mergeConfigFrom($this->_basePath.'/config/'.$this->_packageName.'.php', $this->_packageName);
        $this->mergeConfigFrom($this->_basePath.'/config/dynamic-data-handling.php', 'dynamic-data-handling');
        $this->mergeConfigFrom($this->_basePath.'/config/setup.php', $this->_packageName.'-setup');

        if ($this->app->runningInConsole() === false) {
            $currentModule = request()->route()?->getPrefix() ?? null;

            $this->app->register(EventServiceProvider::class);
            $this->app->register(\BplanBase\Globals\Providers\NavigationServiceProvider::class);
        }
        $this->app->bind(HttpService::class, function() {
            return new HttpService();
        });
        $this->_registerFacadeAlias();

    } // register()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $group
     *              "api", "web"
     *
     * @param       string $getForUsage
     *              "append", "prepend"
     *
     * @param       array|string $append
     *
     * @param       array|string $prepend
     *
     * @return      array
     *
     * @version     1.1.0 / 2025-05-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getMiddlewares(string $group, string $getForUsage, array|string $append = [], array|string $prepend = []): array
    {
        $middlewares = [];

        switch ($group) {
            case 'api':
                if ($getForUsage === 'append') {
                    $middlewares = [
                        LogApiRequest::class,
                        InitApiEnv::class,
                    ];
                } else {
                    //
                }
                break;

            case 'web':
                if ($getForUsage === 'append') {
                    $middlewares = [
                        InitWebEnv::class,
                    ];
                } else {
                    //
                }
                break;

            default:
                //
        }
        if (!empty($append)) {
            $middlewares = array_merge($middlewares, Arr::wrap($append));
        }
        if (!empty($prepend)) {
            $middlewares = array_merge(Arr::wrap($prepend), $middlewares);
        }
        return $middlewares;

    } // getMiddlewares()


} // class PackageServiceProvider extends ServiceProvider {}
