<?php
/**
 * Repository Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Repositories
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Repositories;


use BplanBase\Globals\Foundation\Repositories\BaseRepository;
use BplanBase\Globals\Models\EntityType;
use BplanBase\Globals\Models\EntityTypeModel;
use BplanBase\Globals\Repositories\EntityTypeModelRepository;
use BplanBase\Globals\Repositories\EntityTypeRepository;
use BplanBase\Globals\ValueObjects\PaginateArgs;
use BplanBase\Globals\Registries\Registry;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Schema;


/**
 * Repository Class
 *
 * @version     1.0.0 / 2025-06-17
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class EntityDataRepository extends BaseRepository
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       int|string $entityTypeId
     *
     * @param       bool $ignoreCache
     *
     * @return 	    EntityTypeModel
     *
     * @version     1.0.0 / 2025-06-17
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _resolveEntityTypeModel(int|string $entityTypeId, bool $ignoreCache = false): EntityTypeModel
    {
        $EntityType = EntityTypeRepository::getById($entityTypeId, ignoreCache: $ignoreCache);

        return EntityTypeModelRepository::getById($EntityType->model_id);

    } // _resolveEntityTypeModel()


    /**
     * Returns a collection of models for the given EntityType
     *
     * @param       int|string $entityTypeId
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the
     *              method cache already contains a suitable entry. The storage of
     *              a value in the method cache is not affected by this parameter.
     *
     * @param       array|string|null $orderBy
     *              @see BplanBase\Globals\Helpers\QueryHelper::applyOrderBy() for
     *              details.
     *
     * @param       array|bool|int $paginate
     *              Can be either the number of elements, TRUE or an array. If
     *              TRUE is passed, the default number from the configuration is
     *              used. If an array is passed, then all known arguments for
     *              paginate() (columns, page, pageName, perPage, total) can be
     *              specified in it (all optional).
     *              @see https://api.laravel.com/docs/12.x/Illuminate/Database/Eloquent/Builder.html#method_paginate
     *
     * @param       Builder|null $Query
     *
     * @param       bool $withInactive
     *              By default, only the active data records are returned. By
     *              setting this parameter to TRUE, the inactive data records can
     *              also be read out.
     *
     * @return      Collection|LengthAwarePaginator<Model>
     *
     * @version     1.0.0 / 2025-06-17
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getAll(int|string $entityTypeId, bool $ignoreCache = false, array|string|null $orderBy = null, array|bool|int $paginate = false, Builder|null $Query = null, bool $withInactive = false): Collection|LengthAwarePaginator
    {
        static $cache = [];

        $cacheKey = md5(json_encode(func_get_args()));

        if ($paginate !== false) {
            $ignoreCache = true;
        }
        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $EntityTypeModel = self::_resolveEntityTypeModel($entityTypeId, ignoreCache: $ignoreCache);

        static::$_modelClass = $EntityTypeModel->fully_qualified_class_name;

        $Query = self::_resolveQuery($Query)
            ->where('entity_type_id', '=', $entityTypeId);

        if (Schema::hasColumn($EntityTypeModel->table_name, 'active')) {
            if ($withInactive === false) {
                $Query->active();
            }
        }
        $Query->ordered($orderBy);

        if ($paginate === false) {
            return $cache[$cacheKey] = $Query->get();
        }
        $PaginateArgs = PaginateArgs::from($paginate);

        return $Query->paginate(...$PaginateArgs->toArray());

    } // getAll()


    /**
     * Returns the model instance for the given EntityType and ID
     *
     * @param       int|string $entityTypeId
     *
     * @param       int|string $id
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $returnOrFail
     *              Controls the return of the method.
     *              If no suitable entry can be found in the database, either NULL is returned (TRUE)
     *              or an exception is thrown (FALSE). This allows the result of the method call to
     *              be used directly in the Api controller.
     *
     * @throws      Illuminate\Database\Eloquent\ModelNotFoundException
     *
     * @return      EntityType|null
     *
     * @version     1.0.0 / 2025-06-17
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getById(int|string $entityTypeId, int|string $id, bool $ignoreCache = false, bool $returnOrFail = true): Model|null
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $EntityTypeModel = self::_resolveEntityTypeModel($entityTypeId, ignoreCache: $ignoreCache);

        $modelClass = $EntityTypeModel->fully_qualified_class_name;

        $Query = $modelClass::where('entity_type_id', '=', $entityTypeId);

        if ($returnOrFail === true) {
            return $cache[$cacheKey] = $Query->find($id);
        }
        return $cache[$cacheKey] = $Query->findOrFail($id);

    } // getById()


    /**
     * Returns the model instance for the given EntityType and UUID
     *
     * @param       int|string $entityTypeId
     *
     * @param       string $uuid
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $returnOrFail
     *              Controls the return of the method.
     *              If no suitable entry can be found in the database, either NULL is returned (TRUE)
     *              or an exception is thrown (FALSE). This allows the result of the method call to
     *              be used directly in the Api controller.
     *
     * @throws      Illuminate\Database\Eloquent\ModelNotFoundException
     *
     * @return      EntityType|null
     *
     * @version     1.0.0 / 2025-06-17
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getByUuid(int|string $entityTypeId, int|string $uuid, bool $ignoreCache = false, bool $returnOrFail = true): Model|null
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $EntityTypeModel = self::_resolveEntityTypeModel($entityTypeId, ignoreCache: $ignoreCache);

        $modelClass = $EntityTypeModel->fully_qualified_class_name;

        $Query = $modelClass::where('entity_type_id', '=', $entityTypeId)
            ->where('uuid', '=', $uuid);

        if ($returnOrFail === true) {
            return $cache[$cacheKey] = $Query->first();
        }
        return $cache[$cacheKey] = $Query->firstOrFail();

    } // getByUuid()


} // class EntityDataRepository extends BaseRepository {}
