<?php
/**
 * Base User Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     MaxPlanck
 * @subpackage  Models
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Models;


use BplanBase\Globals\Enums\UserAccessLevel;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Str;
use Laravel\Fortify\TwoFactorAuthenticatable;
// use Laravel\Jetstream\HasProfilePhoto;
// use Laravel\Sanctum\HasApiTokens;


/**
 * Base User Class
 *
 * @version     1.0.0 / 2025-04-15
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class User extends Authenticatable
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    // use HasApiTokens;
    // use HasProfilePhoto;
    use Notifiable;
    use TwoFactorAuthenticatable;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       bool $reverted
     *              Der Parameter $reverted hat beim User keine Auswirkung, weil es keine Vor- und
     *              Nachnamen gibt, die unterschiedlich zusammengestellt werden können. Er ist nur
     *              aus Gründen der Kompatibilität zu den gleichnamigen Methoden bei Employees und
     *              Visitors vorhanden.
     *
     * @return 	    string
     *
     * @version     2.0.0 / 2025-02-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFullName(bool $reverted = false): string
    {
        return $this->name;

    } // getFullName()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-02-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRandomPassword(): string
    {
        if ($this->_randomPassword === null) {
            $this->_randomPassword = Str::random(12);
        }
        return $this->_randomPassword;

    } // getRandomPassword()


    /**
     *
     * @param       array<int, string>|string $roles
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-04-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasAnyRole(array|string $roles): bool
    {
        if (is_string($roles)) {
            $roles = (array) $roles;
        }
        return $this->roles()->whereIn('identifier', $roles)->exists();

    } // hasAnyRole()


    /**
     *
     * @param       int|string|UserAccessLevel $requiredLevel
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-04-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasMinAccessLevel(int|string|UserAccessLevel $requiredLevel): bool
    {
        $userAccessLevel = UserAccessLevel::getValue($this->access_level);

        if ($requiredLevel instanceof UserAccessLevel) {
            return $userAccessLevel >= $requiredLevel->value;

        } elseif (is_int($requiredLevel)) {
            return $userAccessLevel >= $requiredLevel;
        }
        return $userAccessLevel >= UserAccessLevel::getValue($requiredLevel);

    } // hasMinAccessLevel()


    /**
     *
     * @param       string $identifier
     *
     * @return      bool
     *
     * @deprecated  use hasAnyRole() instead.
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     */
    public function hasRole(string $identifier): bool
    {
        $isAdmin = $this->roles->contains(function (Role $role, int $key) use ($identifier) {
            return $role->identifier === 'Admin';
        });

        if($isAdmin === true) {
            return true;
        }
        return $this->roles->contains(function (Role $role, int $key) use ($identifier) {
            return $role->identifier === $identifier;
        });
    } // hasRole()


    /**
     *
     * @return      BelongsToMany
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class);

    } // roles()


    /**
     * Route notifications for the mail channel.
     *
     * @param       Notification $Notification
     *
     * @return      array<string, string>|string
     *
     * @version     1.1.0 / 2025-01-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function routeNotificationForMail(Notification $Notification): array|string
    {
        return [$this->email => $this->getFullName()];

    } // routeNotificationForMail()


    /**
     *
     * @return      HasOne
     *
     * @version     1.0.0 / 2025-04-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function userRecord(): HasOne
    {
        return $this->hasOne(UserRecord::class, 'id');

    } // userRecord()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class User extends Authenticatable {}
