<?php
/**
 * Package Service Provider Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Providers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals;


use BplanBase\Globals\Http\Middleware\CheckAccessLevelOrRoles;
use BplanBase\Globals\Http\Middleware\LogApiRequest;
use BplanBase\Globals\Http\Middleware\SetLocale;
use BplanBase\Globals\Services\HttpService;
use Illuminate\Database\Events\QueryExecuted;
use Illuminate\Routing\Router;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Livewire\Livewire;


/**
 * Package Service Provider Class
 *
 * @version     3.2.0 / 2025-04-25
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class PackageServiceProvider extends ServiceProvider
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_basePath
     */
    protected $_basePath;


    /**
     * @var         string $_packageName
     */
    protected $_packageName = 'globals';


    /**
     * Indicates if loading of the provider is deferred.
     *
     * @var         bool $defer
     */
    protected $defer = false;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareSqlLogging(): void
    {
        if (Config::get('globals.db.sql-logging', false) === true) {
            /*
            **  Register log channel at runtime. */
            $this->_registerLogChannel();
            /*
            **  Log SQL queries. */
            DB::listen(function (QueryExecuted $Query) {
                Log::channel('sql')->info(
                    $Query->sql
                        ."\n".'  Bindings: '.((empty($Query->bindings)) ? '-' : print_r($Query->bindings, true))
                        ."\n".'  Runtime : '.$Query->time.'s'
                        ."\n"
                );
            });
        }
    } // _prepareSqlLogging()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _registerLivewireComponents(): void
    {
        /*
        **  Register Livewire components. */
        if (class_exists(Livewire::class)) {
            Livewire::component('globals::countries.index', \BplanBase\Globals\Livewire\Countries\Index::class);

            //  todo
            // $basePath = $this->_basePath.'/src/Livewire';

            // foreach (File::allFiles($basePath) as $file) {
            //     $relativePath = $file->getRelativePathname();
            //     $class = str_replace(
            //         ['/', '.php'],
            //         ['\\', ''],
            //         '\\BplanBase\\Globals\\Livewire\\'.$relativePath
            //     );
            //     Livewire::component('globals::'.strtolower($file->getFilenameWithoutExtension()), $class);
            // }
        }
    } // _registerLivewireComponents()


    /**
     * Bootstrap the application events.
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-03-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerLogChannel()
    {
        /*
        **  Check whether the channel already exists to avoid duplicate entries. */
        if (!array_key_exists('sql', Config::get('logging.channels', []))) {
            Config::set('logging.channels.sql', [
                'driver' => 'daily',
                'path' => storage_path('logs/sql/sql.log'),
                'level' => 'info',
                'days' => 14,
                'replace_placeholders' => true,
            ]);
        }
    } // _registerLogChannel()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerMiddlewares(Router $Router)
    {
        $Router->aliasMiddleware('access-level.or.roles', CheckAccessLevelOrRoles::class);

    } // _registerMiddlewares()


    /**
     * Bootstrap the application events.
     *
     * @return      void
     *
     * @version     2.2.0 / 2025-04-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function boot(Router $Router): void
    {
        $this->_prepareSqlLogging();
        /*
        **  Inform Laravel about translations. */
        $this->loadTranslationsFrom($this->_basePath.'/lang', $this->_packageName);
        /*
        **  Inform Laravel about views. */
        $this->loadViewsFrom($this->_basePath.'/resources/views', $this->_packageName);

        if ($this->app->runningInConsole()) {
            /*
            **  Inform Laravel about migrations. */
            $this->loadMigrationsFrom($this->_basePath.'/database/migrations');

            $this->publishes([
                $this->_basePath.'/etc/setup/app/Models/User.php'                                          => app_path('Models/User.php'),
                $this->_basePath.'/etc/setup/app/Observers/UserObserver.php'                               => app_path('Observers/UserObserver.php'),
                $this->_basePath.'/etc/setup/database/migrations/0001_01_01_000000_create_users_table.php' => database_path('migrations/0001_01_01_000000_create_users_table.php'),

            ], 'setup');
        }
        $this->_registerLivewireComponents();
        $this->_registerMiddleWares($Router);
        /*
        **  Load package routes. */
        $this->loadRoutesFrom($this->_basePath.'/routes/api.php');
        $this->loadRoutesFrom($this->_basePath.'/routes/web.php');

    } // boot()


    /**
     * Register the service provider.
     *
     * @return      void
     *
     * @version     1.3.0 / 2025-04-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function register(): void
    {
        $this->_basePath = str_replace('\\', '/', realpath(__DIR__.'/..'));
        /*
        **  Load the package configuration. */
        $this->mergeConfigFrom($this->_basePath.'/config/'.$this->_packageName.'.php', $this->_packageName);

        if ($this->app->runningInConsole()) {
            $this->commands([
                \BplanBase\Globals\Console\Commands\GlobalsSetup::class,
            ]);
        }
        $this->app->bind(HttpService::class, function() {
            return new HttpService();
        });
    } // register()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getApiMiddlewares(): array
    {
        return [
            LogApiRequest::class
        ];
    } // getApiMiddlewares()


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getWebMiddlewares(): array
    {
        return [
            SetLocale::class
        ];
    } // getWebMiddlewares()


} // class PackageServiceProvider extends ServiceProvider {}
