<?php
/**
 * Settings Model Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Models
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Models;


use BplanBase\Globals\Observers\SettingObserver;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;


/**
 * Settings Model Class
 *
 * @version     1.0.0 / 2025-05-03
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
#[ObservedBy([SettingObserver::class])]
class Setting extends Model
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * The accessors to append to the model's array form.
     *
     * @var         array<int, string> $appends
     *
     * @version     1.0.0 / 2025-05-03
     */
    protected $appends = [];


    /**
     * @var         array<string, string> $casts
     *
     * @version     1.0.0 / 2025-05-03
     */
    protected $casts = [
        'context' => 'string',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'value'   => 'string',
    ];


    /**
     * The attributes that are mass assignable.
     *
     * @var         array<int, string> $fillable
     *
     * @version     1.0.0 / 2025-05-03
     */
    protected $fillable = [
        'context',
        'key',
        'value',
    ];


    /**
     * The attributes that aren't mass assignable.
     *
     * @var         array<int, string> $guarded
     *
     * @version     1.0.0 / 2025-05-03
     */
    protected $guarded = [
        'created_at',
        'updated_at',
        'uuid',
    ];


    /**
     * The attributes that should be hidden for serialization.
     *
     * @var         array<int, string> $hidden
     *
     * @version     1.0.0 / 2025-05-03
     */
    protected $hidden = [];


    /**
     * The table associated with the model.
     *
     * @var         string $table
     */
    protected $table = 'settings';


    /**
     * Indicates if the model should be timestamped.
     *
     * @var         bool $timestamps
     */
    public $timestamps = true;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Scope: Filter nach Kontext
     *
     * @param       Builder $Query
     *
     * @param       string $context
     *
     * @return      Builder
     *
     * @version     1.0.0 / 2025-05-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function scopeContext(Builder $Query, string $context): Builder
    {
        return $Query->where('context', $context);

    } // scopeContext()


    /**
     * Scope: Filter nach Schlüssel innerhalb eines Kontextes
     *
     * @param       Builder $Query
     *
     * @param       string $context
     *
     * @param       string $key
     *
     * @return      Builder
     *
     * @version     1.0.0 / 2025-05-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function scopeContextKey(Builder $Query, string $context, string $key): Builder
    {
        return $Query->where('context', $context)->where('key', $key);

    } // scopeContextKey()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Wert lesen
     *
     * @param       string $context
     *
     * @param       string $key
     *
     * @param       mixed $default
     *
     * @return      mixed
     *
     * @version     1.0.0 / 2025-05-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getValue(string $context, string $key, mixed $default = null): mixed
    {
        return static::contextKey($context, $key)->value('value') ?? $default;

    } // getValue()


    /**
     * Wert setzen oder aktualisieren
     *
     * @param       string $context
     *
     * @param       string $key
     *
     * @param       mixed $value
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-05-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function setValue(string $context, string $key, mixed $value): void
    {
        static::updateOrCreate(
            [
                'context' => $context,
                'key' => $key

            ], [
                'uuid' => Str::uuid(),
                'value' => $value,
            ]
        );
    } // setValue()


 } // class Setting extends Model {}