<?php
/**
 * Repository Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Repositories
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Repositories;


use App\Models\User;
use Illuminate\Database\Eloquent\Collection;


/**
 * Repository Class
 *
 * @version     3.0.0 / 2025-05-03
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class UserRepository
{

/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Returns a collection of models
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $withInactive
     *              By default, only the active data records are returned. By setting this parameter
     *              to TRUE, the inactive data records can also be read out.
     *
     * @param       bool $withInternal
     *              By default, only the data records marked as “internal = 0” are read out. By
     *              setting this parameter to TRUE, the internal status is ignored and all data
     *              records are read.
     *
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getAll(bool $ignoreCache = false, bool $withInactive = false, bool $withInternal = false): Collection
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = User::query();

        if ($withInactive === false) {
            $Query->where('active', '=', 1);
        }
        if ($withInternal === false) {
            $Query->where('internal', '=', 0);
        }
        return $cache[$cacheKey] = $Query->get();

    } // getAll()


    /**
     * Returns the model instance for the given email
     *
     * @param       string $email
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $returnOrFail
     *              Controls the return of the method.
     *              If no suitable entry can be found in the database, either NULL is returned (TRUE)
     *              or an exception is thrown (FALSE). This allows the result of the method call to
     *              be used directly in the Api controller.
     *
     * @throws      Illuminate\Database\Eloquent\ModelNotFoundException
     *
     * @return      User
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getByEmail(string $email, bool $ignoreCache = false, bool $returnOrFail = true): User
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = User::query();

        if ($returnOrFail === true) {
            return $Query->where('email', '=', $email)
                ->first();
        }
        return $Query->where('email', '=', $email)
            ->firstOrFail();

    } // getByEmail()


    /**
     * Returns the model instance for the given ID
     *
     * @param       int|string $id
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $returnOrFail
     *              Controls the return of the method.
     *              If no suitable entry can be found in the database, either NULL is returned (TRUE)
     *              or an exception is thrown (FALSE). This allows the result of the method call to
     *              be used directly in the Api controller.
     *
     * @throws      Illuminate\Database\Eloquent\ModelNotFoundException
     *
     * @return      User|null
     *
     * @version     2.1.0 / 2025-05-03
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getById(int|string $id, bool $ignoreCache = false, bool $returnOrFail = true): User|null
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = User::query();

        if ($returnOrFail === true) {
            return $Query->find($id);
        }
        return $Query->findOrFail($id);

    } // getById()


    /**
     * Returns the model instance for the given UUID
     *
     * @param       int|string $uuid
     *
     * @param       bool $ignoreCache
     *              Controls the use of the method cache when determining the result.
     *              If TRUE is passed, a database access is carried out even if the method cache
     *              already contains a suitable entry. The storage of a value in the method cache
     *              is not affected by this parameter.
     *
     * @param       bool $returnOrFail
     *              Controls the return of the method.
     *              If no suitable entry can be found in the database, either NULL is returned (TRUE)
     *              or an exception is thrown (FALSE). This allows the result of the method call to
     *              be used directly in the Api controller.
     *
     * @throws      Illuminate\Database\Eloquent\ModelNotFoundException
     *
     * @return      User|null
     *
     * @version     1.0.0 / 2025-05-03
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getByUuid(int|string $uuid, bool $ignoreCache = false, bool $returnOrFail = true): User|null
    {
        static $cache = [];

        $cacheKey = md5(serialize(func_get_args()));

        if ($ignoreCache === false && isset($cache[$cacheKey])) {
            return $cache[$cacheKey];
        }
        $Query = User::where('uuid', '=', $uuid);

        if ($returnOrFail === true) {
            return $Query->first();
        }
        return $Query->firstOrFail();

    } // getByUuid()


} // class UserRepository {}
