<?php
/**
 * Package Service Provider Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Providers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals;


use BplanBase\Globals\Http\Middleware\CheckAccessLevelOrRoles;
use BplanBase\Globals\Http\Middleware\LogApiRequest;
use BplanBase\Globals\Http\Middleware\SetLocale;
use BplanBase\Globals\Services\HttpService;
use BplanBase\Globals\Support\LivewireComponentAutoDiscovery;
use Illuminate\Database\Events\QueryExecuted;
use Illuminate\Routing\Router;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Livewire\Livewire;


/**
 * Package Service Provider Class
 *
 * @version     4.2.0 / 2025-05-10
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class PackageServiceProvider extends ServiceProvider
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_basePath
     */
    protected $_basePath;


    /**
     *
     */
    protected $_baseNamespace = 'BplanModules\\Globals';


    /**
     * @var         string $_packageName
     */
    protected $_packageName = 'globals';


    /**
     * Indicates if loading of the provider is deferred.
     *
     * @var         bool $defer
     */
    protected $defer = false;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareSqlLogging(): void
    {
        if (Config::get('globals.db.sql-logging', false) === true) {
            /*
            **  Register log channel at runtime. */
            $this->_registerLogChannel();
            /*
            **  Log SQL queries. */
            DB::listen(function (QueryExecuted $Query) {
                Log::channel('sql')->info(
                    $Query->sql
                        ."\n".'  Bindings: '.((empty($Query->bindings)) ? '-' : print_r($Query->bindings, true))
                        ."\n".'  Runtime : '.$Query->time.'s'
                        ."\n"
                );
            });
        }
    } // _prepareSqlLogging()


    /**
     * Publish-Definitionen
     *
     * Aus der boot-Methode ausgelagert um den Code übersichtlich zu halten.
     *
     * @return 	    void
     *
     * @version     1.1.0 / 2025-05-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _publishes()
    {
        /*
        **  Dateien die bei jedem Setup neu gepublished werden. */
        $this->publishes([
            $this->_basePath.'/etc/setup/app/Http/Controllers/.gitignore'                                         => app_path('Http/Controllers/.gitignore'),
            $this->_basePath.'/etc/setup/database/migrations/0000_01_01_000000.globals.create_settings_table.php' => database_path('migrations/0000_01_01_000000.globals.create_settings_table.php'),
            $this->_basePath.'/etc/setup/database/migrations/0001_01_01_000000_create_users_table.php'            => database_path('migrations/0001_01_01_000000_create_users_table.php'),

        ], 'setup-forced');
        /*
        **  Dateien die nur einmalig gepublished werden sollen.
        **
        **  Alle Klassen zum Setting-Model würden eigentlich vom Code-Generator direkt
        **  im Projekt erzeugt werden. Da die zugehörige Logik aber individuell und
        **  bereits hier im Globals-Package definiert ist, werden im Projekt leere,
        **  abgeleitete und geschützte Klassen abgelegt. */
        $this->publishes([
            $this->_basePath.'/etc/setup/composer.local.json.in'                 => base_path('composer.local.json.in'),
            $this->_basePath.'/etc/setup/app/Models/Setting.php'                 => app_path('Models/Setting.php'),
            $this->_basePath.'/etc/setup/app/Observers/SettingObserver.php'      => app_path('Observers/SettingObserver.php'),
            $this->_basePath.'/etc/setup/app/Repositories/SettingRepository.php' => app_path('Repositories/SettingRepository.php'),
            $this->_basePath.'/etc/setup/app/Services/SettingService.php'        => app_path('Services/SettingService.php'),
            $this->_basePath.'/etc/setup/routes/web.php'                         => base_path('routes/web.php'),

        ], 'setup');

    } // _publishes()


    /**
     * Bootstrap the application events.
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-03-29
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerLogChannel()
    {
        /*
        **  Check whether the channel already exists to avoid duplicate entries. */
        if (!array_key_exists('sql', Config::get('logging.channels', []))) {
            Config::set('logging.channels.sql', [
                'driver' => 'daily',
                'path' => storage_path('logs/sql/sql.log'),
                'level' => 'info',
                'days' => 14,
                'replace_placeholders' => true,
            ]);
        }
    } // _registerLogChannel()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _registerMiddlewares(Router $Router)
    {
        $Router->aliasMiddleware('access-level.or.roles', CheckAccessLevelOrRoles::class);

    } // _registerMiddlewares()


    /**
     * Bootstrap the application events.
     *
     * @return      void
     *
     * @version     2.4.0 / 2025-05-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function boot(Router $Router): void
    {
        $this->_prepareSqlLogging();
        /*
        **  Inform Laravel about translations. */
        $this->loadTranslationsFrom($this->_basePath.'/lang', $this->_packageName);
        /*
        **  Inform Laravel about views. */
        $this->loadViewsFrom($this->_basePath.'/resources/views', $this->_packageName);

        if ($this->app->runningInConsole()) {
            /*
            **  Init publishes definitions. */
            $this->_publishes();
            /*
            **  Inform Laravel about migrations. */
            $this->loadMigrationsFrom($this->_basePath.'/database/migrations');
        }
        $this->_registerMiddleWares($Router);

        LivewireComponentAutoDiscovery::register($this->_baseNamespace.'\\Livewire', $this->_basePath.'/src/Livewire', $this->_packageName);
        /*
        **  Load package routes. */
        $this->loadRoutesFrom($this->_basePath.'/routes/api.php');
        $this->loadRoutesFrom($this->_basePath.'/routes/web.php');

    } // boot()


    /**
     * Register the service provider.
     *
     * @return      void
     *
     * @version     1.4.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function register(): void
    {
        $this->_basePath = str_replace('\\', '/', realpath(__DIR__.'/..'));
        /*
        **  Load the package configuration. */
        $this->mergeConfigFrom($this->_basePath.'/config/'.$this->_packageName.'.php', $this->_packageName);
        $this->mergeConfigFrom($this->_basePath.'/config/code-generator.php', 'code-generator');

        if ($this->app->runningInConsole()) {
            $this->commands([
                \BplanBase\Globals\Console\Commands\GlobalsSetup::class,
            ]);
        }
        $this->app->bind(HttpService::class, function() {
            return new HttpService();
        });
    } // register()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getApiMiddlewares(): array
    {
        return [
            LogApiRequest::class
        ];
    } // getApiMiddlewares()


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getWebMiddlewares(): array
    {
        return [
            SetLocale::class
        ];
    } // getWebMiddlewares()


} // class PackageServiceProvider extends ServiceProvider {}
