<?php
/**
 * Observer Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Observers
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Observers;


use App\Models\User;
use BplanBase\Globals\Notifications\UserCreated;
use BplanBase\Globals\Helpers\StringHelper;
use BplanBase\Globals\Services\RoleUserService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;


/**
 * Observer Class
 *
 * @version     1.1.0 / 2025-04-18
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class UserObserver
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string MODEL_CLASS
     */
    const MODEL_CLASS = User::class;


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Handle the Models "created" event
     *
     * When a new model is saved for the first time, the creating and created
     * events will dispatch.
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function created(User $Model): void
    {
        /*
        **  Notification an den neuen User senden. */
        $User->notify(new UserCreated($User->getRandomPassword()));

        $AuthUser = auth()->user();

        if ($AuthUser !== null) {
            Log::info('User '.$User->name.' '.$User->email.' created by '.$AuthUser->name.' ('.$AuthUser->email.')');
        }
    } // created()


    /**
     * Handle the Models "creating" event
     *
     * When a new model is saved for the first time, the creating and created
     * events will dispatch.
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function creating(User $Model): void
    {
        /*
        **  Automatische Werte erzeugen und setzen und Werte bereinigen. */
        $Model->uuid = Str::uuid();
        $Model->password = Hash::make($Model->getRandomPassword());
        $Model->email = StringHelper::normalizeEmail($Model->email);

    } // creating()


    /**
     * Handle the Models "deleted" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function deleted(User $Model): void
    {
        //
    } // deleted()


    /**
     * Handle the Models "deleting" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function deleting(User $Model): void
    {
        with(new RoleUserService())->deleteUserRoles($Model);

    } // deleting()


    /**
     * Handle the Models "forceDeleted" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function forceDeleted(User $Model): void
    {
        //
    } // forceDeleted()


    /**
     * Handle the Models "forceDeleting" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function forceDeleting(User $Model): void
    {
        //
    } // forceDeleting()


    /**
     * Handle the Models "replicating" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function replicating(User $Model): void
    {
        //
    } // replicating()


    /**
     * Handle the Models "restored" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function restored(User $Model): void
    {
        //
    } // restored()


    /**
     * Handle the Models "restoring" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function restoring(User $Model): void
    {
        //
    } // restoring()


    /**
     * Handle the Models "retrieved" event
     *
     * The retrieved event will dispatch when an existing model is retrieved
     * from the database.
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function retrieved(User $Model): void
    {
        //
    } // retrieved()


    /**
     * Handle the Models "saved" event
     *
     * The saving / saved events will dispatch when a model is created or
     * updated - even if the model's attributes have not been changed.
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function saved(User $Model): void
    {
        //
    } // saved()


    /**
     * Handle the Models "saving" event
     *
     * The saving / saved events will dispatch when a model is created or
     * updated - even if the model's attributes have not been changed.
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function saving(User $Model): void
    {
        //
    } // saving()


    /**
     * Handle the Models "trashed" event
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function trashed(User $Model): void
    {
        //
    } // trashed()


    /**
     * Handle the Models "updated" event
     *
     * The updating / updated events will dispatch when an existing model is
     * modified and the save method is called.
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function updated(User $Model): void
    {
        //
    } // updated()


    /**
     * Handle the Models "updating" event
     *
     * The updating / updated events will dispatch when an existing model is
     * modified and the save method is called.
     *
     * @param       User $Model
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-04-18
     * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function updating(User $Model): void
    {
        /*
        **  Werte bereinigen. */
        $Model->email = StringHelper::normalizeEmail($Model->email);

    } // updating()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class UserObserver {}
