<?php
/**
 * Setup Commands Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Commands
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Console\Commands;


use BplanBase\Globals\Helpers\StringHelper;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Symfony\Component\Process\Process;


/**
 * Setup Commands Class
 *
 * @version     3.0.0 / 2025-05-12
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class GlobalsSetup extends Command
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string STATUS_DONE
     */
    const STATUS_DONE    = 'DONE';


    /**
     * @var     string STATUS_SKIPPED
     */
    const STATUS_SKIPPED = 'SKIPPED';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_composerJson
     */
    private $_composerJson;


    /**
     * @var     string $description
     */
    protected $description = 'Installs the standard files from the Globals base package';


    /**
     * @var     string $signature
     */
    protected $signature = 'globals:setup';


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.1.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _copyFiles(): void
    {
        /*
        **  Paketdateien publishen.
        **
        **  Diese Dateien werden in jedem Fall kopiert und überschreiben
        **  gegebenenfalls auch bereits vorhandene Dateien. */
        Artisan::call('vendor:publish', [
            '--tag' => 'setup-forced',
            '--force' => true,
        ]);
        $lines = explode("\n", StringHelper::unifyLineBreaks(Artisan::output()));

        $this->info("\n".'   '.array_shift($lines)."\n");

        $this->_preparePublishOutput($lines);
        /*
        **  Diese Dateien werden nur kopiert wenn die Zieldateien noch nicht
        **  existieren. Nur beim ersten Lauf (wenn die Projektkonfiguration
        **  noch nicht existiert) wird der force-Parameter verwendet. */
        $params = [
            '--tag' => 'setup'
        ];
        if (file_exists(config_path('project.php')) === false) {
            $params['--force'] = true;
        }
        Artisan::call('vendor:publish', $params);

        $lines = explode("\n", StringHelper::unifyLineBreaks(Artisan::output()));

        $this->info("\n".'   '.array_shift($lines)."\n");

        $this->_preparePublishOutput($lines);

    } // _copyFiles()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(): void
    {
        $this->_composerJson = json_decode(file_get_contents(base_path('composer.json')), true);

    } // _init()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _linkStorage(): void
    {
        /*
        **  Storage-Link erzeugen. */
        $this->info("\n".'   INFO  Linking storage directory.'."\n");

        if (file_exists(public_path('storage'))) {
            $this->_output('Storage link already exists');

            return;
        }
        Artisan::call('storage:link');

        $this->_output(Artisan::output());

    } // _linkStorage()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _output(string $message, string $status = self::STATUS_DONE, $prependNewLine = false): void
    {
        if ($status === 'DONE') {
            $length = 109;
        }else {
            $length = 106;
        }
        $output = '';

        if ($prependNewLine === true) {
            $output = "\n";
        }
        $output .= str_pad('  • '.$message, $length, '.', STR_PAD_RIGHT).' '.$status;

        $this->info($output);

    } // _output()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _preparePublishOutput(array $lines): void
    {
        $basePath = str_replace('\\', '/', base_path());

        foreach ($lines as $line) {
            $line = trim($line);
            /*
            **  Fall 1: "Copied file [...] to [...]". */
            if (str_starts_with($line, 'Copying')) {
                preg_match_all('/\[(.*?)\]/', $line, $matches);

                $targetPath = $matches[1][1] ?? null;

                if ($targetPath !== null) {
                    $targetPath = str_replace('\\', '/', $targetPath);
                    $targetPath = str_replace($basePath, '[Laravel]', $targetPath);

                    $output = 'Copied '.$targetPath;
                    $status = self::STATUS_DONE;
                }
            } elseif (str_contains($line, 'already exists')) {
                /*
                **  Fall 2: "File [...] already exists". */
                preg_match('/File\s+\[(.*?)\]/', $line, $match);

                $targetPath = $match[1] ?? null;

                if ($targetPath !== null) {
                    $targetPath = str_replace('\\', '/', $targetPath);
                    $targetPath = str_replace($basePath, '[Laravel]', $targetPath);

                    $output = 'Exists '.$targetPath;
                    $status = self::STATUS_SKIPPED;
                }
            }
            $this->_output($output, $status);
        }
    } // _preparePublishOutput()


    /**
     *
     * @param       array $command
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _runShellCommand(array $command): void
    {
        $process = new Process($command);

        $process->run();

        if (!$process->isSuccessful()) {
            $this->error('  Command failed: '.implode(' ', $command));
            $this->error($process->getErrorOutput());

        } else {
            $this->info($process->getOutput());
        }
    } // _runShellCommand()


    /**
     *
     * @return 	    void
     *
     * @version     1.1.0 / 2025-05-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _setupMergePlugin(): void
    {
        $this->info("\n".'   INFO  Update composer.json with merge-plugin.'."\n");

        if (isset($this->_composerJson['require-dev']['wikimedia/composer-merge-plugin'])) {
            $this->_output('Composer merge plugin already exists', self::STATUS_SKIPPED);

            return;
        }
        $this->info('  • Configure "allow-plugins".');

        $this->_runShellCommand([
            'composer',
            'config',
            'allow-plugins.wikimedia/composer-merge-plugin',
            'true',
        ]);
        $this->info('  • Configure "extra.merge-plugin.include".');

        $this->_runShellCommand([
            'composer',
            'config',
            'extra.merge-plugin.include',
            '--json',
            '["composer.local.json"]',
        ]);
        $this->info('  • Configure "extra.merge-plugin.merge-dev".');

        $this->_runShellCommand([
            'composer',
            'config',
            'extra.merge-plugin.merge-dev',
            'true',
        ]);
        $this->info('  • Install package "wikimedia/composer-merge-plugin".');

        $this->_runShellCommand([
            'composer',
            'require',
            'wikimedia/composer-merge-plugin',
            '--dev',
        ]);
        $this->_output('Package installation');

    } // _setupMergePlugin()


    /**
     *
     * @return      void
     *
     * @version     1.1.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function handle()
    {
        $this->_init();

        $this->_copyFiles();
        $this->_linkStorage();
        $this->_setupMergePlugin();

        $this->info("\n".'  Setup completed ✅');

    } // handle()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class GlobalsSetup extends Command {}