<?php
/**
 * Registry Class
 *
 * @version     2.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Registries
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 Wassilios Meletiadis <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Registries;


/**
 * Registry Class
 *
 * @version     1.0.0 / 2025-05-11
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Registry
{

/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     */
    private static $_data = [];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Leert die Registry und gibt den vollständigen, ursprünglichen Inhalt zurück
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function flush(): array
    {
        $return = self::$_data;

        self::$_data = [];

        return $return;

    } // flush()


    /**
     * Gibt einen oder mehrere Werte aus der Registry zurück
     *
     * @param       array|float|int|null|string $keys
     *              NULL, ein einzelner Schlüssel oder ein Array mit den Schlüsseln, deren Werte
     *              zurückgegeben werden sollen.
     *              Durch Übergabe von NULL kann der vollständige Inhalt der Registry zurückgegeben
     *              werden. Mit der Übergabe eines Strings wird nur der Wert des entsprechenden
     *              Schlüssels geliefert und bei Übergabe eines Arrays, wird ein Array mit diesen
     *              Schlüsseln und den zugehörigen Werten ausgeliefert.
     *              Für alle Schlüssel, die nicht in der Registry existieren, ist der Wert NULL.
     *
     * @param       bool|float|int|null|string $returnValue
     *              Ein alternativer Wert, der zurückgegeben werden soll, wenn der oder die Schlüssel
     *              nicht in der Registry existieren.
     *
     * @param       bool $unset
     *              Wenn dieser Parameter auf TRUE gesetzt wird, dann werden die übergebenen Schlüssel
     *              und aus der Registry entfernt.
     *
     * @return      mixed
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function get(array|float|int|null|string $keys = null, bool|float|int|null|string $returnValue = null, bool $unset = false): mixed
    {
        if ($keys === null) {
            if ($unset === true) {
                return self::flush();
            }
            return self::$_data;
        }
        $return = null;

        if (!is_array($keys)) {
            if (!isset(self::$_data[$keys])) {
                return $returnValue;
            }
            $return = self::$_data[$keys];

            if ($unset === true) {
                unset(self::$_data[$keys]);
            }
            return $return;
        }
        $return = [];

        foreach ($keys as $key) {
            if (!isset(self::$_data[$key])) {
                $return[$key] = $returnValue;

                continue;
            }
            $return[$key] = self::$_data[$key];

            if ($unset === true) {
                unset(self::$_data[$key]);
            }
        }
        return $return;

    } // get()


    /**
     * Speichert einen oder mehrere Werte in der Registry
     *
     * @param       array|float|int|string $keys
     *              Ein einzelner, skalarer Wert als Schlüssel, ein Array von Schlüsseln oder ein
     *              Array mit Schlüssel-/Wert-Paaren.
     *              Wenn hier ein Array übergeben wird und der Wert für $values NULL ist, dann werden
     *              die Werte des Arrays mit ihren Schlüsseln in der Registry gespeichert. Wird für
     *              $values jedoch auch ein Array übergeben, dann muss die Anzahl der Werte in beiden
     *              Array-Parametern identisch sein.
     *
     * @param       mixed $values
     *              Ein einzelner Wert oder ein Array von Werten.
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function set(array|float|int|string $keys, mixed $values = null): void
    {
        if (!is_array($keys)) {
            $input = [
                $keys => $values
            ];
        } else {
            if ($values === null) {
                $input = $keys;
            } else {
                $input = array_combine($keys, $values);
            }
        }
        foreach ($input as $key => $value) {
            self::$_data[$key] = $value;
        }
    } // set()


    /**
     * Löscht einen oder mehrere Schlüssel mit den zugehörigen Werten aus der Registry
     *
     * @param       array|float|int|string $keys
     *              Ein einzelner Schlüssel oder ein Array von Schlüsseln.
     *
     * @version     1.0.0 / 2025-05-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function unset(array|float|int|string $keys): void
    {
        if (!is_array($keys)) {
            $keys = (array) $keys;
        }
        foreach ($keys as $key) {
            unset(self::$_data[$key]);
        }
    } // unset()


} // class Registry {}
