<?php
/**
 * Query Helper Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Helpers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Helpers;


use Illuminate\Database\Eloquent\Builder as EloquentBuilder;
use Illuminate\Database\Query\Builder as QueryBuilder;


/**
 * Query Helper Class
 *
 * @version     2.0.0 / 2025-07-25
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class QueryHelper
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       EloquentBuilder|QueryBuilder $FromQuery
     *
     * @param       EloquentBuilder $ToQuery
     *
     * @return      EloquentBuilder
     *
     * @version     1.0.0 / 2025-07-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function applyConditionsFromBuilder(EloquentBuilder|QueryBuilder $FromQuery, EloquentBuilder & $ToQuery): EloquentBuilder
    {
        $wheres = $FromQuery instanceof EloquentBuilder
            ? $FromQuery->getQuery()->wheres
            : $FromQuery->wheres;

        foreach ($wheres ?? [] as $condition) {
            switch ($condition['type']) {
                case 'Basic':
                    $method = $condition['boolean'] === 'or' ? 'orWhere' : 'where';

                    $ToQuery->{$method}($condition['column'], $condition['operator'], $condition['value']);
                    break;

                case 'In':
                    $method = $condition['boolean'] === 'or' ? 'orWhereIn' : 'whereIn';

                    $ToQuery->{$method}($condition['column'], $condition['values']);
                    break;

                case 'Nested':
                    $method = $condition['boolean'] === 'or' ? 'orWhere' : 'where';

                    $ToQuery->{$method}(function ($NestedQuery) use ($condition) {
                        /*
                        **  Rekursiv auf das verschachtelte Query-Objekt anwenden. */
                        self::applyConditionsFromBuilder($condition['query'], $NestedQuery);
                    });
                    break;

                case 'Null':
                    $method = $condition['boolean'] === 'or' ? 'orWhereNull' : 'whereNull';

                    $ToQuery->{$method}($condition['column']);
                    break;

                // Weitere Typen wie 'NotNull', 'Between', etc. könnten hier ergänzt werden
            }
        }
        return $ToQuery;

    } // applyConditionsFromBuilder()


    /**
     * Applies flexible orderBy instructions to the query.
     *
     * @param       EloquentBuilder $Query
     *              The Eloquent query builder instance.
     *
     * @param       array|string|null $orderBy
     *              The ordering instructions.
     *              The $orderBy parameter can take several formats:
     *
     *              1. A single column name as a string:
     *                  'name'
     *
     *              2. An array of column names as strings (default direction: asc):
     *                  ['name', 'created_at']
     *
     *              3. An array of arrays with 'column' and optional 'direction':
     *                  [
     *                      ['column' => 'name'],
     *                      ['column' => 'created_at', 'direction' => 'desc']
     *                  ]
     *
     *              Mixed format is also supported:
     *                  [
     *                      'priority',
     *                      ['column' => 'created_at', 'direction' => 'desc']
     *                  ]
     *
     * @return      Builder
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function applyOrderBy(EloquentBuilder $Query, array|string|null $orderBy): EloquentBuilder
    {
        if ($orderBy === null) {
            return $Query;
        }
        if (is_string($orderBy)) {
            return $Query->orderBy($orderBy);

        } elseif (is_array($orderBy)) {
            foreach ($orderBy as $item) {
                if (is_string($item)) {
                    $Query->orderBy($item);

                } elseif (is_array($item)) {
                    $column = $item['column'] ?? null;
                    $direction = strtolower($item['direction'] ?? 'asc');

                    if ($column !== null) {
                        $Query->orderBy($column, $direction);
                    }
                }
            }
        }
        return $Query;

    } // applyOrderBy()


} // class QueryHelper {}
