<?php
/**
 * AppLog Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Logging
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Logging;


use Illuminate\Support\Facades\Log;
use Stringable;
use Throwable;


/**
 * AppLog Class
 *
 * @version     1.3.0 / 2025-09-01
 * @author      Emilio Cannarozzo <emilio.cannarozzo@bplan-solutions.de>
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class AppLog
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @return      string|Stringable
     *
     * @version     3.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _prefix(string|Stringable|Throwable $message, string|null $prefix, null|string $exceptionMessageIntro): string|Stringable
    {
        if ($message instanceof Throwable) {
            $message = $message->getMessage().' in file '.$message->getFile().' on line '.$message->getLine();

            if ($exceptionMessageIntro !== null) {
                $message = $exceptionMessageIntro.': '.$message;
            }
        }
        if (empty($prefix)) {
            return '|> '.$message;
        }
        return '['.$prefix.']> '.$message;

    } // _prefix()


    /**
     *
     * @param       array|string $context
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-08-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _prepareContext(array|string $context): array
    {
        if (is_array($context)) {
            return $context;
        }
        return (array) trim($context);

    } // _prepareContext()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function alert(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog-composite')->alert(str_repeat(' ', 4).self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // alert()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function critical(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog-composite')->critical(str_repeat(' ', 1).self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // critical()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function debug(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog')->debug(str_repeat(' ', 4).self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // debug()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function emergency(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog-composite')->emergency(self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // emergency()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function error(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog-composite')->error(str_repeat(' ', 4).self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // error()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function info(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog')->info(str_repeat(' ', 5).self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // info()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function notice(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog')->notice(str_repeat(' ', 3).self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // notice()


    /**
     *
     * @param       string|Stringable|Throwable $message
     *
     * @param       string|null $prefix
     *
     * @param       array|string $context
     *
     * @return      string
     *
     * @version     4.0.0 / 2025-09-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function warning(string|Stringable|Throwable $message, string|null $prefix = null, array|string $context = [], null|string $exceptionMessageIntro = null): string
    {
        Log::channel('applog-composite')->warning(str_repeat(' ', 2).self::_prefix($message, $prefix, $exceptionMessageIntro), self::_prepareContext($context), $exceptionMessageIntro);

        return $message;

    } // warning()


} // class AppLog {}
