<?php
/**
 * Funktionalität zur Anzeige von Notifications in Livewire-Komponenten
 *
 * @todo        - Der Javascript-Teil (die Funktion showNotification()), der aktuell noch in der Datei
 *                "resources/js/project.js" platziert ist, muss noch irgendwie ins Paket aufgenommen
 *                werden.
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/globals
 * @subpackage  Traits
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\Globals\Traits\Core;


use BplanBase\Globals\Livewire\Core\Global\Notification;


/**
 * Funktionalität zur Anzeige von Notifications in Livewire-Komponenten
 *
 * Dieser Trait kann via "use" in einer Livewire-Komponente eingebunden werden um seine Funktionalität
 * dort verfügbar zu machen:
 *
 *  <pre>
 *    <code>
 *      use App\bplan\Livewire\NotificationTrait;
 *&nbsp;
 *      class UserForm extends Component
 *      {
 *          use NotificationTrait;
 *          ...
 *      }
 *    </code>
 *  </pre>
 *
 * Ob und wie der Trait eventuell auch in anderen Klassen genutzt werden kann, bei denen es sich
 * nicht um Livewire-Komponenten handelt, wurde noch nicht getestet.
 *
 * Alternativ zur direkten Verwendung dieses Traits, kann die zu erweiternde Component aber auch
 * von der Klasse "FormComponentBase" (@see App\bplan\Livewire\FormComponentBase) abgeleitet werden
 * (nur für Formular-Klassen sinnvoll). Diese Klasse hat die Verwendung dieses Traits bereits
 * implementiert.
 *
 *
 *  <strong>Anwendung</strong>
 *
 * Zur Verwendung der Notifications muss das Notification-Template im Livewire-Stil eingebunden
 * werden. Die Einbindung kann, im Layout- oder im Formular-Template, mit folgender Codezeile
 * erfolgen:
 *
 *  <pre>
 *    <code>
 *      &lt;livewire:bplan.notification /&gt;
 *    </code>
 *  </pre>
 * Anschließend können die Methoden _hideNotification() und _showNotification(), die im in diesem
 * Trait definiert sind, in der abgeleiteten Component-Klasse verwendet werden.
 *
 * @version     1.0.0 / 2025-05-25
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
trait NotificationTrait
{


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Der Name der Methode, die zur Anzeige der Notification aufgerufen werden soll
     *
     * Es stehen zwei mögliche Werte zur Verfügung. "showNotification" initiiert die Anzeige der
     * Notification aus PHP heraus. Mit "showNotificationJS" wird Javascript zur Anzeige verwendet.
     *
     * @var string $_notification_method
     */
    private $_notification_method = 'showNotification';


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Variable zur Verwendung in der Component-Klasse
     *
     * Diese Variable kann in der Komponentenklasse direkt verwendet werden und erspart es so,
     * dass die Component selbst nochmal die Notification-Klasse per "use" einbinden muss.
     *
     *  @var    string $_notification_type_error
     */
    protected static $_notification_type_error = Notification::TYPE__ERROR;


    /**
     * Variable zur Verwendung in der Component-Klasse
     *
     * Diese Variable kann in der Komponentenklasse direkt verwendet werden und erspart es so,
     * dass die Component selbst nochmal die Notification-Klasse per "use" einbinden muss.
     *
     *  @var    string $_notification_type_notice
     */
    protected static $_notification_type_notice = Notification::TYPE__NOTICE;


    /**
     * Variable zur Verwendung in der Component-Klasse
     *
     * Diese Variable kann in der Komponentenklasse direkt verwendet werden und erspart es so,
     * dass die Component selbst nochmal die Notification-Klasse per "use" einbinden muss.
     *
     *  @var    string $_notification_type_success
     */
    protected static $_notification_type_success = Notification::TYPE__SUCCESS;


    /**
     * Variable zur Verwendung in der Component-Klasse
     *
     * Diese Variable kann in der Komponentenklasse direkt verwendet werden und erspart es so,
     * dass die Component selbst nochmal die Notification-Klasse per "use" einbinden muss.
     *
     *  @var    string $_notification_type_warning
     */
    protected static $_notification_type_warning = Notification::TYPE__WARNING;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Blendet die Notification aus
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _hideNotification()
    {
        $this->dispatch('hideNotification');

    } // _hideNotification()


    /**
     * Zeigt eine Notification an
     *
     * @param       string|array    $messages
     *              <br>Die Nachricht(en), die in der Notification angezeigt werden sollen.
     *
     * @param       string  $type
     *              <br>Die Art der Nachricht.
     *              Hierbei handelt es sich um so eine Art Error-Level. In Abhängigkeit vom Wert
     *              dieses Parameters ändert sich die Darstellung der Notification.
     *
     * @return      object  Gibt die Objektinstanz zurück.
     *
     * @todo        - Macht es eventuell Sinn im Normalfall nichts und im Fehlerfall NULL zurückzugeben?
     *
     * @version     1.0.0 / 2025-05-25
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _showNotification($messages, $type = Notification::TYPE__ERROR)
    {
        if (empty($messages)) {
            return $this;
        }
        $this->dispatch($this->_notification_method, type: $type, messages: $messages);

        return $this;

    } // _showNotification()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // trait NotificationTrait {}
