<?php
/**
 * Code Generator Definition Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-code-generator
 * @subpackage  Definitions
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace Bplan\LaravelCodeGenerator\Definitions;


use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Schema\ForeignKeyDefinition as SchemaForeignKeyDefinition;
use Illuminate\Support\Fluent;
use Illuminate\Support\Str;


use Bplan\LaravelCodeGenerator\Definitions\ForeignKeyDefinition;
use Bplan\LaravelCodeGenerator\Enums\ModelType;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator;


/**
 * Code Generator Definition Class
 *
 * @version     2.0.0 / 2024-10-26
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class TypeDefinition
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_additionalRelations
     */
    private $_additionalRelations;


    /**
     * @var     array $_columns
     */
    private $_columns = [];


    /**
     * @var     array $_fieldDefinitions
     */
    private $_fieldDefinitions = [];


    /**
     * @var     array $_foreignKeys
     */
    private $_foreignKeys = [];


    /**
     *
     * @var     string $_modelType
     */
    private $_modelType = ModelType::Model->name;


    /**
     * @var     array $_pivotTables
     */
    private $_pivotTables = [];


    /**
     * @var     array $_routing
     */
    private $_routing;


    /**
     *
     * @var     string $_tableName
     */
    private $_tableName;


    /**
     *
     * @var     string $_tablePrefix
     */
    private $_tablePrefix;


    /**
     * @var     array $_traits
     */
    private $_traits = [];


    /**
     * @var     array $_uses
     */
    private $_uses = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_fieldDefintions
     */
    private static $_fieldDefintions = [];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return      void
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(Blueprint $Blueprint, array $additionalSettings = [])
    {
        $this->_process($Blueprint, $additionalSettings);

        // $this->_writeCache();

    } // __construct()


    /**
     *
     * @param       array|null $additionalColumnSettings
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _createFieldDefinitions(array|null $additionalColumnSettings): self
    {
        self::$_fieldDefintions[$this->_tableName] = [];

        foreach ($this->_columns as $columnName => $columnDefinition) {
            $FieldDefinition = new FieldDefinition($columnDefinition, $additionalColumnSettings[$columnName] ?? null);

            $this->_fieldDefinitions[$columnName] = $FieldDefinition;
            self::$_fieldDefintions[$this->_tableName][$columnName] = $FieldDefinition;
        }
        return $this;

    } // _createFieldDefinitions()


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return      $this
     *
     * @version     1.1.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _process(Blueprint $Blueprint, array $additionalSettings = []): self
    {
        $this->_tableName = $Blueprint->getTable();
        $this->_tablePrefix = $Blueprint->getPrefix();

        if (isset($additionalSettings['pivotTables'])) {
            $this->_pivotTables = array_combine($additionalSettings['pivotTables'], array_reverse($additionalSettings['pivotTables']));

            $this->_modelType = ModelType::Pivot->name;
        }
        $this->_additionalRelations = $additionalSettings['additionalRelations'] ?? [];
        $this->_routing = $additionalSettings['routing'] ?? null;
        $this->_traits = $additionalSettings['traits'] ?? null;
        $this->_uses = $additionalSettings['uses'] ?? null;

        $this->_processColumns($Blueprint->getColumns());
        /*
        **  Commands abarbeiten und FeldDefinitionen ergänzen. */
        $this->_processCommands($Blueprint->getCommands(), $additionalSettings['columns'] ?? null);
        /*
        **  FieldDefinition-Objekte erzeugen. */
        $this->_createFieldDefinitions($additionalSettings['columns'] ?? null);

        return $this;

    } // _process()


    /**
     * Verarbeitet alle Spalten und erstellt einen assoziativen Array mit den Spaltennamen als Schlüssel
     *
     * @param       array $columns
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processColumns(array $columns): self
    {
        foreach ($columns as $Column) {
            $column = $Column->toArray();
            $columnName = $column['name'];

            $column['isForeignKey'] = false;

            ksort($column);

            $this->_columns[$columnName] = $column;
        }
        return $this;

    } // _processColumns()


    /**
     * Verarbeitet Commands und ergänzt die Column-Definitionen in $_fieldDefinitions
     *
     * In den Commands des Blueprint-Objekts sind unter anderem die ForeignKey-Definitionen enthalten.
     *
     * @param       array $commands
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processCommands(array $commands, array|null $additionalColumnSetting): self
    {
        foreach ($commands as $Command) {
            $attributes = $Command->getAttributes();

            if ($Command instanceof SchemaForeignKeyDefinition) {
                $columnName = $attributes['columns'][0];

                $attributes['relationType'] = 'BelongsTo';

                ksort($attributes);

                $this->_columns[$columnName]['isForeignKey'] = true;
                $this->_columns[$columnName]['ForeignKeyDefinition'] = new ForeignKeyDefinition($attributes);
                $this->_columns[$columnName]['reverseRelationName'] = null;

                $this->_foreignKeys[$columnName] = $this->_columns[$columnName];

                if (isset($additionalColumnSetting[$columnName]['reverseRelationName'])) {
                    $this->_foreignKeys[$columnName]['reverseRelationName'] = $additionalColumnSetting[$columnName]['reverseRelationName'];
                }
                ksort($this->_foreignKeys[$columnName]);

                continue;

            } elseif ($Command instanceof Fluent) {
                if ($attributes['name'] === 'create') {
                    continue;
                }
                /*
                    Index-Informationen werden aktuell noch nicht berücksichtigt, könnten aber insbesondere
                    für die Rules relevant werden um zum Beispiel eine unique-Rule zu erzeugen.

                */
                //self::$_collectedIndexes[$this->_tableName][] = $attributes;
            }
        }
        return $this;

    } // _processCommands()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getAdditionalRelations(): array
    {
        return $this->_additionalRelations;

    } // getAdditionalRelations()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFieldDefinitions(string $tableName): array
    {
        return self::$_fieldDefintions[$tableName];

    } // getFieldDefinitions()


    /**
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getForeignKey($columnName): array|null
    {
        if (!isset($this->_foreignKeys[$columnName])) {
            return null;
        }
        return $this->_foreignKeys[$columnName];

    } // getForeignKey()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getForeignKeys(): array
    {
        return $this->_foreignKeys;

    } // getForeignKeys()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPivotTables(): array
    {
        return $this->_pivotTables;

    } // getPivotTables()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getModelType(): string
    {
        return $this->_modelType;

    } // getModelType()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRouting()
    {
        return $this->_routing;

    } // getRouting()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getTableName(): string
    {
        return $this->_tableName;

    } // getTableName()


    /**
     *
     * @param       string $traitsFor
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getTraits(string $traitsFor): array|null
    {
        if (empty($this->_traits[$traitsFor])) {
            return null;
        }
        return $this->_traits[$traitsFor];

    } // getTraits()


    /**
     *
     * @param       string $caseStyle = null
     *
     * @param       bool|null $singular = null
     *
     * @return 	    string
     *
     * @version     2.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getTypeName(string $caseStyle = null, bool|null $singular = null): string
    {
        $tableName = $this->_tableName;

        if ($singular === true) {
            $tableName = Str::singular($this->_tableName);

        } elseif ($singular === false) {
            $tableName = Str::plural($this->_tableName);
        }
        switch ($caseStyle) {
            case 'slug':
                return Str::slug($tableName);
                break;

            default:
                if ($singular === true) {
                    return (string) Str::of($this->_tableName)->studly()->singular();
                }
                return (string) Str::of($this->_tableName)->studly()->plural();
        }
    } // getTypeName()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getUses(string $usesFor): array|null
    {
        if (empty($this->_uses[$usesFor])) {
            return null;
        }
        return $this->_uses[$usesFor];

    } // getUses()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function writeCacheFile(): self
    {
        $cachePath = CodeGenerator::getCachePath();

        ksort($this->_fieldDefinitions);

        unset($this->_columns);

        file_put_contents($cachePath.'/'.$this->_tableName.'.php', '<?php'."\n\n".'return '.var_export($this, true).';'."\n");

        return $this;

    } // writeCacheFile()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class TypeDefinition {}
