<?php
/**
 * Code Generator Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace Bplan\LaravelCodeGenerator\Generators;


use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Schema\ForeignIdColumnDefinition;
use Illuminate\Database\Schema\ForeignKeyDefinition;
use Illuminate\Support\Str;
use Symfony\Component\Console\Output\ConsoleOutput;


use Bplan\LaravelCodeGenerator\Definitions\BackReference;
use Bplan\LaravelCodeGenerator\Definitions\TypeDefinition;
use Bplan\LaravelCodeGenerator\Enums\ModelType;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\BaseGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\CSharpFileGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\JsonApiFileGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\JsonApiGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\LaravelFileGenerator;


/**
 * Code Generator Class
 *
 * @version     5.1.0 / 2024-10-26
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class CodeGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string CODE_GENERATOR_TAG
     */
    const CODE_GENERATOR_TAG = '#CodeGenerator';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_additionalDefinitionOptions
     */
    protected $_additionalDefinitionOptions = [
        'modelType' => ModelType::Model->name,
    ];


    /**
     * @var     string $_modelType
     */
    protected $_modelType;


    /**
     * @var     string $_tableName
     */
    protected $_tableName;


    /**
     * @var     array $_typeDefinition
     */
    protected $_typeDefinition;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_backReferences
     */
    private static $_backReferences = [];


    /**
     * @var     string $_cachePath
     */
    private static $_cachePath;


    /**
     * @var     array $_collectedForeignKeys
     */
    private static $_collectedForeignKeys = [];


    /**
     * @var     array $_collectedIndexes
     */
    private static $_collectedIndexes = [];


    /**
     * @var     array $_collectedTypeDefinitions
     */
    private static $_collectedTypeDefinitions = [];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(Blueprint $Blueprint, array $additionalSettings = [])
    {
        if (config('code-generator.active') !== true) {
            return;
        }
        $this->_init($Blueprint, $additionalSettings);

    } // __construct()


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return 	    $this
     *
     * @version     1.4.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _init(Blueprint $Blueprint, array $additionalSettings = []): self
    {
        $TypeDefinition = new TypeDefinition($Blueprint, $additionalSettings);

        $this->_tableName = $Blueprint->getTable();

        if (isset($additionalSettings['pivotTables'])) {
            $this->_modelType = ModelType::Pivot->name;
        } else {
            $this->_modelType = ModelType::Model->name;
        }
        $this->_processCommands($Blueprint->getCommands());

        self::$_collectedTypeDefinitions[$this->_tableName] = $TypeDefinition;

        $TypeDefinition->writeCacheFile();

        return $this;

    } // _init()


    /**
     * Ergänzt die übergebenen ColumnDefintions um zusätzliche Spalteneinstellungen
     *
     * @param       array $columnDefinitions
     *
     * @param   	array|null $additionalSettings
     *
     * @return 	    array
     *
     * @version     1.2.0 / 2024-10-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _addAdditionalColumnSettings(array $columnDefinitions, array|null $additionalSettings): array
    {
        $name = $columnDefinitions['name'];

        if ($additionalSettings === null) {
            $columnDefinitions = array_merge($columnDefinitions, $this->_additionalColumnSettingOptions);

            ksort($columnDefinitions);

            return $columnDefinitions;
        }
        foreach ($this->_additionalColumnSettingOptions as $option => $default) {
            if (isset($additionalSettings[$option])) {
                $columnDefinitions[$option] = $additionalSettings[$option];

                if ($columnDefinitions['isForeignId'] === true && $option === 'reverseRelationName') {
                    self::$_collectedForeignKeys[$this->_table][$name]['reverseRelationName'] = $additionalSettings[$option];
                }
            } else {
                /*
                **  Sonderbehandlung: In den Defaults ist der angegebene Wert NULL. Wenn die Optionen nicht
                **  in den $additionalSettings enthalten sind, dann wird FALSE eingesetzt. */
                if ($option === 'sortable' || $option === 'readonly') {
                    $default = false;
                }
                $columnDefinitions[$option] = $default;
            }
        }
        ksort($columnDefinitions);

        return $columnDefinitions;

    } // _addAdditionalColumnSettings()


    /**
     * Verarbeitet Commands und speichert die ermittelten Informationen in statischen Variablen
     *
     * In den Commands des Blueprint-Objekts sind unter anderem die ForeignKey-Definitionen enthalten.
     *
     * @param       array $commands
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _processCommands(array $commands): self
    {
        if (!isset(self::$_collectedForeignKeys[$this->_tableName])) {
            self::$_collectedForeignKeys[$this->_tableName] = [];
            // self::$_collectedIndexes[$this->_tableName] = [];
        }
        foreach ($commands as $Command) {
            $attributes = $Command->getAttributes();

            if ($Command instanceof ForeignKeyDefinition) {
                $column = $attributes['columns'][0];
                $relatedTable = $attributes['on'];

                // $strippedColumn = Str::replaceLast('_id', '', $column);

                // unset($attributes['algorithm']);
                // unset($attributes['columns']);
                // unset($attributes['name']);
                // unset($attributes['on']);

                // $attributes['column'] = $column;
                // $attributes['relatedTable'] = $relatedTable;
                // $attributes['relationName'] = Str::slug($strippedColumn, '-');
                // $attributes['relationNameStudlyCaps'] = Str::studly($strippedColumn);
                // $attributes['relationType'] = 'BelongsTo';
                // $attributes['resourceName'] = Str::slug($relatedTable, '-');
                // $attributes['resourceNameCamelCase'] = Str::camel($relatedTable, '-');
                // $attributes['resourceNameStudlyCaps'] = Str::studly($relatedTable);

                // ksort($attributes);

                self::$_collectedForeignKeys[$this->_tableName][$column] = $attributes;

                if (!isset(self::$_backReferences[$relatedTable])) {
                    self::$_backReferences[$relatedTable] = [];
                }
                self::$_backReferences[$relatedTable][] = new BackReference($this->_modelType, $this->_tableName, $column);

                continue;
            }
        }
        return $this;

    } // _processCommands()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.7.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function generate(): void
    {
        $creates = config('code-generator.creates');

        self::output("\n\n".'<info>  CodeGenerator</info> Starting code generation - TestMode '.(config('code-generator.test-mode') == true ? '<info>On</info>' : '<error>Off</error>'), true, 2);

        ksort(self::$_collectedTypeDefinitions);

        foreach (self::$_collectedTypeDefinitions as $tableName => $TypeDefinition) {
            $typeName = $TypeDefinition->getTypeName(singular: true);

            self::output("\n\n".'  Starting code generation for resource <options=bold,underscore>'.$typeName.'</>', true, 2);

            foreach ($creates as $create => $status) {
                if ($status === false) {
                    continue;
                }
                switch ($create) {
                    case 'CSharp';
                        CSharpFileGenerator::process($TypeDefinition);
                        break;

                    case 'JsonApi';
                        JsonApiFileGenerator::process($TypeDefinition);
                        break;

                    case 'Laravel';
                        LaravelFileGenerator::process($TypeDefinition);
                        break;
                }
            }
        }
        if ($creates['JsonApi'] === true) {
            self::output("\n\n".'  Starting global code generation for <options=bold,underscore>JsonApi</>', true, 2);

            JsonApiGenerator::process(self::$_collectedTypeDefinitions, self::$_backReferences);
        }
        $resultCounters = BaseGenerator::getResultCounters();

        $total = array_sum($resultCounters);
        $totalLength = strlen($total);
        $pad = 45;

        self::output("\n"
            ."\n".'  <options=bold,underscore>Results            '.str_repeat(' ', $pad).str_repeat(' ', $totalLength).'</>'
            ."\n".'  <comment>Created Files   '.str_repeat(' ', $pad + 3).str_pad($resultCounters['created'], $totalLength, ' ', STR_PAD_LEFT).'</comment>'
            ."\n".'  <comment>Replaced Files  '.str_repeat(' ', $pad + 3).str_pad($resultCounters['replaced'], $totalLength, ' ', STR_PAD_LEFT).'</comment>'
            ."\n".'  <comment>Protected Files '.str_repeat(' ', $pad + 3).str_pad($resultCounters['protected'], $totalLength, ' ', STR_PAD_LEFT).'</comment>'
            ."\n".'  <options=underscore>Unchanged Files    '.str_repeat(' ', $pad).str_pad($resultCounters['unchanged'], $totalLength, ' ', STR_PAD_LEFT).'</>'
            ."\n".'  <options=bold,underscore>Total              '.str_repeat(' ', $pad).str_pad($total, $totalLength, ' ', STR_PAD_LEFT).'</>'
        );
        self::output("\n".'<fg=green;options=bold>  Done</>', true, 2);

    } // generate()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBackReferences(string $tableName): array|null
    {
        if (!isset(self::$_backReferences[$tableName])) {
            return null;
        }
        return self::$_backReferences[$tableName];

    } // getBackReferences()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     1.1.0 / 2024-10-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBackupPath(null|string $path = null): string
    {
        $backupPath = str_replace('\\', '/', storage_path('app/backup/'.basename(__CLASS__)));

        if ($path === null) {
            return $backupPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $backupPath.$path;
        }
        return $backupPath.'/'.$path;

    } // getBackupPath()


    /**
     *
     * @param       null|string $path
     *
     * @return       string
     *
     * @version      1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBasePath(null|string $path = null): string
    {
        $basePath = str_replace('\\', '/', base_path());

        if ($path === null) {
            return $basePath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $basePath.$path;
        }
        return $basePath.'/'.$path;

    } // getBasePath()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBuildPath(null|string $path = null): string
    {
        $buildPath = str_replace('\\', '/', storage_path('app/build/'.basename(__CLASS__)));

        if ($path === null) {
            return $buildPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $buildPath.$path;
        }
        return $buildPath.'/'.$path;

    } // getBuildPath()


    /**
     *
     * @return       string
     *
     * @version      1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getCachePath(): string
    {
        if (self::$_cachePath !== null) {
            return self::$_cachePath;
        }
        self::$_cachePath = str_replace('\\', '/', storage_path('app/cache/'.basename(__CLASS__)));

        if (!file_exists(self::$_cachePath)) {
            mkdir(self::$_cachePath, 0777, true);
        }
        return self::$_cachePath;

    } // getCachePath()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getStubPath(null|string $path = null): string
    {
        $stubPath = str_replace('\\', '/', realpath(__DIR__.'/../..'))
            .'/stubs/code-generator';

        if ($path === null) {
            return $stubPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $stubPath.$path;
        }
        return $stubPath.'/'.$path;

    } // getStubPath()


    /**
     *
     * @param       string $table
     *
     * @return      array|null
     *
     * @version     2.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getTypeDefinition(string $tableName): null|TypeDefinition
    {
        if (!isset(self::$_collectedTypeDefinitions[$tableName])) {
            return null;
        }
        return self::$_collectedTypeDefinitions[$tableName];

    } // getTypeDefinition()


    /**
     *
     * @param       string $message
     *
     * @param       bool $appendNewline
     *
     * @param       int $indent
     *
     * @return 	    void
     *
     * @version     1.1.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function output(string $message, bool $appendNewline = false, int $indent = 0): void
    {
        static $Output = null;

        if ($Output === null) {
            $Output = new ConsoleOutput();
        }
        $indentString = '';

        if ($indent > 0) {
            $indentString = str_repeat(' ', $indent);
        }
        $Output->writeln($indentString.$message.($appendNewline === true ? "\n" : ''));

    } // output()


} // class CodeGenerator {}
