<?php
/**
 * Base Generator Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace Bplan\LaravelCodeGenerator\Generators\CodeGenerator;


use Bplan\LaravelCodeGenerator\Generators\CodeGenerator;


/**
 * Base Code Generator Class
 *
 * @version     3.0.0 / 2024-10-22
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
abstract class BaseGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     boolean $_buildOnlyMode
     */
    protected $_buildOnlyMode = false;


    /**
     * @var     array $_defaultReplacements
     */
    private $_defaultReplacements;


    /**
     * @var     string $_fileHash
     */
    private $_fileHash;


    /**
     * Der relative Pfad zum Zielverzeichnis, ausgehend vom Wurzelverzeichnis des Projekts
     *
     * Der Wert dieser Variablen muss in den abgeleiteten Generator-Klassen gesetzt werden.
     *
     * @var     string $_filePath
     */
    protected $_filePath;


    /**
     * @var     array $_technicalFieldNames
     */
    protected $_technicalFieldNames = [
        'active',
        'created_at',
        'deleted_at',
        'updated_at',
    ];


    /**
     * @var     array $_traits
     */
    protected $_traits = [];


    /**
     * @var     array $_uses
     */
    protected $_uses = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_resultCounters
     */
    private static $_resultCounters = [
        'created' => 0,
        'replaced' => 0,
        'protected' => 0,
        'unchanged' => 0,
    ];


    /**
     * @var     string $_startTime
     */
    protected static $_startTime;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       array|string
     *              string  'Illuminate\\Database\\Eloquent\\SoftDeletes' oder auch
     *                      'Illuminate\\Database\\Eloquent\\SoftDeletes as MySoftDelete'
     *
     *              array   [
     *                          'Illuminate\\Database\\Eloquent\\SoftDeletes',
     *                              oder auch
     *                          'Illuminate\\Database\\Eloquent\\SoftDeletes as MySoftDelete',
     *                          ...
     *                      ]
     *                      oder auch
     *                      [
     *                          'SoftDeletes' => 'Illuminate\\Database\\Eloquent\\SoftDeletes',
     *                              oder auch
     *                          'MySoftDelete' => 'Illuminate\\Database\\Eloquent\\SoftDeletes as MySoftDelete',
     *                      ]
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _addTrait(array|string $trait): self
    {
        if (is_array($trait)) {
            foreach ($trait as $key => $item) {
                if (is_numeric($key)) {
                    $this->_addTrait($item);

                    continue;
                }
                $this->_traits[$key] = $key;
                $this->_addUse($item);
            }
            return $this;
        }
        if (stripos($trait, ' as ') === false) {
            $name = basename($trait);

        } else {
            list($nul, $name) = preg_split("/ as /i", $trait);
        }
        $this->_traits[$name] = trim($name);
        $this->_addUse($trait);

        return $this;

    } // _addTrait()


    /**
     *
     * @param       array|string $use
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _addUse(array|string $use): self
    {
        if (is_array($use)) {
            foreach ($use as $class) {
                $this->_addUse($class);
            }
            return $this;
        }
        $this->_uses[$use] = $use;

        return $this;

    } // _addUse()


    /**
     *
     * @param       string $file
     *
     * @param       null|string & $ignoreReason
     *
     * @param       null|string & $previousDate
     *
     * @return      bool
     *
     * @version     1.2.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _checkFileWriteStatus(string $file, null|string & $ignoreReason = null): bool
    {
        if (!file_exists($file)) {
            return true;
        }
        $line = fgets(fopen($file, 'r'));

        if (strpos($line, CodeGenerator::CODE_GENERATOR_TAG) !== false) {
            list($tag, $hash, $nul) = explode('Δ', $line, 3);

            $hash = trim($hash);

            if ($hash !== $this->_fileHash) {
                return true;
            }
            $ignoreReason = 'unchanged';

            return false;
        }
        $ignoreReason = 'protected';

        return false;

    } // _checkFileWriteStatus()


    /**
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2024-10-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _init(): self
    {
        if (self::$_startTime === null) {
            self::$_startTime = date('Y-m-d H.i.s');
        }
        $author = config('code-generator.author');
        $copyright = config('code-generator.copyright');
        $devStart = (int) config('code-generator.dev-start');
        $currentYear = (int) date('Y');

        if ($devStart === $currentYear) {
            $copyright = $devStart.' '.$copyright;

        } elseif ($devStart === ($currentYear - 1)) {
            $copyright = $devStart.', '.$currentYear.' '.$copyright;

        } else {
            $copyright = $devStart.' - '.$currentYear.' '.$copyright;
        }
        $this->_defaultReplacements = [
            'author' => $author,
            'code-generator-tag' => CodeGenerator::CODE_GENERATOR_TAG.' Δ {{ hash }} Δ DYNAMICALLY GENERATED FILE - Do not modify this line! Remove it completely to protect the file from changes.',
            'copyright' => $copyright,
            'project' => config('code-generator.project'),
        ];
        return $this;

    } // _init()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _initFileHash(string $fileContents)
    {
        $this->_fileHash = md5($fileContents);

        return $this;

    } // _initFileHash()


    /**
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initTraits(): self
    {
        $traits = $this->_TypeDefinition->getTraits(static::$_generator);

        if ($traits !== null) {
            foreach ($traits as $name => $class) {
                if (is_numeric($name)) {
                    $this->_addTrait($class);

                    continue;
                }
                $this->_addTrait([$name => $class]);
            }
        }
        return $this;

    } // _initTraits()


    /**
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initUses(): self
    {
        $uses = $this->_TypeDefinition->getUses(static::$_generator);

        if ($uses !== null) {
            foreach ($uses as $class) {
                $this->_addUse($class);
            }
        }
        return $this;

    } // _initUses()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareTraitReplacement(): string
    {
        $replacement = '';

        if (!empty($this->_traits)) {
            ksort($this->_traits);
            /*
            **  Ersetzungs-String für die Use-Statements zusammenstellen. */
            $padding = str_repeat(' ', 4);

            foreach ($this->_traits as $trait) {
                $replacement .= "\n".$padding.'use '.$trait.';';
            }
            $replacement .= "\n\n";
        }
        return $replacement;

    } // _prepareTraitReplacement()


    /**
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareUseReplacement(): string
    {
        $replacement = '';

        if (!empty($this->_uses)) {
            ksort($this->_uses);
            /*
            **  Ersetzungs-String für die Traits zusammenstellen. */
            foreach ($this->_uses as $use) {
                $replacement .= "\n".'use '.$use.';';
            }
            $replacement .= "\n\n";
        }
        return $replacement;

    } // _prepareUseReplacement()


    /**
     * Führt Standard-Textersetzungen durch
     *
     * @param       string $fileContents
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2024-10-22
     * @history     BaseGenerator::_makeDefaultReplacements(), 1.1.0 / 2024-10-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _replaceDefaults(string $fileContents): string
    {
        foreach ($this->_defaultReplacements as $placeholder => $replacement) {
            $fileContents = self::replacePlaceholder($placeholder, $replacement, $fileContents);
        }
        $this->_initFileHash($fileContents);

        return $fileContents;

    } // _replaceDefaults()


    /**
     *
     * @param       string $fileContents
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _replaceFinal(string $fileContents): string
    {
        /*
        **  Nachfolgende Ersetzungen dürfen erst ausgeführt werden wenn der FileHash ermittelt wurde, weil sie
        **  bei jeder Erzeugung anders sind und den Hash verändern würden. */
        $fileContents = self::replacePlaceholder('date', date('Y-m-d'), $fileContents);
        $fileContents = self::replacePlaceholder('generated', date('Y-m-d H:i:s'), $fileContents);
        $fileContents = self::replacePlaceholder('hash', $this->_fileHash, $fileContents);

        return $fileContents;

    } // _replaceFinal()


    /**
     *
     * @param       string $fileName
     *
     * @param       array|string $paths
     *
     * @return 	    $this
     *
     * @version     2.4.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _writeFileContents(string $fileName, string $fileContents): self
    {
        $this->_buildOnlyMode = (bool) config('code-generator.test-mode', $this->_buildOnlyMode);
        /*
        **  Dieser Methodenaufruf muss vor der Überprüfung der Datei mit _checkFileWriteStatus() erfolgen,
        **  weil in der Methode der FileHash gebildet wird. */
        $fileContents = $this->_replaceDefaults($fileContents);

        $paths = [
            'build' => CodeGenerator::getBuildPath($this->_filePath),
        ];
        if ($this->_buildOnlyMode === false) {
            $paths['productive'] = CodeGenerator::getBasePath($this->_filePath);
        }
        foreach ($paths as $context => $path) {
            $file = $path.'/'.$fileName;

            $ignoreReason = null;
            /*
            **  Prüfen ob die Datei neu erzeugt werden muss und auch ob sie überhaupt überschrieben werden darf. */
            if ($this->_checkFileWriteStatus($file, $ignoreReason) === false) {
                /*
                **  Die Zieldatei ist geschützt oder unverändert und wird nicht überschrieben. */
                if ($context === 'productive') {
                    if ($ignoreReason === 'protected') {
                        CodeGenerator::output('• <comment>Ignoring protected file</comment> ............ '.$this->_filePath.'/'.$fileName, indent: 2);
                        self::$_resultCounters['protected']++;

                    } else {
                        CodeGenerator::output('• Skipping unchanged file ............ '.$this->_filePath.'/'.$fileName, indent: 2);
                        self::$_resultCounters['unchanged']++;
                    }
                } else {
                    /*
                    **  Ausgaben für die build-Dateien werden nur ausgegeben, wenn der BuildOnly-Modus aktiviert ist. */
                    if ($this->_buildOnlyMode === true) {
                        if ($ignoreReason === 'protected') {
                            CodeGenerator::output('• <comment>Ignoring protected build file</comment> ...... '.$this->_filePath.'/'.$fileName, indent: 2);
                            self::$_resultCounters['protected']++;

                        } else {
                            CodeGenerator::output('• Skipping unchanged build file ...... '.$this->_filePath.'/'.$fileName, indent: 2);
                            self::$_resultCounters['unchanged']++;
                        }
                    }
                }
                continue;
            }
            if (!file_exists($path)) {
                mkdir($path, 0777, true);
            }
            if ($context === 'productive') {
                if (file_exists($file)) {
                    CodeGenerator::output('• <comment>Replacing file</comment> ..................... '.$this->_filePath.'/'.$fileName, indent: 2);
                    self::$_resultCounters['replaced']++;

                } else {
                    CodeGenerator::output('• <comment>Creating new file</comment> .................. '.$this->_filePath.'/'.$fileName, indent: 2);
                    self::$_resultCounters['created']++;
                }
            } else {
                /*
                **  Ausgaben für die build-Dateien werden nur ausgegeben, wenn der BuildOnly-Modus aktiviert ist. */
                if ($this->_buildOnlyMode === true) {
                    if (file_exists($file)) {
                        CodeGenerator::output('• <comment>Replacing build file</comment> ............... '.$this->_filePath.'/'.$fileName, indent: 2);
                        self::$_resultCounters['replaced']++;

                    } else {
                        CodeGenerator::output('• <comment>Creating new build file</comment> ............ '.$this->_filePath.'/'.$fileName, indent: 2);
                        self::$_resultCounters['created']++;
                    }
                }
            }
            /*
            **  Backup der Datei erstellen. */
            $backupPath = CodeGenerator::getBackupPath(self::$_startTime.'/'.$this->_filePath);

            if (!file_exists($backupPath)) {
                mkdir($backupPath, 0777, true);
            }
            if (file_exists($file)) {
                copy($file, $backupPath.'/'.$fileName);
            }
            $fileContents = $this->_replaceFinal($fileContents);
            /*
            **  Datei schreiben. */
            file_put_contents($file, $fileContents);
        }
        return $this;

    } // _writeFileContents()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       null|string $group
     *
     * @return 	    array|int
     *
     * @version     1.0.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getResultCounters(null|string $group = null): array|int
    {
        if ($group === null) {
            return self::$_resultCounters;
        }
        if (!isset(self::$_resultCounters[$group])) {
            return null;
        }
        return self::$_resultCounters[$group];

    } // getResultCounters()


    /**
     * Ersetzt Platzhalter durch die übergebene Zeichenkette
     *
     * @param       string $placeholder
     *
     * @param       string $replace
     *
     * @param       string $subject
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function replacePlaceholder(string $placeholder, string $replace, string $subject): string
    {
        $placeholder = [
            '{{'.$placeholder.'}}',
            '{{ '.$placeholder.' }}',
        ];
        return str_replace($placeholder, $replace, $subject);

    } // replacePlaceholder()


} // abstract class BaseGenerator {}
