<?php
/**
 * JasonApi Code Generator Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace Bplan\LaravelCodeGenerator\Generators\CodeGenerator\JsonApi;


use Illuminate\Support\Str;


use Bplan\LaravelCodeGenerator\Definitions\TypeDefinition;
use Bplan\LaravelCodeGenerator\Enums\ModelType;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\JsonApiGenerator;


/**
 * JasonApi Code Generator Class
 *
 * @version     1.0.0 / 2024-10-26
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class RouteGenerator extends JsonApiGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     boolean $_buildOnlyMode
     */
    protected $_buildOnlyMode = true;


    /**
     * Der relative Pfad zum Zielverzeichnis, ausgehend vom Wurzelverzeichnis des Projekts
     *
     * @var     string $_filePath
     */
    protected $_filePath;


    /**
     * @var     array $_relations
     */
    protected $_relations = [];


    /**
     * @var     array $_resources
     */
    protected $_resources = [];


    /**
     * @var     array $_routingRestrictions
     */
    protected $_routingRestrictions = [];


    /**
     * @var     array $_servers
     */
    protected $_servers = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_generator
     */
    protected static $_generator = 'jsonapi.routes';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $tableName
     *
     * @param       string $name
     *
     * @param       string $type
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _addRelation(string $tableName, string $name, string $type): self
    {
        if (!isset($this->_relations[$tableName])) {
            $this->_relations[$tableName] = [];
        }
        $this->_relations[$tableName][$name] = [
            'name' => $name,
            'type' => $type,
        ];
        return $this;

    } // _addRelation()


    /**
     *
     * @param       string $tableName
     *
     * @param       string $name
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _addResource(string $tableName, string $name): self
    {
        $this->_resources[$tableName] = $name;

        return $this;

    } // _addResource()


    /**
     *
     * @param       string $server
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getPreparedFileContents(string $server): string
    {
        /*
        **  Stub-File auslesen und Ersetzungen durchführen. */
        $fileContents = file_get_contents($this->_stubPath.'/routes.stub');
        /*
        **  Platzhalter-Ersetzungen vornehmen. */
        $fileContents = parent::replacePlaceholder('server', $server, $fileContents);
        $fileContents = parent::replacePlaceholder('resources', $this->_prepareResourceReplacement($server), $fileContents);

        return $fileContents;

    } // _getPreparedFileContents()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareRelationshipReplacement(string $tableName): string
    {
        $replacement = '';

        if (!empty($this->_relations[$tableName])) {
            ksort($this->_relations[$tableName]);

            $relationsReplacement = '';

            foreach ($this->_relations[$tableName] as $relation) {
                $relationContents = file_get_contents($this->_stubPath.'/snippets/routes/relation.stub');

                $relationContents = parent::replacePlaceholder('relation-name', $relation['name'], $relationContents);
                $relationContents = parent::replacePlaceholder('relation-type', $relation['type'], $relationContents);

                $relationsReplacement .= $relationContents;
            }
            $fileContents = file_get_contents($this->_stubPath.'/snippets/routes/relationships.stub');

            $replacement .= parent::replacePlaceholder('relations', $relationsReplacement, $fileContents);
        }
        return $replacement;

    } // _prepareRelationshipReplacement()


    /**
     *
     * @param       string $server
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareResourceReplacement(string $server): string
    {
        $replacement = '';

        if (!empty($this->_resources)) {
            ksort($this->_resources);

            foreach ($this->_resources as $tableName => $resource) {
                $routingRestriction = $this->_routingRestrictions[$server][$tableName];

                if ($routingRestriction === true) {
                    continue;
                }
                $restrictionReplacement = $this->_prepareRoutingRestrictionReplacement($routingRestriction);

                $fileContents = file_get_contents($this->_stubPath.'/snippets/routes/resource.stub');

                $fileContents = parent::replacePlaceholder('resource-name', $resource, $fileContents);
                $fileContents = parent::replacePlaceholder('restriction', $restrictionReplacement, $fileContents);
                $fileContents = parent::replacePlaceholder('relationships', $this->_prepareRelationshipReplacement($tableName), $fileContents);

                $replacement .= $fileContents;
            }
        }
        return $replacement;

    } // _prepareResourceReplacement()


    /**
     *
     * @param       array|bool|string $routingRestrictions
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareRoutingRestrictionReplacement(array|bool|string $routingRestriction): string
    {
        $restrictionReplacement = '';

        if ($routingRestriction !== false) {
            if (is_string($routingRestriction)) {
                if ($routingRestriction === 'readOnly') {
                    $restrictionReplacement = '->readOnly()';

                } else {
                    $restrictionReplacement = '->'.$routingRestriction;
                }
            } else {
                $restrictionReplacement = '->'.$routingRestriction['method'].'(\''.implode('\', \'', $routingRestriction['actions']).'\')';
            }
        }
        return $restrictionReplacement;

    } // _prepareRoutingRestrictionReplacement()


    /**
     *
     * @param       string $tableName
     *
     * @param       array|null $routing
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareRoutingRestrictions(string $tableName, array|null $routing): self
    {
        if ($routing === null) {
            foreach ($this->_servers as $server) {
                $this->_routingRestrictions[$server][$tableName] = false;
            }
            return $this;
        }
        foreach ($this->_servers as $server) {
            if (!isset($routing[$server]) || $routing[$server] === true) {
                $this->_routingRestrictions[$server][$tableName] = false;

                continue;
            }
            if ($routing[$server] === false) {
                $this->_routingRestrictions[$server][$tableName] = true;

                continue;
            }
            $this->_routingRestrictions[$server][$tableName] = $routing[$server];
        }
        return $this;

    } // _prepareRoutingRestrictions()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _initServers(): self
    {
        foreach (config('code-generator.jsonapi-servers') as $server => $nul) {
            $this->_servers[$server] = strtolower($server);
        }
        return $this;

    } // _initServers()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _process(): self
    {
        $this->_initServers();
        $this->_processTypeDefinitions();

        $this->_writeFile();

        return $this;

    } // _process()


    /**
     *
     * @param       TypeDefinition $TypeDefinition
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _processAdditionalRelations(TypeDefinition $TypeDefinition): self
    {
        $tableName = $TypeDefinition->getTableName();
        /*
        **  Zusätzliche Relationen abarbeiten. */
        foreach ($TypeDefinition->getAdditionalRelations() as $additionalRelation) {
            $relationName = $additionalRelation['relationName'];
            $relationType = 'hasMany';

            $this->_addRelation($tableName, $relationName, $relationType);
        }
        return $this;

    } // _processAdditionalRelations()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _processBackReferences(string $tableName): self
    {
        $backReferences = CodeGenerator::getBackReferences($tableName);

        if ($backReferences !== null) {
            /*
            **  HasMany- und BelongsToMany-Relationen aufbereiten. */
            foreach ($backReferences as $BackReference) {
                $columnName = $BackReference->getColumnName();
                $relatedTable = $BackReference->getTableName();

                $RelatedTypeDefinition = CodeGenerator::getTypeDefinition($relatedTable);

                $pivotTables = $RelatedTypeDefinition->getPivotTables();
                /*
                **  ForeignId-Felder haben nicht zwingend eine Constraint (siehe users.current_team_id). Aus dem
                **  Grund muss hier überprüft werden ob es eine entsprechende Definition gibt. */
                if ($BackReference->getModelType() === ModelType::Pivot->name && isset($pivotTables[$tableName])) {
                    $relationName = Str::slug($pivotTables[$tableName]);
                    $relationType = 'hasMany';

                    $this->_addRelation($tableName, $relationName, $relationType);

                } else {
                    $foreignKey = $RelatedTypeDefinition->getForeignKey($columnName);
                    $resourceName = $BackReference->getTableName('slug');

                    if ($foreignKey !== null && !empty($foreignKey['reverseRelationName'])) {
                        $relationName = $foreignKey['reverseRelationName'];
                    } else {
                        $relationName = $resourceName;
                    }
                    $relationType = 'hasMany';
                }
                $this->_addRelation($tableName, $relationName, $relationType);
            }
        }
        return $this;

    } // _processBackReferences()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _processTypeDefinitions(): self
    {
        foreach ($this->_typeDefinitions as $TypeDefinition) {
            $tableName = $TypeDefinition->getTableName();
            $resourceName = $TypeDefinition->getTypeName('slug', singular: false);

            $this->_addResource($tableName, $resourceName);

            $FieldDefinitions = $TypeDefinition->getFieldDefinitions($tableName);
            /*
            **  Felddefinitionen zusammenstellen. */
            foreach ($FieldDefinitions as $FieldDefinition) {
                if ($FieldDefinition->isForeignKey() === false) {
                    continue;
                }
                $relationName = $FieldDefinition->getName('slug');
                $relationType = 'hasOne';

                $this->_addRelation($tableName, $relationName, $relationType);
            }
            $this->_prepareRoutingRestrictions($tableName, $TypeDefinition->getRouting());
            $this->_processBackReferences($tableName);
            $this->_processAdditionalRelations($TypeDefinition);
        }
        return $this;

    } // _processTypeDefinitions()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function _writeFile(): self
    {
        /*
        **  Routen-Dateien für jeden konfigurierten Server zusammenstellen und speichern. */
        foreach ($this->_servers as $server) {
            $fileContents = $this->_getPreparedFileContents($server);
            /*
            **  Routen-Datei für den Server schreiben. */
            $this->_filePath = 'routes/api';

            parent::_writeFileContents($server.'.php', $fileContents);
        }
        return $this;

    } // _writeFile()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class RouteGenerator extends JsonApiGenerator {}
