<?php
/**
 * Code Generator Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace Bplan\LaravelCodeGenerator\Generators;


use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Schema\ForeignIdColumnDefinition;
use Illuminate\Database\Schema\ForeignKeyDefinition;
use Illuminate\Support\Str;
use Symfony\Component\Console\Output\ConsoleOutput;


use Bplan\LaravelCodeGenerator\Definitions\BackReference;
use Bplan\LaravelCodeGenerator\Definitions\TypeDefinition;
use Bplan\LaravelCodeGenerator\Enums\ModelType;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\BaseGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\CSharpFileGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\JsonApiFileGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\JsonApiGenerator;
use Bplan\LaravelCodeGenerator\Generators\CodeGenerator\LaravelFileGenerator;


/**
 * Code Generator Class
 *
 * @version     8.1.0 / 2024-12-28
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class CodeGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string CODE_GENERATOR_TAG
     */
    const CODE_GENERATOR_TAG = '#CodeGenerator';


    /**
     * Wenn der Wert eines Json-Feldes ein Array ist
     *
     * Beispiele:
     *  ["de", "en"]
     *  [{"de":"deutsch"}, {"en":"englisch"}]
     *
     * @var     string JSON_TYPE_ARRAY
     */
    const JSON_TYPE_ARRAY = 'List';


    /**
     * Wenn der Wert eines Json-Feldes ein Objekt ist
     *
     * Beispiel:
     *  {"de":"deutsch", "en":"englisch"}
     *
     * @var     string JSON_TYPE_OBJECT
     */
    const JSON_TYPE_OBJECT = 'Hash';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_modelType
     */
    private $_modelType;


    /**
     * @var         string $_tableName
     */
    private $_tableName;


    /**
     * @var         array $_typeDefinition
     */
    private $_typeDefinition;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         array $_backReferences
     */
    private static $_backReferences = [];


    /**
     * @var         string $_cachePath
     */
    private static $_cachePath;


    /**
     * @var         array $_collectedForeignKeys
     */
    private static $_collectedForeignKeys = [];


    /**
     * @var         array $_collectedIndexes
     */
    private static $_collectedIndexes = [];


    /**
     * @var         array $_collectedTypeDefinitions
     */
    private static $_collectedTypeDefinitions = [];


    /**
     * @var         int $_defaultOutputIndent
     */
    private static $_defaultOutputIndent = 2;


    /**
     * @var         int $_errorGeneratorsMaxLength
     */
    private static $_errorGeneratorsMaxLength = 0;


    /**
     * @var         int $_errorTypeNameMaxLength
     */
    private static $_errorTypeNameMaxLength = 0;


    /**
     * @var         array $_errors
     */
    private static $_errors = [];


    /**
     * Farbedefinitionen für die Ausgabe
     *
     * Mögliche Farbbezeichnungen:
     *  black
     *  blue, bright-blue,
     *  cyan, bright-cyan
     *  gray
     *  green, bright-green
     *  magenta, bright-magenta
     *  red, bright-red
     *  white, bright-white
     *  yellow, bright-yellow
     *
     * @var         array $_outputColors
     * @version     2.0.0 / 2024-12-28
     */
    private static $_outputColors = [
        'creating' => 'bright-green',
        'ignoring' => null,
        'omitting' => 'bright-blue',
        'protecting' => 'bright-red',
        'replacing' => 'bright-yellow',
    ];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(Blueprint $Blueprint, array $additionalSettings = [])
    {
        if (config('code-generator.active') !== true) {
            return;
        }
        $this->_init($Blueprint, $additionalSettings);

    } // __construct()


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return 	    $this
     *
     * @version     1.5.0 / 2024-10-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(Blueprint $Blueprint, array $additionalSettings = []): self
    {
        $TypeDefinition = new TypeDefinition($Blueprint, $additionalSettings);

        $this->_tableName = $Blueprint->getTable();

        if (isset($additionalSettings['pivotTables'])) {
            $this->_modelType = ModelType::Pivot->name;
        } else {
            $this->_modelType = ModelType::Model->name;
        }
        $this->_processCommands($Blueprint->getCommands());

        self::$_collectedTypeDefinitions[$this->_tableName] = $TypeDefinition;

        $TypeDefinition->writeCacheFile();

        return $this;

    } // _init()


    /**
     * Verarbeitet Commands und speichert die ermittelten Informationen in statischen Variablen
     *
     * In den Commands des Blueprint-Objekts sind unter anderem die ForeignKey-Definitionen enthalten.
     *
     * @param       array $commands
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processCommands(array $commands): self
    {
        if (!isset(self::$_collectedForeignKeys[$this->_tableName])) {
            self::$_collectedForeignKeys[$this->_tableName] = [];
            // self::$_collectedIndexes[$this->_tableName] = [];
        }
        foreach ($commands as $Command) {
            $attributes = $Command->getAttributes();

            if ($Command instanceof ForeignKeyDefinition) {
                $column = $attributes['columns'][0];
                $relatedTable = $attributes['on'];

                // $strippedColumn = Str::replaceLast('_id', '', $column);

                // unset($attributes['algorithm']);
                // unset($attributes['columns']);
                // unset($attributes['name']);
                // unset($attributes['on']);

                // $attributes['column'] = $column;
                // $attributes['relatedTable'] = $relatedTable;
                // $attributes['relationName'] = Str::slug($strippedColumn, '-');
                // $attributes['relationNameStudlyCaps'] = Str::studly($strippedColumn);
                // $attributes['relationType'] = 'BelongsTo';
                // $attributes['resourceName'] = Str::slug($relatedTable, '-');
                // $attributes['resourceNameCamelCase'] = Str::camel($relatedTable, '-');
                // $attributes['resourceNameStudlyCaps'] = Str::studly($relatedTable);

                // ksort($attributes);

                self::$_collectedForeignKeys[$this->_tableName][$column] = $attributes;

                if (!isset(self::$_backReferences[$relatedTable])) {
                    self::$_backReferences[$relatedTable] = [];
                }
                self::$_backReferences[$relatedTable][] = new BackReference($this->_modelType, $this->_tableName, $column);

                continue;
            }
        }
        return $this;

    } // _processCommands()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.2.0 / 2024-10-14
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _finish(): void
    {
        $resultCounters = BaseGenerator::getResultCounters();

        $total = array_sum($resultCounters);
        $totalLength = strlen($total);
        $pad = 45;

        $createdText = 'Created Files   '.str_repeat(' ', $pad + 3).str_pad($resultCounters['created'], $totalLength, ' ', STR_PAD_LEFT);
        $protectedText = 'Protected Files '.str_repeat(' ', $pad + 3).str_pad($resultCounters['protected'], $totalLength, ' ', STR_PAD_LEFT);
        $replacedText = 'Replaced Files  '.str_repeat(' ', $pad + 3).str_pad($resultCounters['replaced'], $totalLength, ' ', STR_PAD_LEFT);

        $output = "\n"
            ."\n".'   <options=bold,underscore>Results            '.str_repeat(' ', $pad).str_repeat(' ', $totalLength).'  </>';

        $output .= "\n".'  |'.(($resultCounters['created'] > 0) ? ' <fg='.self::$_outputColors['creating'].'>'.$createdText.'</>' : ' '.$createdText).' |';
        $output .= "\n".'  |'.(($resultCounters['replaced'] > 0) ? ' <fg='.self::$_outputColors['replacing'].'>'.$replacedText.'</>' : ' '.$replacedText).' |';
        $output .= "\n".'  |'.(($resultCounters['protected'] > 0) ? ' <fg='.self::$_outputColors['protecting'].'>'.$protectedText.'</>' : ' '.$protectedText).' |';

        $output .= "\n".'  |<options=underscore> Unchanged Files    '.str_repeat(' ', $pad).str_pad($resultCounters['unchanged'], $totalLength, ' ', STR_PAD_LEFT).' </>|'
            ."\n".'  |<options=bold,underscore> Total              '.str_repeat(' ', $pad).str_pad($total, $totalLength, ' ', STR_PAD_LEFT).' </>|';

        self::output($output);
        self::output('<fg=green;options=bold>Done</>', appendNewline: 1, indent: 2, prependNewline: 1);

        if (!empty(self::$_errors)) {
            self::output('<options=bold>'.count(self::$_errors).' Error(s)</>', appendNewline: 1, prependNewline: 1);

            foreach (self::$_errors as $error) {
                self::output('<fg=red>['
                    .str_pad($error['generator'].']</> ', self::$_errorGeneratorsMaxLength + 5, ' ')
                    .'> '
                    .str_pad($error['typeName'].'', self::$_errorTypeNameMaxLength, ' ').' : '
                    .$error['message'], bullet: true);
            }
        }
    } // _finish()


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2024-11-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function addError(string $generator, string $typeName, string $error)
    {
        self::$_errors[] = [
            'generator' => $generator,
            'message' => $error,
            'typeName' => $typeName,
        ];
        self::$_errorGeneratorsMaxLength = max(self::$_errorTypeNameMaxLength, strlen($generator));
        self::$_errorTypeNameMaxLength = max(self::$_errorTypeNameMaxLength, strlen($typeName));

    } // addError()


    /**
     *
     * @return 	    void
     *
     * @version     1.9.0 / 2024-11-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function generate(): void
    {
        $creates = config('code-generator.creates');

        self::output("\n\n".'<info>  CodeGenerator</info> Starting code generation - TestMode '.(config('code-generator.test-mode') == true ? '<info> on </info>' : '<error> off </error>'), appendNewline: true);

        ksort(self::$_collectedTypeDefinitions);

        foreach (self::$_collectedTypeDefinitions as $tableName => $TypeDefinition) {
            $typeName = $TypeDefinition->getTypeName('studly', singular: true);

            self::output('Starting code generation for resource <options=bold,underscore>'.$typeName.'</>', appendNewline: 1, prependNewline: 2);

            foreach ($creates as $create => $status) {
                if ($status === false) {
                    continue;
                }
                switch ($create) {
                    case 'CSharp';
                        CSharpFileGenerator::process($TypeDefinition);
                        break;

                    case 'JsonApi';
                        JsonApiFileGenerator::process($TypeDefinition);
                        break;

                    case 'Laravel';
                        LaravelFileGenerator::process($TypeDefinition);
                        break;
                }
            }
        }
        if ($creates['JsonApi'] === true) {
            self::output('Starting global code generation for <options=bold,underscore>JsonApi</>', appendNewline: 1, prependNewline: 2);

            JsonApiGenerator::process(self::$_collectedTypeDefinitions, self::$_backReferences);
        }
        self::_finish();

    } // generate()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBackReferences(string $tableName): array|null
    {
        if (!isset(self::$_backReferences[$tableName])) {
            return null;
        }
        return self::$_backReferences[$tableName];

    } // getBackReferences()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     1.1.0 / 2024-10-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBackupPath(null|string $path = null): string
    {
        $backupPath = str_replace('\\', '/', storage_path('app/backup/'.basename(__CLASS__)));

        if ($path === null) {
            return $backupPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $backupPath.$path;
        }
        return $backupPath.'/'.$path;

    } // getBackupPath()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     2.0.0 / 2024-11-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBasePath(null|string $path = null, null|string $package = null): string
    {
        $basePath = str_replace('\\', '/', base_path());

        if ($package !== null) {
            $basePath .= '/vendor/'.$package;
        }
        if ($path === null) {
            return $basePath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $basePath.$path;
        }
        return $basePath.'/'.$path;

    } // getBasePath()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     2.0.0 / 2024-11-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBuildPath(null|string $path = null, null|string $package = null): string
    {
        $buildPath = str_replace('\\', '/', storage_path('app/build/'.basename(__CLASS__)));

        if ($package !== null) {
            $buildPath .= '/vendor/'.$package;
        }
        if ($path === null) {
            return $buildPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $buildPath.$path;
        }
        return $buildPath.'/'.$path;

    } // getBuildPath()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getCachePath(): string
    {
        if (self::$_cachePath !== null) {
            return self::$_cachePath;
        }
        self::$_cachePath = str_replace('\\', '/', storage_path('app/cache/'.basename(__CLASS__)));

        if (!file_exists(self::$_cachePath)) {
            mkdir(self::$_cachePath, 0777, true);
        }
        return self::$_cachePath;

    } // getCachePath()


    /**
     *
     * @param       string $status
     *
     * @return      void
     *
     * @version     1.0.0 / 2024-10-14
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getOutputColor(string $status)
    {
        if (!isset(self::$_outputColors[$status])) {
            return null;
        }
        return self::$_outputColors[$status];

    } // getOutputColor()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getStubPath(null|string $path = null): string
    {
        $stubPath = str_replace('\\', '/', realpath(__DIR__.'/../..'))
            .'/stubs/code-generator';

        if ($path === null) {
            return $stubPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $stubPath.$path;
        }
        return $stubPath.'/'.$path;

    } // getStubPath()


    /**
     *
     * @param       string $tableName
     *
     * @return      array|null
     *
     * @version     2.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getTypeDefinition(string $tableName): null|TypeDefinition
    {
        if (!isset(self::$_collectedTypeDefinitions[$tableName])) {
            return null;
        }
        return self::$_collectedTypeDefinitions[$tableName];

    } // getTypeDefinition()


    /**
     *
     * @param       string $message
     *
     * @param       int|null $appendNewline
     *
     * @param       bool $bullet
     *
     * @param       null|string $color
     *
     * @param       string $file
     *
     * @param       int $indent
     *
     * @param       int|null $prependNewline
     *
     * @return 	    void
     *
     * @version     2.2.0 / 2024-11-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function output(string $message, int|null $appendNewline = null, bool $bullet = false, null|string $color = null, string $file = null, int $indent = 0, int|null $prependNewline = null): void
    {
        static $Output = null;

        if ($Output === null) {
            $Output = new ConsoleOutput();
        }
        $outputString = ($prependNewline !== null ? str_repeat("\n", $prependNewline) : '')
            .($indent > 0 || self::$_defaultOutputIndent ? str_repeat(' ', $indent + self::$_defaultOutputIndent) : '')
            .($bullet === true ? '• ' : '')
            .$message
            .($file !== null ? ' <fg=gray>'.str_repeat('.', 30 - strlen($message)).'</> '.$file : '').''
            .($appendNewline !== null ? str_repeat("\n", $appendNewline) : '');

        if ($color !== null) {
            $outputString = '<fg='.$color.'>'.$outputString.'</>';
        }
        $Output->writeln($outputString);

    } // output()


} // class CodeGenerator {}
