<?php
/**
 * JasonApi Code Generator Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Generators\CodeGenerator\JsonApi;


use BplanBase\CodeGenerator\Definitions\TypeDefinition;
use BplanBase\CodeGenerator\Enums\Generator;
use BplanBase\CodeGenerator\Enums\ModelType;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\JsonApiGenerator;
use Illuminate\Support\Str;


/**
 * JasonApi Code Generator Class
 *
 * @version     1.10.0 / 2025-02-18
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class RouteGenerator extends JsonApiGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Der Name der obersten Verzeichnisebene nach dem BasePath (z.B. "app").
     *
     * @var     string $_baseDir
     */
    protected $_baseDir = 'routes';


    /**
     * @var     boolean $_buildOnlyMode
     */
    protected $_buildOnlyMode = false;


    /**
     * Der relative Pfad zum Zielverzeichnis, ausgehend vom Wurzelverzeichnis des Projekts
     *
     * @var     string $_filePath
     */
    protected $_filePath;


    /**
     * @var     string $_packageName
     */
    private $_packageName;


    /**
     * @var     array $_packages
     */
    private $_packages = [];


    /**
     * @var     array $_relations
     */
    private $_relations = [];


    /**
     * @var     array $_resources
     */
    private $_resources = [];


    /**
     * @var     array $_routingRestrictions
     */
    private $_routingRestrictions = [];


    /**
     * @var     array $_servers
     */
    private $_servers = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_generator
     */
    protected static $_generator = Generator::JsonApiRoute->value;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $tableName
     *
     * @param       string $name
     *
     * @param       string $type
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2024-11-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _addRelation(string $tableName, string $name, string $type): self
    {
        if (!isset($this->_relations[$this->_packageName][$tableName])) {
            $this->_relations[$this->_packageName][$tableName] = [];
        }
        if (isset($this->_relations[$this->_packageName][$tableName][$name])) {
            CodeGenerator::addError(self::$_generator, $tableName, 'Multiple relations ['.$name.']');

            if (empty($this->_errors[$this->_packageName][$tableName])) {
                $this->_errors[$this->_packageName][$tableName][$name][] = $this->_relations[$this->_packageName][$tableName][$name];
            }
            $this->_errors[$this->_packageName][$tableName][$name][] = [
                'name' => $name,
                'type' => $type,
            ];
            $name .= '-failure';
        }
        $this->_relations[$this->_packageName][$tableName][$name] = [
            'name' => $name,
            'type' => $type,
        ];
        return $this;

    } // _addRelation()


    /**
     *
     * @param       string $tableName
     *
     * @param       string $name
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2024-11-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _addResource(string $tableName, string $name): self
    {
        $this->_resources[$this->_packageName][$tableName] = $name;

        return $this;

    } // _addResource()


    /**
     *
     * @param       string $package
     *
     * @param       string $server
     *
     * @return 	    string
     *
     * @version     2.2.0 / 2025-02-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getPreparedFileContents(string $package, string $server): string
    {
        /*
        **  Stub-File auslesen und Ersetzungen durchführen. */
        $fileContents = file_get_contents($this->_stubPath.'/routes.stub');

        if ($package === CodeGenerator::PROJECT_KEY_NAME) {
            /*
            **  Die API-Routen im Projekt sind automatisch/implizit mit dem Präfix "api"
            **  versehen, alleine dadurch, dass sie in der Routen-Datei "api.php" definiert
            **  sind. Das funktioniert weil die Routendatei in "bootstrap/app.php" mit der
            **  Methode withRouting() entsprechend registriert wird. */
            $prefix = $server;
            $middleware = '';

        } else {
            /*
            **  In Paketen muss dem Präfix explizit "api" vorangestellt werden. Außerdem muss
            **  die api-Middleware eingebunden werden. */
            $prefix = 'api/'.$server;
            $middleware = "\n".'    ->middleware(\'api\')';

            $fileContents = parent::replacePlaceholder('package', $package, $fileContents);
        }
        $fileContents = parent::replacePlaceholder('middleware', $middleware, $fileContents);
        $fileContents = parent::replacePlaceholder('prefix', $prefix, $fileContents);
        $fileContents = parent::replacePlaceholder('resources', $this->_prepareResourceReplacement($package, $server), $fileContents);
        $fileContents = parent::replacePlaceholder('server', $server, $fileContents);

        return $fileContents;

    } // _getPreparedFileContents()


    /**
     *
     * @param       string $package
     *
     * @param       string $tableName
     *
     * @return 	    string
     *
     * @version     2.0.0 / 2024-11-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareRelationshipReplacement(string $package, string $tableName): string
    {
        $replacement = '';

        if (!empty($this->_relations[$package][$tableName])) {
            ksort($this->_relations[$package][$tableName]);

            $relationsReplacement = '';

            foreach ($this->_relations[$package][$tableName] as $relation) {
                $relationContents = file_get_contents($this->_stubPath.'/snippets/routes/relation.stub');

                $relationContents = parent::replacePlaceholder('relation-name', $relation['name'], $relationContents);
                $relationContents = parent::replacePlaceholder('relation-type', $relation['type'], $relationContents);

                $relationsReplacement .= $relationContents;
            }
            $fileContents = file_get_contents($this->_stubPath.'/snippets/routes/relationships.stub');

            $replacement .= parent::replacePlaceholder('relations', $relationsReplacement, $fileContents);
        }
        return $replacement;

    } // _prepareRelationshipReplacement()


    /**
     *
     * @param       string $package
     *
     * @param       string $server
     *
     * @return 	    string
     *
     * @version     2.0.0 / 2024-11-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareResourceReplacement(string $package, string $server): string
    {
        $replacement = '';

        if (!empty($this->_resources[$package])) {
            ksort($this->_resources[$package]);

            foreach ($this->_resources[$package] as $tableName => $resource) {
                $routingRestriction = $this->_routingRestrictions[$server][$tableName];

                if ($routingRestriction === true) {
                    continue;
                }
                $restrictionReplacement = $this->_prepareRoutingRestrictionReplacement($routingRestriction);

                $fileContents = file_get_contents($this->_stubPath.'/snippets/routes/resource.stub');

                $fileContents = parent::replacePlaceholder('resource-name', $resource, $fileContents);
                $fileContents = parent::replacePlaceholder('restriction', $restrictionReplacement, $fileContents);
                $fileContents = parent::replacePlaceholder('relationships', $this->_prepareRelationshipReplacement($package, $tableName), $fileContents);

                $replacement .= $fileContents;
            }
        }
        return $replacement;

    } // _prepareResourceReplacement()


    /**
     *
     * @param       array|bool|string $routingRestrictions
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2024-11-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareRoutingRestrictionReplacement(array|bool|string $routingRestriction): string
    {
        $restrictionReplacement = '';

        if ($routingRestriction !== false) {
            if (is_string($routingRestriction)) {
                if ($routingRestriction === 'readOnly') {
                    $restrictionReplacement = '->readOnly()';

                } elseif ($routingRestriction === 'relationOnly') {
                    /*
                    **  Die Routing-Definition "relationOnly" bewirkt, dass zwar eine Routen-Definition zur Resource
                    **  angelegt wird, durch "only()" werden aber keine Routen erzeugt. Relationen-Routen sind so noch
                    **  möglich.  */
                    $restrictionReplacement = '->only()';

                } else {
                    $restrictionReplacement = '->'.$routingRestriction;
                }
            } else {
                $restrictionReplacement = '->'.$routingRestriction['method'].'(\''.implode('\', \'', $routingRestriction['actions']).'\')';
            }
        }
        return $restrictionReplacement;

    } // _prepareRoutingRestrictionReplacement()


    /**
     *
     * @param       string $tableName
     *
     * @param       array|null $routing
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareRoutingRestrictions(string $tableName, array|null $routing): self
    {
        if ($routing === null) {
            /*
            **  Die routing-Option hat eine einschränkende Funktion. Wenn keine Definition zum Routing
            **  getroffen wird, dann werden alle Routen erzeugt.
            **
            **  FALSE = keine Einschränkungen */
            foreach ($this->_servers as $server) {
                $this->_routingRestrictions[$server][$tableName] = false;
            }
            return $this;
        }
        foreach ($this->_servers as $server) {
            /*
            **  Wenn "routing" explizit auf TRUE gesetzt wurde, dann hat das den geichen Effekt wie eine fehlende
            **  Definition zum Routing. Es bestehen keine Einschränkungen. */
            if (!isset($routing[$server]) || $routing[$server] === true) {
                $this->_routingRestrictions[$server][$tableName] = false;

                continue;
            }
            if ($routing[$server] === false) {
                /*
                **  Der Wert FALSE beim Routing unterdrückt die Routendefinition für die Resource vollständig. Es
                **  werden auch keine Routen für die Relationships erzeugt.
                **
                **  TRUE = volle Beschränkung */
                $this->_routingRestrictions[$server][$tableName] = true;

                continue;
            }
            /*
            **  Alle weiteren Werte (Strings oder Arrays) werden unverändert übernommen und später ausgewertet/angewendet. */
            $this->_routingRestrictions[$server][$tableName] = $routing[$server];
        }
        return $this;

    } // _prepareRoutingRestrictions()


    /**
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2025-02-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _initServers(): self
    {
        foreach ($this->_jsonApiServers as $server => $nul) {
            $this->_servers[$server] = strtolower($server);
        }
        return $this;

    } // _initServers()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _process(): self
    {
        $this->_initServers();
        $this->_processTypeDefinitions();

        $this->_writeFile();

        return $this;

    } // _process()


    /**
     *
     * @param       TypeDefinition $TypeDefinition
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processAdditionalRelations(TypeDefinition $TypeDefinition): self
    {
        $tableName = $TypeDefinition->getTableName();
        /*
        **  Zusätzliche Relationen abarbeiten. */
        foreach ($TypeDefinition->getAdditionalRelations() as $additionalRelation) {
            $relationName = $additionalRelation['relationName'];
            $relationType = 'hasMany';

            $this->_addRelation($tableName, $relationName, $relationType);
        }
        return $this;

    } // _processAdditionalRelations()


    /**
     *
     * @param       string $packageName
     *
     * @param       string $tableName
     *
     * @return 	    $this
     *
     * @version     2.0.0 / 2025-02-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processBackReferences(string $packageName, string $tableName): self
    {
        $backReferences = CodeGenerator::getBackReferences($tableName);

        if ($backReferences !== null) {
            /*
            **  HasMany- und BelongsToMany-Relationen aufbereiten. */
            foreach ($backReferences as $BackReference) {
                $columnName = $BackReference->getColumnName();
                $relatedTable = $BackReference->getTableName();

                $RelatedTypeDefinition = CodeGenerator::getTypeDefinition($relatedTable);

                $pivotTables = $RelatedTypeDefinition->getPivotTables();
                /*
                **  ForeignId-Felder haben nicht zwingend eine Constraint (siehe users.current_team_id). Aus dem
                **  Grund muss hier überprüft werden ob es eine entsprechende Definition gibt. */
                if ($BackReference->getModelType() === ModelType::Pivot->name && isset($pivotTables[$tableName])) {
                    $relationName = Str::slug($pivotTables[$tableName]);

                } else {
                    $relationName = $BackReference->getTableName('slug');

                    $foreignKey = $RelatedTypeDefinition->getForeignKey($columnName);

                    if ($foreignKey !== null && !empty($foreignKey['reverseRelationName'])) {
                        $relationName = $foreignKey['reverseRelationName'];
                    }
                }
                if ($packageName === CodeGenerator::PROJECT_KEY_NAME) {
                    /*
                    **  Zum Projekt selbst werden alle Relationen hinzugefügt. */
                    $this->_addRelation($tableName, $relationName, 'hasMany');

                } else {
                    /*
                    **  Auf Paket-Ebene werden Rückwärts-Relationen innerhalb des Pakets hinzugefügt. */
                    if ($packageName === $RelatedTypeDefinition->getPackage()?->getName()) {
                        $this->_addRelation($tableName, $relationName, 'hasMany');
                    }
                }
            }
        }
        return $this;

    } // _processBackReferences()


    /**
     *
     * @return 	    $this
     *
     * @version     1.5.0 / 2025-02-18
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processTypeDefinitions(): self
    {
        foreach ($this->_typeDefinitions as $TypeDefinition) {
            $tableName = $TypeDefinition->getTableName();

            if ($TypeDefinition->isForPackage()) {
                $resourceName = $TypeDefinition->getTypeName('slug', singular: false);

                if (($prefix = $TypeDefinition->getPackage()->getPrefix()) !== null) {
                    $resourceName = $prefix.'-'.$resourceName;
                }
                $this->_packageName = $TypeDefinition->getPackage()->getName();

            } else {
                $resourceName = $TypeDefinition->getTypeName('slug', singular: false);

                $this->_packageName = CodeGenerator::PROJECT_KEY_NAME;
            }
            $this->_packages[$this->_packageName] = $this->_packageName;

            $this->_addResource($tableName, $resourceName);

            $FieldDefinitions = $TypeDefinition->getFieldDefinitions($tableName);
            /*
            **  Felddefinitionen zusammenstellen. */
            foreach ($FieldDefinitions as $FieldDefinition) {
                if ($FieldDefinition->isForeignKey() === false) {
                    continue;
                }
                $relationName = $FieldDefinition->getName('slug');
                $relationType = 'hasOne';

                $this->_addRelation($tableName, $relationName, $relationType);
            }
            $this->_prepareRoutingRestrictions($tableName, $TypeDefinition->getRouting());
            $this->_processBackReferences($this->_packageName, $tableName);
            $this->_processAdditionalRelations($TypeDefinition);
        }
        return $this;

    } // _processTypeDefinitions()


    /**
     *
     * @return 	    $this
     *
     * @version     1.5.0 / 2025-02-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _writeFile(): self
    {
        /*
        **  Pakete abarbeiten. */
        foreach ($this->_packages as $package) {
            $packageParam = $package === CodeGenerator::PROJECT_KEY_NAME ? null : $package;

            $this->_filePath = 'api';
            /*
            **  Routen-Dateien für jeden konfigurierten Server zusammenstellen und speichern. */
            foreach ($this->_servers as $server) {
                $fileContents = $this->_getPreparedFileContents($package, $server);
                /*
                **  Routen-Datei für den Server schreiben. */
                parent::_writeFileContents($server.'.php', $fileContents, $packageParam);
            }
            /*
            **  Stub-File auslesen und Ersetzungen durchführen. */
            $fileContents = file_get_contents($this->_stubPath.'/routes.api.stub');

            $this->_filePath = null;
            /*
            **  api-Routendatei erstellen. */
            parent::_writeFileContents('api.php', $fileContents, $packageParam);
        }
        return $this;

    } // _writeFile()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class RouteGenerator extends JsonApiGenerator {}
