<?php
/**
 * Generator Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Generators\CodeGenerator\CSharp;


use BplanBase\CodeGenerator\Definitions\FieldDefinition;
use BplanBase\CodeGenerator\Definitions\TypeDefinition;
use BplanBase\CodeGenerator\Enums\Generator;
use BplanBase\CodeGenerator\Enums\ModelType;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\CSharpFileGenerator;
use Illuminate\Support\Str;


/**
 * Generator Class
 *
 * @version     3.1.0 / 2024-12-28
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class JsonApiSerializerGenerator extends CSharpFileGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Der Name der obersten Verzeichnisebene nach dem BasePath (z.B. "app").
     *
     * @var     string $_baseDir
     */
    protected $_baseDir = 'CSharp';


    /**
     * @var     boolean $_buildOnlyMode
     */
    protected $_buildOnlyMode = true;


    /**
     * Der relative Pfad zum Zielverzeichnis, ausgehend vom Wurzelverzeichnis des Projekts
     *
     * @var     string $_filePath
     */
    protected $_filePath = 'JsonApiSerializer';


    /**
     * @var     array $_idProperty
     */
    private $_idProperty;


    /**
     * @var     array $_jsonProperties
     */
    private $_jsonProperties = [];


    /**
     * @var     array $_relationFields
     */
    private $_relationFields = [];


    /**
     * @var     string $_resourceName
     */
    private $_resourceName;


    /**
     * @var     array $_technicalFields
     */
    private $_technicalFields = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_generator
     */
    protected static $_generator = Generator::CSharpJsonApiSerializer->value;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $relationType
     *
     * @param       string $relationName
     *
     * @param       string $typeName
     *
     * @param       string $name
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-11-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _addRelation(string $relationType, string $relationName, string $typeName, string $name): self
    {
        $key = strtolower($relationName);

        if (isset($this->_relationFields[$key])) {
            $key .= '-failure';
            $name .= 'Failure';
            $relationName .= '-failure';
        }
        $this->_relationFields[$key] = [
            'name' => $name,
            'relation-name' => $relationName,
            'relation-type' => $relationType,
            'type-name' => $typeName,
        ];
        return $this;

    } // _addRelation()


    /**
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2024-11-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getPreparedFileContents(): string
    {
        /*
        **  Stub-File auslesen und Ersetzungen durchführen. */
        $fileContents = file_get_contents($this->_stubPath.'/jsonapi-serializer.stub');
        /*
        **  Platzhalter-Ersetzungen vornehmen. */
        $fileContents = parent::replacePlaceholder('properties', $this->_preparePropertyReplacement(), $fileContents);
        $fileContents = parent::replacePlaceholder('resource-name', $this->_resourceName, $fileContents);
        $fileContents = parent::replacePlaceholder('relations', $this->_prepareRelationReplacement(), $fileContents);
        $fileContents = parent::replacePlaceholder('type-name', $this->_typeName, $fileContents);

        return $fileContents;

    } // _getPreparedFileContents()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-11-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _init(): self
    {
        parent::_init();

        $this->_resourceName = strtolower(Str::plural($this->_typeName));

        return $this;

    } // _init()


    /**
     * Bereitet den Ersetzungs-String für den fields-Platzhalter auf
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-11-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _preparePropertyReplacement(): string
    {
        /*
        **  Ersetzungs-String für die Relationen-Methoden zusammenstellen. */
        $fileContents = file_get_contents($this->_stubPath.'/snippets/jsonapi-serializer/property.stub');

        $fileContents = parent::replacePlaceholder('name', $this->_idProperty['name'], $fileContents);
        $fileContents = parent::replacePlaceholder('property-name', $this->_idProperty['property-name'], $fileContents);
        $fileContents = parent::replacePlaceholder('type', $this->_idProperty['type'], $fileContents);

        $replacement = $fileContents;

        if (!empty($this->_technicalFields)) {
            ksort($this->_technicalFields);

            foreach ($this->_technicalFields as $property) {
                $fileContents = file_get_contents($this->_stubPath.'/snippets/jsonapi-serializer/property.stub');

                $fileContents = parent::replacePlaceholder('name', $property['name'], $fileContents);
                $fileContents = parent::replacePlaceholder('property-name', $property['property-name'], $fileContents);
                $fileContents = parent::replacePlaceholder('type', $property['type'], $fileContents);

                $replacement .= $fileContents;
            }
        }
        if (!empty($this->_jsonProperties)) {
            ksort($this->_jsonProperties);

            foreach ($this->_jsonProperties as $property) {
                $fileContents = file_get_contents($this->_stubPath.'/snippets/jsonapi-serializer/property.stub');

                $fileContents = parent::replacePlaceholder('name', $property['name'], $fileContents);
                $fileContents = parent::replacePlaceholder('property-name', $property['property-name'], $fileContents);
                $fileContents = parent::replacePlaceholder('type', $property['type'], $fileContents);

                $replacement .= $fileContents;
            }
        }
        return $replacement;

    } // _preparePropertyReplacement()


    /**
     * Bereitet den Ersetzungs-String für den fields-Platzhalter auf
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-11-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareRelationReplacement(): string
    {
        /*
        **  Ersetzungs-String für die Relationen-Methoden zusammenstellen. */
        ksort($this->_relationFields);

        $replacement = '';

        foreach ($this->_relationFields as $relation) {
            if ($relation['relation-type'] === 'BelongsTo') {
                $fileContents = file_get_contents($this->_stubPath.'/snippets/jsonapi-serializer/relation.stub');
            } else {
                $fileContents = file_get_contents($this->_stubPath.'/snippets/jsonapi-serializer/relation-list.stub');
            }
            $fileContents = parent::replacePlaceholder('relation-name', $relation['relation-name'], $fileContents);
            $fileContents = parent::replacePlaceholder('name', $relation['name'], $fileContents);
            $fileContents = parent::replacePlaceholder('type-name', $relation['type-name'], $fileContents);

            $replacement .= $fileContents;
        }
        return $replacement;

    } // _prepareRelationReplacement()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-11-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _process(): self
    {
        $this->_processTypeDefinition();

        $this->_writeFile();

        return $this;

    } // _process()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-11-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processBackReferences(): self
    {
        $backReferences = CodeGenerator::getBackReferences($this->_tableName);

        if ($backReferences !== null) {
            /*
            **  HasMany- und BelongsToMany-Relationen aufbereiten. */
            foreach ($backReferences as $BackReference) {
                $columnName = $BackReference->getColumnName();
                $relatedTable = $BackReference->getTableName();

                $RelatedTypeDefinition = CodeGenerator::getTypeDefinition($relatedTable);

                $pivotTables = $RelatedTypeDefinition->getPivotTables();
                /*
                **  ForeignId-Felder haben nicht zwingend eine Constraint (siehe users.current_team_id). Aus dem
                **  Grund muss hier überprüft werden ob es eine entsprechende Definition gibt. */
                if ($BackReference->getModelType() === ModelType::Pivot->name && isset($pivotTables[$this->_tableName])) {
                    $name = Str::studly($pivotTables[$this->_tableName]);
                    $relationName = Str::slug($pivotTables[$this->_tableName]);
                    $typeName = Str::studly(Str::singular($pivotTables[$this->_tableName]));
                    $relationType = 'BelongsToMany';

                } else {
                    $name = $BackReference->getTableName('studly', singular: false);
                    $relationName = $BackReference->getTableName('slug', singular: false);
                    $typeName = $BackReference->getTableName('studly', singular: true);
                    $relationType = 'HasMany';

                    $foreignKey = $RelatedTypeDefinition->getForeignKey($columnName);

                    if ($foreignKey !== null && !empty($foreignKey['reverseRelationName'])) {
                        $relationName = $foreignKey['reverseRelationName'];
                        $name = Str::studly($relationName);
                    }
                }
                $this->_addRelation($relationType, $relationName, $typeName, $name);
            }
        }
        return $this;

    } // _processBackReferences()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-11-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processTypeDefinition(): self
    {
        foreach ($this->_TypeDefinition->getFieldDefinitions($this->_tableName) as $FieldDefinition) {
            if ($FieldDefinition->getAttribute('isForeignKey') === true) {
                $ForeignKeyDefinition = $FieldDefinition->getForeignKeyDefinition();

                if ($ForeignKeyDefinition === null) {
                    continue;
                }
                $relationName = $ForeignKeyDefinition->getRelationName();
                $name = $ForeignKeyDefinition->getName('studly');
                $typeName = $ForeignKeyDefinition->getTypeName();

                $this->_addRelation('BelongsTo', $relationName, $typeName, $name);

                continue;
            }
            $columnName = $FieldDefinition->getAttribute('columnName');
            $type = $FieldDefinition->getAttribute('type');

            $jsonName = $FieldDefinition->getName('camel');
            $varName = $FieldDefinition->getName('studly');

            switch ($type) {
                case 'bigInteger':
                case 'integer':
                case 'unsignedInteger':
                    $abstractType = 'int';
                    break;

                case 'boolean':
                    $abstractType = 'bool';
                    break;

                case 'datetime':
                case 'timestamp':
                    $abstractType = 'DateTime';
                    break;

                case 'enum':
                case 'geometry':
                    $abstractType = 'string';
                    break;

                case 'json':
                    $abstractType = 'Dictionary<string,string>';
                    break;

                default:
                    $abstractType = $type;
            }
            if ($columnName === 'id') {
                $abstractType = 'string';
                $varName = 'Id';

                $this->_idProperty = [
                    'name' => $varName,
                    'property-name' => $jsonName,
                    'type' => $abstractType,
                ];
                continue;

            }
            if (FieldDefinition::isTechnicalField($columnName)) {
                $this->_technicalFields[$jsonName] = [
                    'name' => $varName,
                    'property-name' => $jsonName,
                    'type' => $abstractType,
                ];
                continue;
            }
            $this->_jsonProperties[$jsonName] = [
                'name' => $varName,
                'property-name' => $jsonName,
                'type' => $abstractType,
            ];
        }
        $this->_processBackReferences();

        return $this;

    } // _processTypeDefinition()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-11-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _writeFile(): self
    {
        $fileContents = $this->_getPreparedFileContents();
        /*
        **  Klasse schreiben. */
        parent::_writeFileContents($this->_typeName.'.cs', $fileContents);

        return $this;

    } // _writeFile()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class JsonApiSerializerGenerator extends CSharpFileGenerator {}
