<?php
/**
 * Code Generator Field Element Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Elements
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Elements;


use BplanBase\CodeGenerator\Enums\CaseStyle;
use BplanBase\CodeGenerator\Enums\Number;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\Helpers\StringHelper;
use BplanBase\CodeGenerator\Traits\HasAttributes;
use Illuminate\Support\Str;


/**
 * Code Generator Field Element Class
 *
 * @version     2.0.0 / 2025-03-05
 * @history     Definitions\FieldDefinition. 1.0.0 / 2025-02-27
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Field
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use HasAttributes;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         array $_attributes<string, mixed>
     *
     * @version     1.0.0 / 2025-03-04
     */
    protected $_attributes = [
        'abstractType' => null,
        'autoIncrement' => false,
        'cast' => null,
        'columnName' => null,
        'columns' => null,
        'comment' => null,
        'default' => false,
        'filterable' => null,
        'guarded' => false,
        'hidden' => false,
        'index' => null,
        'isForeignKey' => false,
        'isTechnicalField' => false,
        'length' => null,
        'metaType' => null,
        'notSparseField' => false,
        'nullable' => false,
        'precision' => null,
        //'readOnly' => false,
        'readOnly' => null,
        'references' => null,
        'relatedTable' => null,
        // 'relatedResourceName' => '', // relatedTable -> slug
        'relationName' => null,
        'relationType' => null,
        'reverseRelationName' => null,
        'rules' => null,
        'ruleType' => null,
        //'sortable' => false,
        'sortable' => null,
        'unsigned' => false,
    ];


    /**
     * @var     string $_baseName
     */
    private $_baseName;


    /**
     * @var     Package $_Package
     */
    private $_Package;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Technische Felder sind automatisch immer sortable und (abgesehen von "active") auch
     * immer readOnly
     *
     * @var         array $_technicalFieldNames
     *
     * @version     1.0.0 / 2025-03-04
     */
    private static $_technicalFieldNames = [
        'active',
        'created_at',
        'deleted_at',
        'updated_at',
        'uuid',
    ];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Package $Package
     *
     * @param       array $columnDefinition
     *
     * @param       array|null $additionalSettings
     *
     * @param       array|null $foreignKeyAttributes
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(Package $Package, array $columnDefinition, array|null $additionalSettings, array|null $foreignKeyAttributes)
    {
        $this->_Package = $Package;

        $this->_init($columnDefinition, $additionalSettings, $foreignKeyAttributes);

    } // __construct()


    /**
     *
     * @param       string $columnName
     *
     * @param       string $type
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _evaluateType(string $columnName, string $type): array
    {
        switch ($type) {
            case 'bigInteger':
                $abstractType = 'integer';
                $metaType = 'numeric';
                break;

            case 'boolean':
                $abstractType = 'bool';
                $metaType = 'boolean';
                break;

            case 'enum':
            case 'geometry':
            case 'json':
                $abstractType = 'string';
                $metaType = 'character';
                break;

            case 'timestamp':
                $abstractType = 'datetime';
                $metaType = 'datetime';
                break;

            default:
                $abstractType = $type;
                $metaType = 'undefined';
        }
        if ($columnName === 'id') {
            $abstractType = 'string';
            $metaType = 'numeric';
        }
        $this->_attributes['abstractType'] = $abstractType;
        $this->_attributes['metaType'] = $metaType;

        return $this->_attributes;

    } // _evaluateType()


    /**
     *
     * @param       array $columnDefinition
     *
     * @param       array|null $additionalSettings
     *
     * @param       array|null $foreignKeyAttributes
     *
     * @return      $this
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(array $columnDefinition, array|null $additionalSettings, array|null $foreignKeyAttributes): self
    {
        $columnName = $columnDefinition['name'];

        $columnDefinition['name'] = $columnDefinition['isForeignKey']
            ? Str::camel(Str::replaceLast('_id', '', $columnName))
            : Str::camel($columnName);

        $this->_baseName = $this->_Package->prepareBaseName($columnDefinition['name']);

        $type = $columnDefinition['type'];

        if (in_array($columnName, self::$_technicalFieldNames) && $columnName !== 'active') {
            $columnDefinition['readOnly'] = $columnDefinition['readOnly'] ?? true;
            $columnDefinition['sortable'] = $columnDefinition['sortable'] ?? true;

        } elseif ($columnName === 'id') {
            // /*
            // **  Die ID ist automatisch immer sortierbar. Wenn sie das nicht sein soll, dann muss das über "notSortable"
            // **  gesteuert werden. */
            // if (isset($columnDefinition['sortable']) && $columnDefinition['sortable'] === false) {
            //     $columnDefinition['notSortable'] = true;
            // } else {
            //     $columnDefinition['notSortable'] = false;
            // }
        }
        $this->_attributes = array_merge(
            $this->_evaluateType($columnName, $type),
            $columnDefinition,
            $additionalSettings ?? []
        );
        $this->_attributes['columnName'] = $columnName;
        $this->_attributes['isTechnicalField'] = in_array($columnName, self::$_technicalFieldNames);

        if ($foreignKeyAttributes !== null) {
            $this->_attributes['columns'] = $foreignKeyAttributes['columns'];
            $this->_attributes['index'] = $foreignKeyAttributes['index'];
            $this->_attributes['isForeignKey'] = true;
            $this->_attributes['relatedTable'] = $foreignKeyAttributes['on'];
            $this->_attributes['relationType'] = $foreignKeyAttributes['relationType'];
            $this->_attributes['references'] = $foreignKeyAttributes['references'];

            if ($this->_attributes['relationName'] === null) {
                $this->_attributes['relationName'] = $this->_baseName;
            }
        }
        ksort($this->_attributes);

        return $this;

    } // _init()


    /**
     *
     * @param       CaseStyle $CaseStyle
     *
     * @param       Number $Number
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getBaseName(CaseStyle $CaseStyle, Number $Number)
    {
        return StringHelper::reformat($this->_baseName, $CaseStyle, $Number);

    } // getBaseName()


    /**
     *
     * @return      Package
     *
     * @version     1.0.0 / 2025-03-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPackage(): Package
    {
        return $this->_Package;

    } // getPackage()


    /**
     *
     * @return      null|Type
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRelatedType(): null|Type
    {
        return CodeGenerator::getType($this->_attributes['relatedTable']);

    } // getRelatedType()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getReverseRelationName(): string
    {
        return $this->_attributes['reverseRelationName'];

    } // getReverseRelationName()


    /**
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function isForeignKey(): bool
    {
        return $this->_attributes['isForeignKey'];

    } // isForeignKey()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $name
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function isTechnicalField(string $name): bool
    {
        return in_array($name, self::$_technicalFieldNames);

    } // isTechnicalField()


} // class Field {}
