<?php
/**
 * Code Generator Type Element Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Elements
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Elements;


use BplanBase\CodeGenerator\Elements\Field;
use BplanBase\CodeGenerator\Enums\CaseStyle;
use BplanBase\CodeGenerator\Enums\Generator;
use BplanBase\CodeGenerator\Enums\ModelType;
use BplanBase\CodeGenerator\Enums\Number;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\Helpers\StringHelper;
use BplanBase\CodeGenerator\Traits\HasAttributes;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Schema\ForeignKeyDefinition;
use Illuminate\Support\Fluent;


/**
 * Code Generator Type Element Class
 *
 * @version     1.0.0 / 2025-03-04
 * @history     Definitions\TypeDefinition, 9.0.0 / 2025-02-15
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Type
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use HasAttributes;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_additionalRelations
     */
    private $_additionalRelations;


    /**
     * @var     array $_appends
     */
    private $_appends;


    /**
     * @var     array $_attributes<string, mixed>
     */
    protected $_attributes = [
        'tableName' => null,
        'tablePrefix' => null,
        'typeName' => null,
    ];


    /**
     * @var     string $_baseName
     */
    private $_baseName;


    /**
     * @var     array $_columns
     */
    private $_columns = [];


    /**
     * @var     array $_excludes
     */
    private $_excludes = [];


    /**
     * @var     array $_fields
     */
    private $_fields = [];


    /**
     * @var     array $_foreignKeyAttributes
     */
    private $_foreignKeyAttributes = [];


    /**
     * @var     array $_foreignKeys
     */
    private $_foreignKeys = [];


    /**
     * @var     array $_indexQueries
     */
    private $_indexQueries = [];


    /**
     * @var     array $_jsonApiServers
     */
    private $_jsonApiServers;


    /**
     *
     * @var     string $_modelType
     */
    private $_modelType = ModelType::Model->name;


    /**
     * @var     Package $_Package
     */
    private $_Package;


    /**
     * @var     array $_pivotTables
     */
    private $_pivotTables = [];


    /**
     * @var     array $_routing
     */
    private $_routing;


    /**
     * @var     array $_traits
     */
    private $_traits = [];


    /**
     * @var     array $_uses
     */
    private $_uses = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_possibleExcludes
     */
    private static $_possibleExcludes = [
        Generator::LaravelActionCreate->value,
        Generator::LaravelActionDelete->value,
        Generator::LaravelActionUpdate->value,
    ];


    /**
     * @var     array $_fieldStorage
     */
    private static $_fieldStorage = [];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Package $Package
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(Package $Package, Blueprint $Blueprint, array $additionalSettings = [])
    {
        $this->_jsonApiServers = config('code-generator.jsonapi-servers');
        $this->_Package = $Package;

        $this->_process($Blueprint, $additionalSettings);

    } // __construct()


    /**
     *
     * @param       array|null $additionalColumnSettings
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _createFields(array|null $additionalColumnSettings): self
    {
        $tableName = $this->_attributes['tableName'];

        self::$_fieldStorage[$tableName] = [];

        foreach ($this->_columns as $columnName => $columnDefinition) {
            $Field = new Field($this->_Package, $columnDefinition, $additionalColumnSettings[$columnName] ?? null, $this->_foreignKeyAttributes[$columnName] ?? null);

            $this->_fields[$columnName] = $Field;

            self::$_fieldStorage[$tableName][$columnName] = $Field;
        }
        return $this;

    } // _createFields()


    /**
     *
     * @param       array $excludes
     *
     * @param       array & $additionalSettings
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2025-02-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareExcludes(array $excludes, array & $additionalSettings): self
    {
        foreach ($excludes as $exclude) {
            if (!in_array($exclude, self::$_possibleExcludes)) {
                continue;
            }
            $this->_excludes[$exclude] = $exclude;
        }
        if (isset($this->_excludes[Generator::LaravelActionCreate->value])
         && isset($this->_excludes[Generator::LaravelActionDelete->value])
         && isset($this->_excludes[Generator::LaravelActionUpdate->value])) {
            $this->_excludes[Generator::LaravelService->value] = Generator::LaravelService->value;

            if (!isset($additionalSettings['routing'])) {
                $additionalSettings['routing'] = [];
            }
            foreach ($this->_jsonApiServers as $server => $status) {
                if ($status === false) {
                    continue;
                }
                $server = strtolower($server);

                $additionalSettings['routing'][$server] = 'readOnly';
            }
        } elseif (isset($this->_excludes[Generator::LaravelActionCreate->value])
          || isset($this->_excludes[Generator::LaravelActionDelete->value])
          || isset($this->_excludes[Generator::LaravelActionUpdate->value])) {
            $actions = [];

            if (!isset($additionalSettings['routing'])) {
                $additionalSettings['routing'] = [];
            }
            foreach ($this->_jsonApiServers as $server => $status) {
                $server = strtolower($server);
                /*
                **  Wenn eh nur die Lese- oder Relationen-Routen implementiert werden sollen, dann kann
                **  der aktuelle Schleifendurchlauf gleich abgebrochen werden. */
                if (isset($additionalSettings['routing'][$server])
                  && ($additionalSettings['routing'][$server] === false
                   || $additionalSettings['routing'][$server] === 'readOnly'
                   || $additionalSettings['routing'][$server] === 'relationsOnly')) {
                    continue;
                }
                if (!isset($additionalSettings['routing'][$server])) {
                    $additionalSettings['routing'][$server] = [
                        'actions' => [],
                        'method' => 'except',
                    ];
                } else {
                    if (!isset($additionalSettings['routing'][$server]['actions'])) {
                        $additionalSettings['routing'][$server]['actions'] = [];
                    }
                    if (!isset($additionalSettings['routing'][$server]['method'])) {
                        $additionalSettings['routing'][$server]['method'] = 'except';

                    } elseif ($additionalSettings['routing'][$server]['method'] !== 'except') {
                        $additionalSettings['routing'][$server]['actions'] = [];
                        $additionalSettings['routing'][$server]['method'] = 'except';
                    }
                }
                foreach (self::$_possibleExcludes as $exclude) {
                    if (!isset($this->_excludes[$exclude])) {
                        continue;
                    }
                    switch ($exclude) {
                        case Generator::LaravelActionCreate->value:
                            $additionalSettings['routing'][$server]['actions'][] = 'store';
                            break;

                        case Generator::LaravelActionDelete->value:
                            $additionalSettings['routing'][$server]['actions'][] = 'destroy';
                            break;

                        case Generator::LaravelActionUpdate->value:
                            $additionalSettings['routing'][$server]['actions'][] = 'update';
                            break;
                    }
                }
            }
        }
        return $this;

    } // _prepareExcludes()


    /**
     *
     * @param       array $indexQueries
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2025-02-15
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareIndexQueries(array $indexQueries): self
    {
        foreach ($this->_jsonApiServers as $server => $status) {
            $lowerServer = strtolower($server);

            if (isset($indexQueries[$lowerServer])) {
                $this->_jsonApiServers[$server] = true;

                $this->_indexQueries[$lowerServer] = $indexQueries[$lowerServer];
            }
            /*
            **  Die Einstellung für alle Server überschreibt immer die für einzelne Server.
            **
            **  Eigentlich würde man eine spezifischere Definition gegenüber einer allgemeinen Definition bevorzugen,
            **  allerdings macht die Verwendung von "ALL_SERVERS" in Kombination mit der spezifischen Angabe eines
            **  bestimmten Servers keinen Sinn. Deshalb wird hier die allgemeine Angabe der spezifischeren vorgezogen. */
            if (isset($indexQueries[CodeGenerator::ALL_SERVERS_KEY])) {
                $this->_jsonApiServers[$server] = true;

                $this->_indexQueries[$lowerServer] = $indexQueries[CodeGenerator::ALL_SERVERS_KEY];
            }
        }
        return $this;

    } // _prepareIndexQueries()


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return      $this
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _process(Blueprint $Blueprint, array $additionalSettings = []): self
    {
        $tableName = $Blueprint->getTable();
        $tablePrefix = $Blueprint->getPrefix();

        $this->_baseName = $this->_Package->prepareBaseName($tableName);
        //  todo    Prüfen, wofür der TypeName noch benötigt wird. Eventuell können darüber die Namen der Laravel Core-Klassen "eingefroren" werden.
        if (isset($additionalSettings['typeName'])) {
            $typeName = $additionalSettings['typeName'];
        } else {
            $typeName = $tableName;
        }
        $this->_attributes['tableName'] = $tableName;
        $this->_attributes['tablePrefix'] = $tablePrefix;
        $this->_attributes['typeName'] = $typeName;

        if (isset($additionalSettings['pivotTables'])) {
            $this->_pivotTables = array_combine($additionalSettings['pivotTables'], array_reverse($additionalSettings['pivotTables']));

            $this->_modelType = ModelType::Pivot->name;
        }
        if (isset($additionalSettings['exclude'])) {
            $this->_prepareExcludes($additionalSettings['exclude'], $additionalSettings);
        }
        if (isset($additionalSettings['indexQueries'])) {
            $this->_prepareIndexQueries($additionalSettings['indexQueries']);
        }
        $this->_additionalRelations = $additionalSettings['additionalRelations'] ?? [];
        $this->_routing = $additionalSettings['routing'] ?? null;
        $this->_traits = $additionalSettings['traits'] ?? null;
        $this->_uses = $additionalSettings['uses'] ?? null;

        $this->_processAppends($additionalSettings['appends'] ?? null);
        $this->_processColumns($Blueprint->getColumns());
        /*
        **  Commands abarbeiten und FeldDefinitionen ergänzen. */
        $this->_processCommands($Blueprint->getCommands(), $additionalSettings['columns'] ?? null);
        /*
        **  Field-Objekte erzeugen. */
        $this->_createFields($additionalSettings['columns'] ?? null);

        return $this;

    } // _process()


    /**
     * Bereitet die Werte der appends-Option auf
     *
     * @param       array|null $appends
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processAppends(array|null $appends): self
    {
        $this->_appends = [];

        if ($appends === null) {
            return $this;
        }
        /*
        **  [
        **      'label' => [
        **          'generate' => CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_MODEL_ARRAY + CodeGenerator::APPENDS_MODEL_METHOD,
        **          'return' => '$this->last_name.\', \'.$this->first_name',
        **      ]
        **  ]
        */
        foreach ($appends as $key => $value) {
            $field = $key;

            if (is_array($value)) {
                $generate = isset($value['generate'])
                    ? (int) $value['generate']
                    : CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_LARAVEL_MODEL_ARRAY + CodeGenerator::APPENDS_LARAVEL_MODEL_METHOD;

                $return = isset($value['return']) ? $value['return'] : 'null;';

            } else {
                if (is_numeric($value)) {
                    /*
                    **  [
                    **      'profile_photo_url' => CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_MODEL_ARRAY,
                    **  ]
                    */
                    $generate = $value;

                } else {
                    $generate = CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_LARAVEL_MODEL_ARRAY + CodeGenerator::APPENDS_LARAVEL_MODEL_METHOD;
                }
                $return = 'null;';
            }
            $generators = [];

            if ($generate & CodeGenerator::APPENDS_JSONAPI_SCHEMA) {
                $generators[Generator::JsonApiSchema->value] = true;
            }
            if ($generate & CodeGenerator::APPENDS_LARAVEL_MODEL_ARRAY) {
                $generators[Generator::LaravelModel->value.'.array'] = true;
            }
            if ($generate & CodeGenerator::APPENDS_LARAVEL_MODEL_METHOD) {
                $generators[Generator::LaravelModel->value.'.method'] = true;
            }
            if (substr($return, -1) !== ';') {
                $return .= ';';
            }
            $this->_appends[$field] = [
                'field' => $field,
                'generate' => $generate,
                'generators' => $generators,
                'return' => $return,
            ];
        }
        return $this;

    } // _processAppends()


    /**
     * Verarbeitet alle Spalten und erstellt einen assoziativen Array mit den Spaltennamen als Schlüssel
     *
     * @param       array $columns
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processColumns(array $columns): self
    {
        foreach ($columns as $Column) {
            $column = $Column->toArray();
            $columnName = $column['name'];

            $column['isForeignKey'] = false;

            ksort($column);

            $this->_columns[$columnName] = $column;
        }
        return $this;

    } // _processColumns()


    /**
     * Verarbeitet Commands und ergänzt die Column-Definitionen in $_fields
     *
     * In den Commands des Blueprint-Objekts sind unter anderem die ForeignKey-Definitionen enthalten.
     *
     * @param       array $commands
     *
     * @param       array|null $additionalColumnSetting
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processCommands(array $commands, array|null $additionalColumnSetting): self
    {
        foreach ($commands as $Command) {
            $attributes = $Command->getAttributes();

            if ($Command instanceof ForeignKeyDefinition) {
                $columnName = $attributes['columns'][0];

                $attributes['relationType'] = 'BelongsTo';

                ksort($attributes);

                $this->_columns[$columnName]['isForeignKey'] = true;
                $this->_columns[$columnName]['relationName'] = null;
                $this->_columns[$columnName]['reverseRelationName'] = null;

                $this->_foreignKeyAttributes[$columnName] = $attributes;

                $this->_foreignKeys[$columnName] = $this->_columns[$columnName];

                if (isset($additionalColumnSetting[$columnName]['relationName'])) {
                    $this->_foreignKeys[$columnName]['relationName'] = $additionalColumnSetting[$columnName]['relationName'];
                }
                if (isset($additionalColumnSetting[$columnName]['reverseRelationName'])) {
                    $this->_foreignKeys[$columnName]['reverseRelationName'] = $additionalColumnSetting[$columnName]['reverseRelationName'];
                }
                ksort($this->_foreignKeys[$columnName]);

                continue;

            } elseif ($Command instanceof Fluent) {
                if ($attributes['name'] === 'create') {
                    continue;
                }
                /*
                    Index-Informationen werden aktuell noch nicht berücksichtigt, könnten aber insbesondere
                    für die Rules relevant werden um zum Beispiel eine unique-Rule zu erzeugen.

                */
                //self::$_collectedIndexes[$this->_attributes['tableName']][] = $attributes;
            }
        }
        return $this;

    } // _processCommands()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getAdditionalRelations(): array
    {
        return $this->_additionalRelations;

    } // getAdditionalRelations()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-12-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getAppends(): array
    {
        return $this->_appends;

    } // getAppends()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getBaseName(CaseStyle $CaseStyle, Number $Number)
    {
        return StringHelper::reformat($this->_baseName, $CaseStyle, $Number);

    } // getBaseName()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFields(string $tableName): array
    {
        return self::$_fieldStorage[$tableName];

    } // getFields()


    /**
     *
     * @param       string $columnName
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getForeignKey(string $columnName): array|null
    {
        if (!isset($this->_foreignKeys[$columnName])) {
            return null;
        }
        return $this->_foreignKeys[$columnName];

    } // getForeignKey()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getForeignKeys(): array
    {
        return $this->_foreignKeys;

    } // getForeignKeys()


    /**
     *
     * @return 	    null|string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getIndexQuery(string $server): null|string
    {
        $server = strtolower($server);

        return $this->_indexQueries[$server] ?? null;

    } // getIndexQuery()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getJsonApiServers(): array
    {
        return $this->_jsonApiServers;

    } // getJsonApiServers()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPivotTables(): array
    {
        return $this->_pivotTables;

    } // getPivotTables()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getModelType(): string
    {
        return $this->_modelType;

    } // getModelType()


    /**
     *
     * @return 	    Package
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPackage(): Package
    {
        return $this->_Package;

    } // getPackage()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRouting(): array
    {
        return $this->_routing;

    } // getRouting()


    /**
     *
     * @param       string $traitsFor
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getTraits(string $traitsFor): array|null
    {
        if (empty($this->_traits[$traitsFor])) {
            return null;
        }
        return $this->_traits[$traitsFor];

    } // getTraits()


    /**
     *
     * @param       string $usesFor
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getUses(string $usesFor): array|null
    {
        if (empty($this->_uses[$usesFor])) {
            return null;
        }
        return $this->_uses[$usesFor];

    } // getUses()


    /**
     *
     * @param       string $name
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasColumn(string $name): bool
    {
        return isset($this->_fields[$name]);

    } // hasColumn()


    /**
     *
     * @param       Generator|string $generator
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasExclude(Generator|string $generator): bool
    {
        if ($generator instanceof Generator) {
            $generator = $generator->value;
        }
        return isset($this->_excludes[$generator]);

    } // hasExclude()


    /**
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function isForPackage(): bool
    {
        return $this->_Package->isForPackage();

    } // isForPackage()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function writeCacheFile(): self
    {
        $cachePath = CodeGenerator::getCachePath();

        ksort($this->_fields);

        unset($this->_columns);

        file_put_contents($cachePath.'/'.$this->_attributes['tableName'].'.php', '<?php'."\n\n".'return '.var_export($this, true).';'."\n");

        return $this;

    } // writeCacheFile()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class Type {}
