<?php
/**
 * Code Generator Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Generators;


use BplanBase\CodeGenerator\Elements\BackReference;
use BplanBase\CodeGenerator\Elements\Package;
use BplanBase\CodeGenerator\Elements\Type;
use BplanBase\CodeGenerator\Enums\CaseStyle;
use BplanBase\CodeGenerator\Enums\ModelType;
use BplanBase\CodeGenerator\Enums\Number;
use BplanBase\CodeGenerator\Generators\CodeGenerator\BaseGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\CSharpFileGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\JsonApiFileGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\JsonApiGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\LaravelFileGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\LaravelGenerator;
use BplanBase\CodeGenerator\Generators\CodeGenerator\WinMergeGenerator;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Schema\ForeignKeyDefinition;
use Symfony\Component\Console\Output\ConsoleOutput;


/**
 * Code Generator Class
 *
 * @version     11.1.0 / 2025-03-05
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class CodeGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string ALL_SERVERS_KEY
     */
    const ALL_SERVERS_KEY = '*';


    /**
     * @var     int APPENDS_JSONAPI_SCHEMA
     */
    const APPENDS_JSONAPI_SCHEMA       = 1;


    /**
     * @var     int APPENDS_MODEL_ARRAY
     */
    const APPENDS_LARAVEL_MODEL_ARRAY  = 2;


    /**
     * @var     int APPENDS_MODEL_METHOD
     */
    const APPENDS_LARAVEL_MODEL_METHOD = 4;


    /**
     * @var     string CODE_GENERATOR_TAG
     */
    const CODE_GENERATOR_TAG = '#CodeGenerator';


    /**
     * Wenn der Wert eines Json-Feldes ein Array ist
     *
     * Beispiele:
     *  ["de", "en"]
     *  [{"de":"deutsch"}, {"en":"englisch"}]
     *
     * @var     string JSON_TYPE_ARRAY
     */
    const JSON_TYPE_ARRAY = 'List';


    /**
     * Wenn der Wert eines Json-Feldes ein Objekt ist
     *
     * Beispiel:
     *  {"de":"deutsch", "en":"englisch"}
     *
     * @var     string JSON_TYPE_OBJECT
     */
    const JSON_TYPE_OBJECT = 'Hash';


    /**
     * @var     string PROJECT_KEY_NAME
     */
    const PROJECT_KEY_NAME = '.project';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_currentBasePath
     */
    private $_currentBasePath;


    /**
     * @var         string $_modelType
     */
    private $_modelType;


    /**
     * @var         array $_packages
     */
    private $_packages = [];


    /**
     * @var         string $_tableName
     */
    private $_tableName;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         array $_backReferences
     */
    private static $_backReferences = [];


    /**
     * @var         string $_cachePath
     */
    private static $_cachePath;


    /**
     * @var         array $_collectedForeignKeys
     */
    private static $_collectedForeignKeys = [];


    /**
     * @var         array $_collectedIndexes
     */
    private static $_collectedIndexes = [];


    /**
     * @var         array $_collectedTypes
     */
    private static $_collectedTypes = [];


    /**
     * @var         int $_defaultOutputIndent
     */
    private static $_defaultOutputIndent = 2;


    /**
     * @var         int $_errorGeneratorsMaxLength
     */
    private static $_errorGeneratorsMaxLength = 0;


    /**
     * @var         int $_errorTypeNameMaxLength
     */
    private static $_errorTypeNameMaxLength = 0;


    /**
     * @var         array $_errors
     */
    private static $_errors = [];


    /**
     * @var         array $_messages
     */
    private static $_messages = [];


    /**
     * Farbedefinitionen für die Ausgabe
     *
     * Mögliche Farbbezeichnungen:
     *  black
     *  blue, bright-blue,
     *  cyan, bright-cyan
     *  gray
     *  green, bright-green
     *  magenta, bright-magenta
     *  red, bright-red
     *  white, bright-white
     *  yellow, bright-yellow
     *
     * @var         array $_outputColors
     * @version     2.0.0 / 2024-12-28
     */
    private static $_outputColors = [
        'creating' => 'bright-green',
        'ignoring' => null,
        'omitting' => 'bright-blue',
        'protecting' => 'bright-red',
        'replacing' => 'bright-yellow',
    ];


    /**
     * @var         array $_packageSettingsCache
     */
    private static $_packageSettingsCache = [];


    /**
     * @var         string $_projectBasePath
     */
    private static $_projectBasePath;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(Blueprint $Blueprint, array $additionalSettings = [])
    {
        if (config('code-generator.active') !== true) {
            return;
        }
        $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 2);

        $this->_currentBasePath = str_replace('\\', '/', realpath(dirname($backtrace[1]['file']).'/../..'));

        if (self::$_projectBasePath === null) {
            /*
            **  Es wird davon ausgegangen, dass die erste Instanz des CodeGenerators in einer
            **  Migration des Projekts - und nicht in der eines Pakets - erzeugt wird. Deshalb
            **  wird aus dem Pfad dieser Migration der BasePath für das Projekt ermittelt.
            **
            **  @todo   Prüfen ob nicht eventuell aus dem CWD der in jedem Fall richtige
            **          Projektpfad ermittelt werden kann. */
            self::$_projectBasePath = $this->_currentBasePath;
        }
        $this->_init($Blueprint, $additionalSettings);

    } // __construct()


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return 	    $this
     *
     * @version     1.6.0 / 2025-03-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(Blueprint $Blueprint, array $additionalSettings): self
    {
        if (self::$_projectBasePath === $this->_currentBasePath) {
            $additionalSettings['package'] = Package::getPackageConfig($this->_currentBasePath, isProject: true);

        } else {
            /*
            **  Es wird davon ausgegangen, dass es sich um die Code-Erzeugung für ein Paket handelt
            **  wenn sich der Projekt-Basispfad vom aktuellen Basispfad unterscheidet. */
            $additionalSettings['package'] = Package::getPackageConfig($this->_currentBasePath, isProject: false);
        }
        $this->_tableName = $Blueprint->getTable();

        if (isset($additionalSettings['pivotTables'])) {
            $this->_modelType = ModelType::Pivot->name;
        } else {
            $this->_modelType = ModelType::Model->name;
        }
        $Package = $this->_getPackage($additionalSettings['package']);

        $Type = new Type($Package, $Blueprint, $additionalSettings);

        $this->_processCommands($Blueprint->getCommands());

        self::$_collectedTypes[$this->_tableName] = $Type;

        $Type->writeCacheFile();

        return $this;

    } // _init()


    /**
     * Verarbeitet Commands und speichert die ermittelten Informationen in statischen Variablen
     *
     * In den Commands des Blueprint-Objekts sind unter anderem die ForeignKey-Definitionen enthalten.
     *
     * @param       array $commands
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processCommands(array $commands): self
    {
        if (!isset(self::$_collectedForeignKeys[$this->_tableName])) {
            self::$_collectedForeignKeys[$this->_tableName] = [];
            // self::$_collectedIndexes[$this->_tableName] = [];
        }
        foreach ($commands as $Command) {
            $attributes = $Command->getAttributes();

            if ($Command instanceof ForeignKeyDefinition) {
                $column = $attributes['columns'][0];
                $relatedTable = $attributes['on'];

                self::$_collectedForeignKeys[$this->_tableName][$column] = $attributes;

                if (!isset(self::$_backReferences[$relatedTable])) {
                    self::$_backReferences[$relatedTable] = [];
                }
                self::$_backReferences[$relatedTable][] = new BackReference($this->_modelType, $this->_tableName, $column);

                continue;
            }
        }
        return $this;

    } // _processCommands()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return 	    void
     *
     * @version     1.3.0 / 2025-03-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _finish(): void
    {
        $resultCounters = BaseGenerator::getResultCounters();

        $total = array_sum($resultCounters);
        $totalLength = strlen($total);
        $pad = 45;

        $createdText = 'Created Files   '.str_repeat(' ', $pad + 3).str_pad($resultCounters['created'], $totalLength, ' ', STR_PAD_LEFT);
        $protectedText = 'Protected Files '.str_repeat(' ', $pad + 3).str_pad($resultCounters['protected'], $totalLength, ' ', STR_PAD_LEFT);
        $replacedText = 'Replaced Files  '.str_repeat(' ', $pad + 3).str_pad($resultCounters['replaced'], $totalLength, ' ', STR_PAD_LEFT);

        $output = "\n"
            ."\n".'   <options=bold,underscore>Results            '.str_repeat(' ', $pad).str_repeat(' ', $totalLength).'  </>';

        $output .= "\n".'  |'.(($resultCounters['created'] > 0) ? ' <fg='.self::$_outputColors['creating'].'>'.$createdText.'</>' : ' '.$createdText).' |';
        $output .= "\n".'  |'.(($resultCounters['replaced'] > 0) ? ' <fg='.self::$_outputColors['replacing'].'>'.$replacedText.'</>' : ' '.$replacedText).' |';
        $output .= "\n".'  |'.(($resultCounters['protected'] > 0) ? ' <fg='.self::$_outputColors['protecting'].'>'.$protectedText.'</>' : ' '.$protectedText).' |';

        $output .= "\n".'  |<options=underscore> Unchanged Files    '.str_repeat(' ', $pad).str_pad($resultCounters['unchanged'], $totalLength, ' ', STR_PAD_LEFT).' </>|'
            ."\n".'  |<options=bold,underscore> Total              '.str_repeat(' ', $pad).str_pad($total, $totalLength, ' ', STR_PAD_LEFT).' </>|';

        self::output($output);
        self::output('<fg=green;options=bold>Done</>', appendNewline: 1, indent: 2, prependNewline: 1);

        if (!empty(self::$_messages)) {
            foreach (self::$_messages as $message) {
                self::output($message['text'], bullet: true, color: $message['color']);
            }
            self::output('', appendNewline: 1);
        }
        if (!empty(self::$_errors)) {
            self::output('<options=bold>'.count(self::$_errors).' Error(s)</>', appendNewline: 1, prependNewline: 1);

            foreach (self::$_errors as $error) {
                self::output('<fg=red>['
                    .str_pad($error['generator'].']</> ', self::$_errorGeneratorsMaxLength + 5, ' ')
                    .'> '
                    .str_pad($error['typeName'].'', self::$_errorTypeNameMaxLength, ' ').' : '
                    .$error['message'], bullet: true);
            }
        }
    } // _finish()


    /**
     *
     * @param       array $packageConfig
     *
     * @return 	    Package
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getPackage(array $packageConfig): Package
    {
        $packageName = $packageConfig['name'];

        if (isset($this->_packages[$packageName])) {
            return $this->_packages[$packageName];
        }
        return $this->_packages[$packageName] = new Package($packageConfig);

    } // _getPackage()


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2024-11-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function addError(string $generator, string $typeName, string $error)
    {
        self::$_errors[] = [
            'generator' => $generator,
            'message' => $error,
            'typeName' => $typeName,
        ];
        self::$_errorGeneratorsMaxLength = max(self::$_errorTypeNameMaxLength, strlen($generator));
        self::$_errorTypeNameMaxLength = max(self::$_errorTypeNameMaxLength, strlen($typeName));

    } // addError()


    /**
     *
     * @param       string $text
     *
     * @param       null|string $color
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-03-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function addMessage(string $text, null|string $color = null)
    {
        self::$_messages[] = [
            'color' => $color,
            'text' => $text,
        ];
    } // addMessage()


    /**
     *
     * @return 	    void
     *
     * @version     1.12.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function generate(): void
    {
        $creates = config('code-generator.creates');

        self::output("\n\n".'<info>  CodeGenerator</info> Starting code generation - TestMode '.(config('code-generator.test-mode') == true ? '<info> on </info>' : '<error> off </error>'), appendNewline: true);

        ksort(self::$_collectedTypes);

        if ($creates['Laravel'] === true) {
            self::output('Starting global code generation for <options=bold,underscore>Laravel</>', appendNewline: 1, prependNewline: 2);

            LaravelGenerator::process(self::$_collectedTypes);
        }
        foreach (self::$_collectedTypes as $tableName => $Type) {
            $typeName = $Type->getBaseName(CaseStyle::Studly, Number::Singular);

            self::output('Starting code generation for resource <options=bold,underscore>'.$typeName.'</>', appendNewline: 1, prependNewline: 2);

            foreach ($creates as $create => $status) {
                if ($status === false) {
                    continue;
                }
                switch ($create) {
                    case 'CSharp';
                        CSharpFileGenerator::process($Type);
                        break;

                    case 'JsonApi';
                        JsonApiFileGenerator::process($Type);
                        break;

                    case 'Laravel';
                        LaravelFileGenerator::process($Type);
                        break;
                }
            }
        }
        if ($creates['JsonApi'] === true) {
            self::output('Starting global code generation for <options=bold,underscore>JsonApi</>', appendNewline: 1, prependNewline: 2);

            JsonApiGenerator::process(self::$_collectedTypes, self::$_backReferences);
        }
        self::output('Starting global code generation for <options=bold,underscore>WinMerge</>', appendNewline: 1, prependNewline: 2);

        WinMergeGenerator::process(self::$_collectedTypes, self::$_backReferences);

        self::_finish();

    } // generate()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2024-10-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBackReferences(string $tableName): array|null
    {
        if (!isset(self::$_backReferences[$tableName])) {
            return null;
        }
        return self::$_backReferences[$tableName];

    } // getBackReferences()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     1.1.0 / 2024-10-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBackupPath(null|string $path = null): string
    {
        $backupPath = str_replace('\\', '/', storage_path('app/backup/'.basename(__CLASS__)));

        if ($path === null) {
            return $backupPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $backupPath.$path;
        }
        return $backupPath.'/'.$path;

    } // getBackupPath()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     2.0.0 / 2024-11-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBasePath(null|string $path = null, null|string $package = null): string
    {
        $basePath = str_replace('\\', '/', base_path());

        if ($package !== null) {
            $basePath .= '/vendor/'.$package;
        }
        if ($path === null) {
            return $basePath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $basePath.$path;
        }
        return $basePath.'/'.$path;

    } // getBasePath()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     2.0.0 / 2024-11-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getBuildPath(null|string $path = null, null|string $package = null): string
    {
        $buildPath = str_replace('\\', '/', storage_path('app/build/'.basename(__CLASS__)));

        if ($package !== null) {
            $buildPath .= '/vendor/'.$package;
        }
        if ($path === null) {
            return $buildPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $buildPath.$path;
        }
        return $buildPath.'/'.$path;

    } // getBuildPath()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getCachePath(): string
    {
        if (self::$_cachePath !== null) {
            return self::$_cachePath;
        }
        self::$_cachePath = str_replace('\\', '/', storage_path('app/cache/'.basename(__CLASS__)));

        if (!file_exists(self::$_cachePath)) {
            mkdir(self::$_cachePath, 0777, true);
        }
        return self::$_cachePath;

    } // getCachePath()


    /**
     * Liefert den absoluten Pfad zum Wurzelverzeichnis des CodeGenerator-Pakets
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-02-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getGeneratorBasePath(): string
    {
        return str_replace('\\', '/', realpath(__DIR__.'/../..'));

    } // getGeneratorBasePath()


    /**
     *
     * @param       string $status
     *
     * @return      null|string
     *
     * @version     1.0.0 / 2024-10-14
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getOutputColor(string $status): null|string
    {
        if (!isset(self::$_outputColors[$status])) {
            return null;
        }
        return self::$_outputColors[$status];

    } // getOutputColor()


    /**
     *
     * @param       null|string $path
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getStubPath(null|string $path = null): string
    {
        $stubPath = str_replace('\\', '/', realpath(__DIR__.'/../..'))
            .'/stubs/code-generator';

        if ($path === null) {
            return $stubPath;
        }
        $path = trim(str_replace('\\', '/', $path));

        if (strpos($path, '/') === 0) {
            return $stubPath.$path;
        }
        return $stubPath.'/'.$path;

    } // getStubPath()


    /**
     *
     * @param       string $tableName
     *
     * @return      null|Type
     *
     * @version     2.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getType(string $tableName): null|Type
    {
        if (!isset(self::$_collectedTypes[$tableName])) {
            return null;
        }
        return self::$_collectedTypes[$tableName];

    } // getType()


    /**
     *
     * @param       string $message
     *
     * @param       int|null $appendNewline
     *
     * @param       bool $bullet
     *
     * @param       null|string $color
     *
     * @param       string $file
     *
     * @param       int $indent
     *
     * @param       int|null $prependNewline
     *
     * @return 	    void
     *
     * @version     2.3.0 / 2025-03-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function output(string $message, int|null $appendNewline = null, bool $bullet = false, null|string $color = null, string $file = null, int $indent = 0, int|null $prependNewline = null): void
    {
        static $Output = null;

        if ($Output === null) {
            $Output = new ConsoleOutput();
        }
        $totalIndent = $indent + self::$_defaultOutputIndent;

        $outputString = ($prependNewline !== null ? str_repeat("\n", $prependNewline) : '')
            .($indent > 0 || self::$_defaultOutputIndent ? str_repeat(' ', $indent + self::$_defaultOutputIndent) : '')
            .($bullet === true ? '• ' : '')
            .str_replace("\n", "\n".str_repeat(' ', $totalIndent + (($bullet === true) ? 2 : 0)), $message)
            .($file !== null ? ' <fg=gray>'.str_repeat('.', 30 - strlen($message)).'</> '.$file : '').''
            .($appendNewline !== null ? str_repeat("\n", $appendNewline) : '');

        if ($color !== null) {
            $outputString = '<fg='.$color.'>'.$outputString.'</>';
        }
        $Output->writeln($outputString);

    } // output()


} // class CodeGenerator {}
