<?php
/**
 * Code Generator Package Element Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Elements
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Elements;


use BplanBase\CodeGenerator\Enums\CaseStyle;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\Helpers\StringHelper;
use Illuminate\Support\Str;


/**
 * Code Generator Package Element Class
 *
 * @version     4.1.0 / 2025-05-31
 * @history     Definitions\Package, 2.0.0 / 2025-02-18
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class PackageConfig
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_config
     */
    private $_config;


    /**
     * @var     bool $_isProject
     */
    private $_isProject = false;


    /**
     * @var     string $_packagePath
     */
    private $_packagePath;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_configFileName
     */
    private static $_configFileName = 'package-code-generator.php';


    /**
     * Eine Map zum Auflösen des Projekt-/Paketnames zum jeweiligen Pfad
     *
     * @var     array $_packagePathMap
     */
    private static $_packagePathMap = [];


    /**
     * Ein Array in dem die Konfigurationen des Projekts und aller Pakete gesammelt werden
     *
     * Als Schlüssel wird der jeweilige BasePath verwendet.
     *
     * @var     array $_packageSettingsCache
     */
    private static $_packageSettingsCache = [];


    /**
     * Der Name des Projekts
     *
     * @var     string $_projectPackageName
     */
    private static $_projectPackageName;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       array $settings
     *
     * @version     1.0.0 / 2025-04-02
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(string $packagePath, bool $isForProject)
    {
        $this->_isProject = $isForProject;
        $this->_packagePath = $packagePath;

        $this->_init();

    } // __construct()


    /**
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2025-04-02
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(): self
    {
        $configFile = $this->_packagePath.'/config/'.self::$_configFileName;

        if (file_exists($configFile)) {
            $tmp = require $configFile;

            $this->_config = $tmp['package'];

        } else {
            $this->_config = self::get($this->_packagePath, $this->_isProject);
        }
        return $this;

    } // _init()


    /**
     *
     * @param       null|string $setting
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-02
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getConfig(null|string $setting = null)
    {
        if ($setting === null) {
            return $this->_config;
        }
        if (!isset($this->_config[$setting])) {
            return null;
        }
        return $this->_config[$setting];

    } // getConfig()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Erzeugt eine neue Konfigurationsdatei für das aktuelle Package/Projekt
     *
     * @param       string $basePath
     *              Der absolute Pfad zum Wurzelverzeichnis des Packages/Projekts, dessen Konfiguration
     *              gelesen werden soll.
     *
     * @param       bool $isProject
     *              Teilt der Methode mit ob der übergebene Pfad zum aktuellen Projekt oder zu einem
     *              Package gehört.
     *
     * @return 	    array
     *
     * @version     1.2.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _buildPackageConfig(string $basePath, bool $isProject = false): array
    {
        /*
        **  Die neue Konfigurationsdatei wird auf Basis der Defaults erzeugt. */
        $packageConfigDefaults = config('code-generator-defaults.package');

        if ($isProject === true) {
            /*
            **  Für das Projekt werden Daten aus der Projektkonfiguration verwendet. */
            /*
                @todo   Autoren müssen aus der CodeGenerator-Konfiguration (die eventuell über die env-Datei
                        übersteuert werden) verwendet werden.
            */
            $authors = $packageConfigDefaults['authors'];
            $copyright = $packageConfigDefaults['copyright'];
            $fullPackageName = self::_prepareProjectName(config('app.name'));
            $packageBaseDir = 'app';
            $packageName = $fullPackageName;
            $packageNamespace = 'App';
            $packagePrefix = null;
            $vendorName = null;

            self::$_projectPackageName = $fullPackageName;

        } else {
            /*
            **  Bei Paketen wird die composer.json ausgelesen um den Paketnamen und
            **  die Autoren zu ermitteln. */
            $ComposerJson = self::_readComposerJson($basePath);

            $authors = [];

            foreach ($ComposerJson->authors as $author) {
                $authors[] = $author->name.' <'.$author->email.'>';
            }
            $copyright = $ComposerJson?->extra?->copyright ?? $packageConfigDefaults['copyright'];
            $fullPackageName = $ComposerJson->name;
            $packageBaseDir = $packageConfigDefaults['baseDir'];
            /*
            **  Namespace aus dem Paketnamen erstellen. */
            list($vendorName, $packageName) = explode('/', $fullPackageName);

            $packageNamespace = Str::studly($vendorName).'\\\\'.Str::studly($packageName);
            /*
            **  Das Prefix wird auch aus dem Paketnamen ermittelt. Dazu wird der Vendor-Name überprüft.
            **  wenn er aus mehreren Teilen besteht, dann wird der letzte Teil des Namens verwendet.
            **  Ansonsten wird der vollständige Vendor-Name als Prefix genutzt. */
            if (strpos($vendorName, '-') !== false) {
                $tmp = explode('-', $vendorName);

                $packagePrefix = array_pop($tmp);

            } else {
                $packagePrefix = $vendorName;
            }
        }
        self::$_packagePathMap[$fullPackageName] = $basePath;
        /*
        **  Daten für den Dateiinhalt zusammenstellen. */
        $configFileData = [
            'authors' => $authors,
            'baseDir' => $packageBaseDir,
            'basePath' => null,
            'copyright' => $copyright,
            'devStart' => $packageConfigDefaults['devStart'],
            'isProject' => $isProject,
            'name' => $fullPackageName,
            'namespace' => $packageNamespace,
            'packageName' => $packageName,
            'prefix' => $packagePrefix,
            'usePrefix' => false,
            'vendorName' => $vendorName,
            'version' => 1.0,
        ];
        return self::_writePackageConfig($basePath, $configFileData, update: false);

    } // _buildPackageConfig()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _prepareAuthors()
    {
        $ComposerJson = self::_readComposerJson($basePath);

        $authors = [];

        foreach ($ComposerJson->authors as $author) {
            $authors[] = $author->name.' <'.$author->email.'>';
        }
        return $authors;

    } // _prepareAuthors()


    /**
     *
     * @param       string $name
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-04-02
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _prepareProjectName(string $name): string
    {
        return StringHelper::reformat(str_replace(' ', '-', trim($name)), CaseStyle::Studly);

    } // _prepareProjectName()


    /**
     *
     * @param       string $basePath
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _readComposerJson(string $basePath)
    {
        $composerJson = trim(file_get_contents($basePath.'/composer.json'));

        return json_decode($composerJson);

    } // _readComposerJson()


    /**
     * Prüft anhand verschiedener Kriterien ob die Konfigurationsdatei aktualisiert werden muss
     *
     * @param       string $basePath
     *
     * @param       array & $config
     *
     * @return 	    bool
     *
     * @version     1.1.0 / 2025-04-02
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _requireUpdate(string $basePath, array & $config): bool
    {
        if ($config['package']['isProject'] === true) {
            /*
            **  Bei Projekten wird überprüft ob sich der Projektname geändert hat. */
            $projectName = self::_prepareProjectName(config('app.name'));

            if ($config['package']['name'] !== $projectName) {
                $config['package']['name'] = $projectName;
                $config['package']['packageName'] = $projectName;

                self::$_projectPackageName = $projectName;
                self::$_packagePathMap[$projectName] = $basePath;

                return true;
            }
        }
        $defaultConfigVersion = config('code-generator-defaults.packageConfigVersion');

        if (!isset($config['packageConfigVersion']) || ($config['packageConfigVersion'] < $defaultConfigVersion)) {
            return true;
        }
        return false;

    } // _requireUpdate()


    /**
     * Aktualisiert eine Konfigurationsdatei
     *
     * @param       string $basePath
     *              Der absolute Pfad zum Wurzelverzeichnis des Packages/Projekts, dessen Konfiguration
     *              gelesen werden soll.
     *
     * @param       bool $isProject
     *              Teilt der Methode mit ob der übergebene Pfad zum aktuellen Projekt oder zu einem
     *              Package gehört.
     *
     * @param       array $currentConfig
     *
     * @return 	    array
     *
     * @version     1.1.0 / 2025-04-03
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _updatePackageConfig(string $basePath, array $currentConfig): array
    {
        $defaultConfig = config('code-generator-defaults.package');

        unset($defaultConfig['authors']);

        $updatedConfig = array_merge($defaultConfig, $currentConfig['package']);

        $updatedConfig['namespace'] = str_replace('\\', '\\\\', $updatedConfig['namespace']);

        if ($currentConfig['package']['isProject'] === false) {
            /*
            **  Bei Paketen wird die composer.json ausgelesen um den Paketnamen und
            **  die Autoren zu ermitteln. */
            $ComposerJson = self::_readComposerJson($currentConfig['package']['basePath']);

            $authors = [];

            foreach ($ComposerJson->authors as $author) {
                $authors[] = $author->name.' <'.$author->email.'>';
            }
            $updatedConfig['authors'] = $authors;
        }
        ksort($updatedConfig);

        return self::_writePackageConfig($basePath, $updatedConfig, update: true);

    } // _updatePackageConfig()


    /**
     * Schreibt eine Konfigurationsdatei
     *
     * Da nicht nur irgendwelche Werte, sondern auch PHP-Code in die Datei geschrieben werden muss, kann
     * nicht einfach mit var_export() gearbeitet werden. Stattdessen wird der Dateiinhalt Zeile für Zeile
     * zusammengestellt.
     *
     * @return 	    array
     *
     * @version     1.2.0 / 2025-05-31
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _writePackageConfig(string $basePath, array $configFileData, bool $update): array
    {
        $packageConfigDir = 'config';
        $packageConfigPath = $basePath.'/'.$packageConfigDir;
        $packageConfigFile = $packageConfigPath.'/'.self::$_configFileName;

        $configFileContents = '<?'.'php'
            ."\n\n".'return ['
            ."\n".'    \'package\' => [';

        foreach ($configFileData as $key => $value) {
            $configFileContents .= "\n".'        '."'".$key."'".' => ';

            if ($key === 'basePath') {
                $configFileContents .= "str_replace('\\\\', '/', realpath(dirname(__FILE__).'/..'))";

            } elseif ($key === 'version') {
                $configFileContents .= sprintf('%01.1f', $value);

            } elseif (is_bool($value)) {
                $configFileContents .= ($value ? 'true' : 'false');

            } elseif ($value === null) {
                $configFileContents .= 'null';

            } elseif (is_numeric($value)) {
                $configFileContents .= $value;

            } elseif (is_array($value)) {
                $configFileContents .= '[';

                foreach ($value as $k => $v) {
                    if (!is_numeric($k)) {
                        $k = "'".$k."'";
                    }
                    $configFileContents .= "\n".'            '.$k.' => '."'".$v."',";
                }
                $configFileContents .= "\n".'        ]';

            } else {
                $configFileContents .= "'".$value."'";
            }
            $configFileContents .= ',';
        }
        $configFileContents .= "\n".'    ],'
            ."\n".'    \'packageConfigVersion\' => '.config('code-generator-defaults.packageConfigVersion').','
            ."\n".'];'
            ."\n";

        if (!file_exists($packageConfigPath)) {
            mkdir($packageConfigPath, 0777, true);
        }
        /*
        ** Neue Konfigurationsdatei speichern und direkt laden. */
        file_put_contents($packageConfigFile, $configFileContents);

        $config = require $packageConfigFile;

        $config['package']['isDevPackage'] = false;

        if ($config['package']['isProject'] === false) {
            $projectConfig = self::getProjectConfig();

            $path = $projectConfig['basePath'].'/vendor/'.$config['package']['vendorName'].'/globals';
            $realPath = str_replace('\\', '/', realpath($path));
            /*
            **  Wenn der Path und der RealPath voneinander abweichen, dann bedeutet das,
            **  dass das Paket über einen Symlink als Entwicklungspaket eingebunden ist. */
            $config['package']['isDevPackage'] = ($path !== $realPath);
        }
        if ($update === true) {
            $message = 'Updated CodeGenerator configuration for '.$configFileData['name'].' > '.$packageConfigDir.'/'.self::$_configFileName;

        } else {
            $message = 'Created CodeGenerator configuration for '.$configFileData['name'].' > '.$packageConfigDir.'/'.self::$_configFileName
                ."\n".'Add this file to the version control system.';
        }
        CodeGenerator::addMessage($message, color: CodeGenerator::getOutputColor('creating'));

        return self::$_packageSettingsCache[$basePath] = $config['package'];

    } // _writePackageConfig()


    /**
     * Liefert die Konfiguration zu einem Package
     *
     * @param       string $basePath
     *              Der absolute Pfad zum Wurzelverzeichnis des Packages/Projekts, dessen Konfiguration
     *              gelesen werden soll.
     *
     * @param       bool $isProject
     *              Teilt der Methode mit ob der übergebene Pfad zum aktuellen Projekt oder zu einem
     *              Package gehört.
     *
     * @return      array
     *
     * @version     1.2.0 / 2025-05-31
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function get(string $basePath, bool $isProject): array
    {
        /*
        **  Wenn die Konfiguration bereits einmal gelesen wurde, dann wird sie aus
        **  dem Cache geladen. */
        if (isset(self::$_packageSettingsCache[$basePath])) {
            $config = self::$_packageSettingsCache[$basePath];

            $packageName = $config['name'];

            if ($isProject === true && self::$_projectPackageName === null) {
                self::$_projectPackageName = $packageName;
            }
            if (!isset(self::$_packagePathMap[$packageName])) {
                self::$_packagePathMap[$packageName] = $basePath;
            }
            return $config;
        }
        $packageConfigFile = $basePath.'/config/'.self::$_configFileName;
        /*
        **  Sollten keine Konfigurationdaten im Cache gefunden werden, wird überprüft ob
        **  das Paket bereits eine Konfigurationsdatei hat. */
        if (file_exists($packageConfigFile)) {
            $config = require $packageConfigFile;
            /*
            **  ConfigVersion < 1.2
            **  In älteren Konfigurationsdateien gab es immer nur einen Autor im Schlüssel "author".
            **  In der neueren Definition heißt der Schlüssel "authors" und ist ein Array mit
            **  mindestens einem Element. */
            if (isset($config['package']['author'])) {
                $config['package']['authors'] = [
                    $config['package']['author'],
                ];
                unset($config['package']['author']);
            }
            if (self::_requireUpdate($basePath, $config) === true) {
                return self::_updatePackageConfig($basePath, $config);
            }
            $packageName = $config['package']['name'];

            if ($isProject === true && self::$_projectPackageName === null) {
                self::$_projectPackageName = $packageName;
            }
            if (!isset(self::$_packagePathMap[$packageName])) {
                self::$_packagePathMap[$packageName] = $basePath;
            }
            $config['package']['isDevPackage'] = false;

            if ($config['package']['isProject'] === false) {
                $projectConfig = self::getProjectConfig();

                $path = $projectConfig['basePath'].'/vendor/'.$config['package']['vendorName'].'/globals';
                $realPath = str_replace('\\', '/', realpath($path));
                /*
                **  Wenn der Path und der RealPath voneinander abweichen, dann bedeutet das,
                **  dass das Paket über einen Symlink als Entwicklungspaket eingebunden ist. */
                $config['package']['isDevPackage'] = ($path !== $realPath);
            }
            return self::$_packageSettingsCache[$basePath] = $config['package'];
        }
        /*
        **  Wenn weder Konfigurationsdaten im Cache existieren noch eine Konfigurationsdatei vorhanden
        **  ist, dann wird eine neue Konfigurationsdatei erstellt. */
        return self::_buildPackageConfig($basePath, $isProject);

    } // get()


    /**
     * Liefert die Konfiguration zum Projekt
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-04-02
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getProjectConfig(): array
    {
        $projectPath = self::$_packagePathMap[self::$_projectPackageName];

        return self::$_packageSettingsCache[$projectPath];

    } // getProjectConfig()


} // class PackageConfig {}
