<?php
/**
 * Code Generator Package Element Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Elements
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Elements;


use BplanBase\CodeGenerator\Enums\CaseStyle;
use BplanBase\CodeGenerator\Enums\Number;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\Helpers\StringHelper;
use Illuminate\Support\Str;


/**
 * Code Generator Package Element Class
 *
 * @version     6.0.0 / 2025-03-16
 * @history     Definitions\Package, 2.0.0 / 2025-02-18
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Package
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_baseDir
     */
    private $_baseDir = 'src';


    /**
     * @var     string $_basePath
     */
    private $_basePath;


    /**
     * @var     string $_devStart
     */
    private $_devStart;


    /**
     * @var     bool $_isProject
     */
    private $_isProject = false;


    /**
     * @var     string $_name
     */
    private $_name;


    /**
     * @var     string $_namespace
     */
    private $_namespace;


    /**
     * @var     string $_packageName
     */
    private $_packageName;


    /**
     * @var     string $_prefix
     */
    private $_prefix;


    /**
     * @var     bool $_usePrefix
     */
    private $_usePrefix = false;


    /**
     * @var     string $_vendorName
     */
    private $_vendorName;


    /**
     * @var     float $_version
     */
    private $_version = 1.0;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Eine Map zum Auflösen des Projekt-/Paketnames zum jeweiligen Pfad
     *
     * @var     array $_packagePathMap
     */
    private static $_packagePathMap = [];


    /**
     * Ein Array in dem die Konfigurationen des Projekts und aller Pakete gesammelt werden
     *
     * Als Schlüssel wird der jeweilige BasePath verwendet.
     *
     * @var     array $_packageSettingsCache
     */
    private static $_packageSettingsCache = [];


    /**
     * Der Name des Projekts
     *
     * @var     string $_projectPackageName
     */
    private static $_projectPackageName;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       array $settings
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(array $settings)
    {
        $this->_devStart = date('Y');

        foreach ($settings as $key => $value) {
            $this->{'_'.$key} = $value;
        }
    } // __construct()


    /**
     * Vergleicht das übergebene Package mit dem aktuellen Package-Objekt
     *
     * @param       Package|string $package
     *              Wahlweise ein Package-Objekt oder der Name eines Pakets.
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-03-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function comparePackage(Package|string $package): bool
    {
        if ($package instanceof Package) {
            return $this->_name === $package->getName();
        }
        return $this->_name === $package;

    } // comparePackage()


    /**
     * Liefert den Namen des Verzeichnisses in dem die Klassen gespeichert werden
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getBaseDir(): string
    {
        return $this->_baseDir;

    } // getBaseDir()


    /**
     * Liefert den Pfad zum Wurzelverzeichnis
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getBasePath(): string
    {
        return $this->_basePath;

    } // getBasePath()


    /**
     * Liefert das Startjahr der Entwicklung des aktuellen Packages
     *
     * @return      int
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getDevStart(): int
    {
        return $this->_devStart;

    } // getDevStart()


    /**
     * Liefert den Namen des aktuellen Packages
     *
     * @return      null|string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getName(): null|string
    {
        return $this->_name;

    } // getName()


    /**
     * Liefert den Namespace des aktuellen Packages
     *
     * @return      null|string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getNamespace(): null|string
    {
        return $this->_namespace;

    } // getNamespace()


    /**
     * Liefert den reinen Paketnamen
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-03-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPackageName(): string
    {
        return $this->_packageName;

    } // getPackageName()


    /**
     * Liefert das Präfix des aktuellen Packages
     *
     * @return      null|string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPrefix(): null|string
    {
        return $this->_prefix;

    } // getPrefix()


    /**
     * Liefert den Vendor-Namen
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-03-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getVendorName(): string
    {
        return $this->_vendorName;

    } // getVendorName()


    /**
     * Liefert die Version aus der Package-Konfiguration
     *
     * @return      float
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getVersion(): float
    {
        return $this->_version;

    } // getVersion()


    /**
     * Liefert die Information, ob das aktuelle Objekt zum Projekt oder zu einem Package gehört
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-03-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function isForProject(): bool
    {
        return $this->_isProject === true;

    } // isForProject()


    /**
     * Bereitet die übergebene Zeichenkette für den Einsatz als BaseName auf
     *
     * Der BaseName ist DER Wert, aus dem sämtliche Benennungen (Klassen- und Relationennamen,
     * Ressourcenbezeichnungen usw.) gebildet werden. Er ist immer singular und im CaseStyle
     * "slug". Außerdem ist er mit dem Package-/Projekt-Präfix versehen, wenn die Einstellungen
     * des Pakets/Projekts es erfordern.
     *
     * @param       string $sourceString
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function prepareBaseName(string $sourceString): string
    {
        $sourceString = StringHelper::reformat($sourceString, CaseStyle::Slug, Number::Singular);

        if (!empty($this->_prefix)) {
            if ($this->_usePrefix === true) {
                /*
                **  Wenn ein Prefix verwendet werden soll, dann wird es hinzugefügt, sofern es nicht
                **  bereits vorhanden ist. */
                if (stripos($sourceString, $this->_prefix) !== 0) {
                    $sourceString = $this->_prefix.'-'.$sourceString;
                }
            } else {
                /*
                **  Wenn kein Prefix verwendet werden soll, dann wird es entfernt, sollte es bereits
                **  exisitieren. */
                if (stripos($sourceString, $this->_prefix) === 0) {
                    $sourceString = substr($sourceString, strlen($this->_prefix) + 1);
                }
            }
        }
        return $sourceString;

    } // prepareBaseName()


    /**
     * Liefert die Information, ob für das aktuelle Package das Präfix genutzt werden soll
     *
     * @return      float
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function usePrefix(): bool
    {
        return $this->_usePrefix;

    } // usePrefix()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Erzeugt eine neue Konfigurationsdatei für das aktuelle Package/Projekt
     *
     * @param       string $basePath
     *              Der absolute Pfad zum Wurzelverzeichnis des Packages/Projekts, dessen Konfiguration
     *              gelesen werden soll.
     *
     * @param       bool $isProject
     *              Teilt der Methode mit ob der übergebene Pfad zum aktuellen Projekt oder zu einem
     *              Package gehört.
     *
     * @return 	    array
     *
     * @version     1.2.0 / 2025-03-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _buildPackageConfig(string $basePath, bool $isProject = false): array
    {
        /*
        **  Die neue Konfigurationsdatei wird auf Basis der Defaults erzeugt. */
        $packageConfigDefaults = config('code-generator-defaults.package');

        if ($isProject === true) {
            /*
            **  Für das Projekt werden Daten aus der Projektkonfiguration verwendet. */
            $fullPackageName = config('app.name');
            $packageBaseDir = 'app';
            $packageNamespace = 'App';
            $packagePrefix = null;

            self::$_projectPackageName = $fullPackageName;

        } else {
            /*
            **  Bei Paketen wird die composer.json ausgelesen um den Paketnamen zu ermitteln. */
            $composerJson = trim(file_get_contents($basePath.'/composer.json'));

            $ComposerJson = json_decode($composerJson);

            $fullPackageName = $ComposerJson->name;
            $packageBaseDir = $packageConfigDefaults['baseDir'];
            /*
            **  Namespace aus dem Paketnamen erstellen. */
            list($vendorName, $packageName) = explode('/', $fullPackageName);

            $packageNamespace = Str::studly($vendorName).'\\\\'.Str::studly($packageName);
            /*
            **  Das Prefix wird auch aus dem Paketnamen ermittelt. Dazu wird der Vendor-Name überprüft.
            **  wenn er aus mehreren Teilen besteht, dann wird der letzte Teil des Namens verwendet.
            **  Ansonsten wird der vollständige Vendor-Name als Prefix genutzt. */
            if (strpos($vendorName, '-') !== false) {
                $tmp = explode('-', $vendorName);

                $packagePrefix = array_pop($tmp);

            } else {
                $packagePrefix = $vendorName;
            }
        }
        self::$_packagePathMap[$fullPackageName] = $basePath;
        /*
        **  Daten für den Dateiinhalt zusammenstellen. */
        $configFileData = [
            'baseDir' => $packageBaseDir,
            'devStart' => $packageConfigDefaults['devStart'],
            'isProject' => $isProject,
            'name' => $fullPackageName,
            'namespace' => $packageNamespace,
            'packageName' => $packageName,
            'prefix' => $packagePrefix,
            'usePrefix' => false,
            'vendorName' => $vendorName,
            'version' => 1.0,
        ];
        return self::_writePackageConfig($basePath, $configFileData, update: false);

    } // _buildPackageConfig()


    /**
     * Aktualisiert eine Konfigurationsdatei
     *
     * @param       string $basePath
     *              Der absolute Pfad zum Wurzelverzeichnis des Packages/Projekts, dessen Konfiguration
     *              gelesen werden soll.
     *
     * @param       bool $isProject
     *              Teilt der Methode mit ob der übergebene Pfad zum aktuellen Projekt oder zu einem
     *              Package gehört.
     *
     * @param       array $currentConfig
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _updatePackageConfig(string $basePath, array $currentConfig): array
    {
        $defaultConfig = config('code-generator-defaults.package');
        $updatedConfig = array_merge($defaultConfig, $currentConfig['package']);

        $updatedConfig['namespace'] = str_replace('\\', '\\\\', $updatedConfig['namespace']);

        return self::_writePackageConfig($basePath, $updatedConfig, update: true);

    } // _updatePackageConfig()


    /**
     * Schreibt eine Konfigurationsdatei
     *
     * Da nicht nur irgendwelche Werte, sondern auch PHP-Code in die Datei geschrieben werden muss, kann
     * nicht einfach mit var_dump() gearbeitet werden. Stattdessen wird der Dateiinhalt Zeile für Zeile
     * zusammengestellt.
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private static function _writePackageConfig(string $basePath, array $configFileData, bool $update): array
    {
        $packageConfigDir = 'config';
        $packageConfigFileName = 'package-code-generator.php';
        $packageConfigPath = $basePath.'/'.$packageConfigDir;
        $packageConfigFile = $packageConfigPath.'/'.$packageConfigFileName;

        $configFileContents = '<?'.'php'
            ."\n\n".'return ['
            ."\n".'    \'package\' => [';

        foreach ($configFileData as $key => $value) {
            $configFileContents .= "\n".'        '."'".$key."'".' => ';

            if (is_bool($value)) {
                $configFileContents .= ($value ? 'true' : 'false');

            } elseif ($value === null) {
                $configFileContents .= 'null';

            } elseif (is_numeric($value)) {
                $configFileContents .= $value;

            } elseif ($key === 'basePath') {
                $configFileContents .= "str_replace('\\\\', '/', realpath(dirname(__FILE__).'/..'))";

            } elseif ($key === 'version') {
                $configFileContents .= sprintf('%01.1f', $value);

            } else {
                $configFileContents .= "'".$value."'";
            }
            $configFileContents .= ',';
        }
        $configFileContents .= "\n".'    ],'
            ."\n".'    \'packageConfigVersion\' => '.config('code-generator-defaults.packageConfigVersion').','
            ."\n".'];'
            ."\n";

        if (!file_exists($packageConfigPath)) {
            mkdir($packageConfigPath, 0777, true);
        }
        /*
        ** Neue Konfigurationsdatei speichern und direkt laden. */
        file_put_contents($packageConfigFile, $configFileContents);

        $config = require $packageConfigFile;

        if ($update === true) {
            $message = 'Updated CodeGenerator configuration for '.$configFileData['name'].' > '.$packageConfigDir.'/'.$packageConfigFileName;

        } else {
            $message = 'Created CodeGenerator configuration for '.$configFileData['name'].' > '.$packageConfigDir.'/'.$packageConfigFileName
                ."\n".'Add this file to the version control system.';
        }
        CodeGenerator::addMessage($message, color: CodeGenerator::getOutputColor('creating'));

        return self::$_packageSettingsCache[$basePath] = $config['package'];

    } // _writePackageConfig()


    /**
     * Liefert die Konfiguration zu einem Package
     *
     * @param       string $basePath
     *              Der absolute Pfad zum Wurzelverzeichnis des Packages/Projekts, dessen Konfiguration
     *              gelesen werden soll.
     *
     * @param       bool $isProject
     *              Teilt der Methode mit ob der übergebene Pfad zum aktuellen Projekt oder zu einem
     *              Package gehört.
     *
     * @return      array
     *
     * @version     3.1.0 / 2025-03-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getPackageConfig(string $basePath, bool $isProject): array
    {
        /*
        **  Wenn die Konfiguration bereits einmal gelesen wurde, dann wird sie aus
        **  dem Cache geladen. */
        if (isset(self::$_packageSettingsCache[$basePath])) {
            $config = self::$_packageSettingsCache[$basePath];

            $packageName = $config['name'];

            if ($isProject === true && self::$_projectPackageName === null) {
                self::$_projectPackageName = $packageName;
            }
            if (!isset(self::$_packagePathMap[$packageName])) {
                self::$_packagePathMap[$packageName] = $basePath;
            }
            return $config;
        }
        $packageConfigFile = $basePath.'/config/package-code-generator.php';
        $updateConfig = false;
        /*
        **  Sollten keine Konfigurationdaten im Cache gefunden werden, wird überprüft ob
        **  das Paket eine Konfigurationsdatei hat. */
        if (file_exists($packageConfigFile)) {
            $config = require $packageConfigFile;

            $defaultConfigVersion = config('code-generator-defaults.packageConfigVersion');

            if (!isset($config['packageConfigVersion']) || ($config['packageConfigVersion'] < $defaultConfigVersion)) {
                return self::_updatePackageConfig($basePath, $config);
            }
            $packageName = $config['package']['name'];

            if ($isProject === true && self::$_projectPackageName === null) {
                self::$_projectPackageName = $packageName;
            }
            if (!isset(self::$_packagePathMap[$packageName])) {
                self::$_packagePathMap[$packageName] = $basePath;
            }
            return self::$_packageSettingsCache[$basePath] = $config['package'];
        }
        /*
        **  Wenn weder Konfigurationsdaten im Cache existieren noch eine Konfigurationsdatei vorhanden
        **  ist, dann wird eine neue Konfigurationsdatei erstellt. */
        return self::_buildPackageConfig($basePath, $isProject);

    } // getPackageConfig()


    /**
     * Liefert die Konfiguration zum Projekt
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getProjectPackageConfig(): array
    {
        $projectPath = self::$_packagePathMap[self::$_projectPackageName];

        return self::$_packageSettingsCache[$projectPath];

    } // getProjectPackageConfig()


} // class Package {}
