<?php
/**
 * Helper Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Helpers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 bplan-solutions GmbH & Co. KG <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Helpers;


use BplanBase\CodeGenerator\Enums\CaseStyle;
use BplanBase\CodeGenerator\Enums\Number;
use Illuminate\Support\Str;


/**
 * Helper Class
 *
 * @version     1.0.0 / 2025-03-04
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class StringHelper
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $string
     *
     * @param       CaseStyle|null $CaseStyle
     *              Steuert die Schreibweise des zurückgegebenen Werts.
     *              Beispiele anhand der Zeichenketten "deleted_at", "mainUser", "user-role":
     *
     *                  Parameter       Rückgaben
     *                  camel       >   deletedAt   /   mainUser    /   userRole
     *                  lower       >   deletedat   /   mainuser    /   user-role
     *                  slug        >   deleted-at  /   main-user   /   user-role
     *                  snake       >   deleted_at  /   main_user   /   user_role
     *                  studly      >   DeletedAt   /   MainUser    /   UserRole
     *                  upper       >   DELETED_AT  /   MAINUSER    /   USERROLE
     *
     * @param       Number|null $Number
     *              Der grammatikalische Numerus "plural" oder "singular".
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function reformat(string $string, CaseStyle|null $CaseStyle = null, Number|null $Number = null): string
    {
        if ($Number !== null) {
            switch ($Number) {
                case Number::Plural:
                    $string = Str::plural($string);
                    break;

                case Number::Singular:
                    $string = Str::singular($string);
                    break;
            }
        }
        if ($CaseStyle !== null) {
            switch ($CaseStyle) {
                case CaseStyle::Camel:
                    $string = Str::camel($string);
                    break;

                case CaseStyle::Lower:
                    $string = strtolower($string);
                    break;

                case CaseStyle::Slug:
                    $string = Str::slug(self::reformat($string, CaseStyle::Snake), '-');
                    break;

                case CaseStyle::Snake:
                    $string = Str::snake($string);
                    break;

                case CaseStyle::Studly:
                    $string = Str::studly($string);
                    break;

                case CaseStyle::Upper:
                    $string = strtoupper($string);
                    break;
            }
        }
        return $string;

    } // reformat()


} // class StringHelper {}
