<?php
/**
 * Code Generator Type Element Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Elements
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Elements;


use BplanBase\CodeGenerator\Elements\Field;
use BplanBase\CodeGenerator\Enums\CaseStyle;
use BplanBase\CodeGenerator\Enums\Generator;
use BplanBase\CodeGenerator\Enums\ModelType;
use BplanBase\CodeGenerator\Enums\Number;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\Helpers\StringHelper;
use BplanBase\CodeGenerator\Traits\HasAttributes;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Schema\ForeignKeyDefinition;
use Illuminate\Support\Fluent;
use Illuminate\Support\Str;


/**
 * Code Generator Type Element Class
 *
 * @version     5.1.0 / 2025-06-19
 * @history     Definitions\TypeDefinition, 9.0.0 / 2025-02-15
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Type
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    use HasAttributes;


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_additionalRelations
     */
    private $_additionalRelations = [];


    /**
     * @var     array $_appends
     */
    private $_appends;


    /**
     * @var     array $_attributes<string, mixed>
     */
    protected $_attributes = [
        'tableName' => null,
        'tablePrefix' => null,
        'typeName' => null,
    ];


    /**
     * @var     string $_baseName
     */
    private $_baseName;


    /**
     * Ein Array mit allen Spalten der Datenbanktabelle
     *
     * Dieser Array wird verwendet um die Field-Objekte zu erzeugen.
     *
     * @var     array $_columns
     */
    private $_columns = [];


    /**
     * @var     array $_excludes
     */
    private $_excludes = [];


    /**
     * @var     array $_extends
     */
    private $_extends;


    /**
     * @var     array $_fields
     */
    private $_fields = [];


    /**
     * @var     array $_foreignKeyAttributes
     */
    private $_foreignKeyAttributes = [];


    /**
     * @var     array $_foreignKeys
     */
    private $_foreignKeys = [];


    /**
     * @var     array $_indexQueries
     */
    private $_indexQueries = [];


    /**
     * @var     array $_jsonApiServers
     */
    private $_jsonApiServers;


    /**
     *
     * @var     ModelType $_modelType
     */
    private $_ModelType = ModelType::Model;


    /**
     * @var     Package $_Package
     */
    private $_Package;


    /**
     * @var     array $_pivotAppendFields
     */
    private $_pivotAppendFields;


    /**
     * @var     array|bool $_pivotAppends
     */
    private $_pivotAppends;


    /**
     * @var     array $_pivotFields
     */
    private $_pivotFields = [];


    /**
     * @var     array $_pivotFieldTables
     */
    private $_pivotFieldTables = [];


    /**
     * @var     array $_pivotTables
     */
    private $_pivotTables = [];


    /**
     * @var     array $_routing
     */
    private $_routing = [];


    /**
     * @var     array $_traits
     */
    private $_traits = [];


    /**
     * @var     array $_uses
     */
    private $_uses = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         array $_possibleExcludes
     * @version     3.0.0 / 2025-06-08
     */
    private static $_possibleExcludes = [
        Generator::LaravelActionCreate->value,
        Generator::LaravelActionDelete->value,
        Generator::LaravelActionUpdate->value,
        Generator::LaravelController->value,
        Generator::LaravelModel->value,
        Generator::LaravelObserver->value,
        Generator::LaravelRepository->value,
        // Generator::LaravelService->value,
        Generator::LivewireIndex->value,
        Generator::LivewireModalCreate->value,
        Generator::LivewireModalEdit->value,
    ];


    /**
     * @var     array $_fieldStorage
     */
    private static $_fieldStorage = [];


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       Package $Package
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return      void
     *
     * @version     1.1.0 / 2025-04-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(Package $Package, Blueprint $Blueprint, array $additionalSettings = [])
    {
        $this->_jsonApiServers = CodeGenerator::getJsonApiServers();
        $this->_Package = $Package;

        $this->_process($Blueprint, $additionalSettings);

    } // __construct()


    /**
     * Erzeugt Field-Objekte für alle Spalten der Datenbanktabelle
     *
     * Die Field-Objekte werden in der Objektvariablen $_fields, mit dem Spaltennamen als Schlüssel,
     * abgelegt. Außerdem werden sie zusätzlich noch zur Liste der Felder der Tabelle, in der statischen
     * Variablen $_fieldStorage, hinzugefügt.
     *
     * @param       array|null $additionalColumnSettings
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _createFieldObjects(array|null $additionalColumnSettings): self
    {
        $tableName = $this->_attributes['tableName'];

        if (!isset(self::$_fieldStorage[$tableName])) {
            self::$_fieldStorage[$tableName] = [];
        }
        foreach ($this->_columns as $columnName => $columnDefinition) {
            if ($this->hasField($columnName)) {
                $Field = $this->_getField($columnName);

                $Field->update($columnDefinition, $additionalColumnSettings[$columnName] ?? null, $this->_foreignKeyAttributes[$columnName] ?? null);

            } else {
                $Field = new Field($this->_Package, $columnDefinition, $additionalColumnSettings[$columnName] ?? null, $this->_foreignKeyAttributes[$columnName] ?? null);
            }
            $this->_fields[$columnName] = $Field;

            self::$_fieldStorage[$tableName][$columnName] = $Field;
        }
        return $this;

    } // _createFieldObjects()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    Field
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function _getField(string $fieldName): Field
    {
        return $this->_fields[$fieldName];

    } // _getField()


    /**
     *
     * @param       array|null $excludes
     *
     * @param       array & $additionalSettings
     *
     * @return 	    $this
     *
     * @version     2.0.0 / 2025-03-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareExcludes(array|null $excludes, array & $additionalSettings): self
    {
        if ($excludes === null) {
            return $this;
        }
        foreach ($excludes as $exclude) {
            if (!in_array($exclude, self::$_possibleExcludes)) {
                continue;
            }
            $this->_excludes[$exclude] = $exclude;
        }
        if (isset($this->_excludes[Generator::LaravelActionCreate->value])
         && isset($this->_excludes[Generator::LaravelActionDelete->value])
         && isset($this->_excludes[Generator::LaravelActionUpdate->value])) {
            $this->_excludes[Generator::LaravelService->value] = Generator::LaravelService->value;

            if (!isset($additionalSettings['routing'])) {
                $additionalSettings['routing'] = [];
            }
            foreach ($this->_jsonApiServers as $server => $status) {
                if ($status === false) {
                    continue;
                }
                $server = strtolower($server);

                $additionalSettings['routing'][$server] = 'readOnly';
            }
        } elseif (isset($this->_excludes[Generator::LaravelActionCreate->value])
          || isset($this->_excludes[Generator::LaravelActionDelete->value])
          || isset($this->_excludes[Generator::LaravelActionUpdate->value])) {
            $actions = [];

            if (!isset($additionalSettings['routing'])) {
                $additionalSettings['routing'] = [];
            }
            foreach ($this->_jsonApiServers as $server => $status) {
                $server = strtolower($server);
                /*
                **  Wenn eh nur die Lese- oder Relationen-Routen implementiert werden sollen, dann kann
                **  der aktuelle Schleifendurchlauf gleich abgebrochen werden. */
                if (isset($additionalSettings['routing'][$server])
                  && ($additionalSettings['routing'][$server] === false
                   || $additionalSettings['routing'][$server] === 'readOnly'
                   || $additionalSettings['routing'][$server] === 'relationsOnly')) {
                    continue;
                }
                if (!isset($additionalSettings['routing'][$server])) {
                    $additionalSettings['routing'][$server] = [
                        'actions' => [],
                        'method' => 'except',
                    ];
                } else {
                    if (!isset($additionalSettings['routing'][$server]['actions'])) {
                        $additionalSettings['routing'][$server]['actions'] = [];
                    }
                    if (!isset($additionalSettings['routing'][$server]['method'])) {
                        $additionalSettings['routing'][$server]['method'] = 'except';

                    } elseif ($additionalSettings['routing'][$server]['method'] !== 'except') {
                        $additionalSettings['routing'][$server]['actions'] = [];
                        $additionalSettings['routing'][$server]['method'] = 'except';
                    }
                }
                foreach (self::$_possibleExcludes as $exclude) {
                    if (!isset($this->_excludes[$exclude])) {
                        continue;
                    }
                    switch ($exclude) {
                        case Generator::LaravelActionCreate->value:
                            $additionalSettings['routing'][$server]['actions'][] = 'store';
                            break;

                        case Generator::LaravelActionDelete->value:
                            $additionalSettings['routing'][$server]['actions'][] = 'destroy';
                            break;

                        case Generator::LaravelActionUpdate->value:
                            $additionalSettings['routing'][$server]['actions'][] = 'update';
                            break;
                    }
                }
            }
        }
        return $this;

    } // _prepareExcludes()


    /**
     *
     * @param       array|null $indexQueries
     *
     * @return 	    $this
     *
     * @version     2.0.0 / 2025-03-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareIndexQueries(array|null $indexQueries): self
    {
        if ($indexQueries === null) {
            return $this;
        }
        foreach ($this->_jsonApiServers as $server => $status) {
            $lowerServer = strtolower($server);

            if (isset($indexQueries[$lowerServer])) {
                $this->_jsonApiServers[$server] = true;

                $this->_indexQueries[$lowerServer] = $indexQueries[$lowerServer];
            }
            /*
            **  Die Einstellung für alle Server überschreibt immer die für einzelne Server.
            **
            **  Eigentlich würde man eine spezifischere Definition gegenüber einer allgemeinen Definition bevorzugen,
            **  allerdings macht die Verwendung von "ALL_SERVERS" in Kombination mit der spezifischen Angabe eines
            **  bestimmten Servers keinen Sinn. Deshalb wird hier die allgemeine Angabe der spezifischeren vorgezogen. */
            if (isset($indexQueries[CodeGenerator::ALL_SERVERS_KEY])) {
                $this->_jsonApiServers[$server] = true;

                $this->_indexQueries[$lowerServer] = $indexQueries[CodeGenerator::ALL_SERVERS_KEY];
            }
        }
        return $this;

    } // _prepareIndexQueries()


    /**
     *
     * @param       array|null $pivotSettings
     *
     * @return 	    $this
     *
     * @version     2.1.0 / 2025-05-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _preparePivotSettings(array|null $pivotSettings): self
    {
        if ($pivotSettings === null) {
            return $this;
        }
        $this->_pivotAppends = $pivotSettings['appends'] ?? true;

        if ($this->_pivotAppends !== false) {
            $this->_pivotAppendFields = [];
        }
        $this->_pivotFields = $pivotSettings['relations'];
        $this->_pivotFieldTables = array_flip($this->_pivotFields);

        $this->_pivotTables = array_combine($this->_pivotFields, array_reverse($this->_pivotFields));

        $this->_ModelType = ModelType::Pivot;

        return $this;

    } // _preparePivotSettings()


    /**
     * Startet die Verarbeitung
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return      $this
     *
     * @version     1.3.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _process(Blueprint $Blueprint, array $additionalSettings = []): self
    {
        $tableName = $Blueprint->getTable();
        $tablePrefix = $Blueprint->getPrefix();

        $this->_baseName = $this->_Package->prepareBaseName($tableName);
        //  todo    Prüfen, wofür der TypeName noch benötigt wird. Eventuell können darüber die Namen der Laravel Core-Klassen "eingefroren" werden.
        if (isset($additionalSettings['typeName'])) {
            $typeName = $additionalSettings['typeName'];
        } else {
            $typeName = $tableName;
        }
        $this->_attributes['tableName'] = $tableName;
        $this->_attributes['tablePrefix'] = $tablePrefix;
        $this->_attributes['typeName'] = $typeName;

        $this->_processAdditionalSettings($additionalSettings);
        /*
        **  Die Felder werden mehrfach verarbeitet. Im ersten Verarbeitungsschritt, in der Methode
        **  _processColumns(), werden zunächst alle Felder abgearbeitet um sie in $_columns zu sammeln.
        **  Im zweiten Verarbeitungsschritt, in der Methode _processCommands() werden dann nur die
        **  ForeignKey-Felder berücksichtigt. Dabei werden dann auch die gesammelten Feldinformationen
        **  zu den ForeignKey-Feldern nochmal ergänzt. */
        $this->_processColumns($Blueprint->getColumns());
        $this->_processCommands($Blueprint->getCommands(), $additionalSettings['columns'] ?? null);
        /*
        **  Field-Objekte erzeugen. */
        $this->_createFieldObjects($additionalSettings['columns'] ?? null);

        return $this;

    } // _process()


    /**
     *
     * @param       array $additionalSettings
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processAdditionalSettings(array $additionalSettings, bool $forUpdate = false): self
    {
        $this->_preparePivotSettings($additionalSettings['pivot'] ?? null);
        $this->_prepareExcludes($additionalSettings['exclude'] ?? null, $additionalSettings);
        $this->_prepareIndexQueries($additionalSettings['indexQueries'] ?? null);

        $this->_additionalRelations = $additionalSettings['additionalRelations'] ?? $this->_additionalRelations;
        $this->_extends = $additionalSettings['extends'] ?? $this->_extends;
        $this->_routing = $additionalSettings['routing'] ?? $this->_routing;
        $this->_traits = $additionalSettings['traits'] ?? $this->_traits;
        $this->_uses = $additionalSettings['uses'] ?? $this->_uses;

        $this->_processAppends($additionalSettings['appends'] ?? null);

        return $this;

    } // _processAdditionalSettings()


    /**
     * Bereitet die Werte der appends-Option auf
     *
     * @param       array|null $appends
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2025-06-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processAppends(array|null $appends): self
    {
        $this->_appends = [];

        if ($appends === null) {
            return $this;
        }
        /*
        **  [
        **      'label' => [
        **          'generate' => CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_MODEL_ARRAY + CodeGenerator::APPENDS_MODEL_METHOD,
        **          'return' => '$this->last_name.\', \'.$this->first_name',
        **      ]
        **  ]
        */
        foreach ($appends as $key => $value) {
            if (is_numeric($key)) {
                $key = $value;
            }
            $field = $key;

            $returnType = 'mixed';

            if (is_array($value)) {
                $generate = null;
                $return = 'null';

                if (isset($value['generate'])) {
                    if (is_bool($value['generate'])) {
                        if ($value['generate'] === true) {
                            $generate = CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_LARAVEL_MODEL_ARRAY + CodeGenerator::APPENDS_LARAVEL_MODEL_METHOD;
                        }
                    } else {
                        $generate = (int) $value['generate'];
                    }
                } else {
                    $generate = CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_LARAVEL_MODEL_ARRAY + CodeGenerator::APPENDS_LARAVEL_MODEL_METHOD;
                }
                if (isset($value['return'])) {
                    if (is_bool($value['return'])) {
                        $return = ($value['return'] === true) ? 'true' : 'false';
                        $returnType = 'bool';

                    } else {
                        $return = $value['return'];
                    }
                }
                if (isset($value['return-type'])) {
                    $returnType = $value['return-type'];
                }
                $default = $value['default'] ?? null;
                $depends = $value['depends'] ?? Str::plural($field);
                $type = $value['type'] ?? 'default';

            } else {
                if (is_numeric($value)) {
                    /*
                    **  [
                    **      'profile_photo_url' => CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_MODEL_ARRAY,
                    **  ]
                    */
                    $generate = $value;

                } else {
                    $generate = CodeGenerator::APPENDS_JSONAPI_SCHEMA + CodeGenerator::APPENDS_LARAVEL_MODEL_ARRAY + CodeGenerator::APPENDS_LARAVEL_MODEL_METHOD;
                }
                $default = null;
                $depends = null;
                $return = 'null';
                $type = 'default';
            }
            $generators = [];

            if ($generate & CodeGenerator::APPENDS_JSONAPI_SCHEMA) {
                $generators[Generator::JsonApiSchema->value] = true;
            }
            if ($generate & CodeGenerator::APPENDS_LARAVEL_MODEL_ARRAY) {
                $generators[Generator::LaravelModel->value.'.array'] = true;
            }
            if ($generate & CodeGenerator::APPENDS_LARAVEL_MODEL_METHOD) {
                $generators[Generator::LaravelModel->value.'.method'] = true;
            }
            if (substr($return, -1) !== ';') {
                //$return .= ';';
            }
            $this->_appends[$field] = [
                'default' => $default,
                'depends' => $depends,
                'field' => $field,
                'generate' => $generate,
                'generators' => $generators,
                'return' => $return,
                'return-type' => $returnType,
                'type' => $type,
            ];
        }
        return $this;

    } // _processAppends()


    /**
     * Verarbeitet die Liste der Columns aus dem Blueprint-Objekt und sammelt sie im Array $_columns
     *
     * In $_columns wird ein ein assoziatives Array mit den Spaltennamen als Schlüssel aufgebaut.
     *
     * Die Feldinformationen der ForeignKeys in dieser Liste werden in einem späteren Verarbeitungsschritt,
     * in der Methode _processCommands(), noch um Informationen aus dem Blueprint-Objekt ergänzt.
     *
     * @param       array $columns
     *
     * @return 	    void
     *
     * @version     1.1.0 / 2025-03-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processColumns(array $columns): self
    {
        foreach ($columns as $Column) {
            $column = $Column->toArray();
            $columnName = $column['name'];

            $column['isForeignKey'] = false;

            ksort($column);

            $this->_columns[$columnName] = $column;

            if ($this->_ModelType === ModelType::Pivot) {
                if ($columnName !== 'id' && !isset($this->_pivotFields[$columnName])) {
                    if ($this->_pivotAppends === true) {
                        $this->_pivotAppendFields[] = $columnName;

                    } elseif (is_array($this->_pivotAppends) && in_array($columnName, $this->_pivotAppends)) {
                        $this->_pivotAppendFields[] = $columnName;
                    }
                }
            }
        }
        if ($this->_ModelType === ModelType::Pivot && is_array($this->_pivotAppendFields)) {
            sort($this->_pivotAppendFields);
        }
        return $this;

    } // _processColumns()


    /**
     * Verarbeitet Commands aus dem Blueprint-Objekt und ermittelt Informationen zu ForeignKeys
     *
     * Die Informationen der ForeignKeys werden verwendet um die entsprechenden Felder im Array $_columns,
     * der zuvor in der Methode _processColumns() vorbereitet wurde, zu ergänzen. Außerdem werden die
     * Objektvariablen $_foreignKeyAttributes und $_foreignKeys zusammengestellt.
     *
     * @param       array $commands
     *
     * @param       array|null $additionalColumnSetting
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2025-03-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processCommands(array $commands, array|null $additionalColumnSetting): self
    {
        foreach ($commands as $Command) {
            $attributes = $Command->getAttributes();

            if ($Command instanceof ForeignKeyDefinition) {
                $columnName = $attributes['columns'][0];
                $relatedTable = $attributes['on'];

                $attributes['relationType'] = 'BelongsTo';

                ksort($attributes);

                $isPivotField = false;

                if (isset($this->_pivotFields[$columnName])) {
                    $isPivotField = true;

                    // $this->_pivotTables[$this->_attributes['tableName']] = $relatedTable;
                }
                $this->_columns[$columnName]['index'] = $attributes['index'];
                $this->_columns[$columnName]['isForeignKey'] = true;
                $this->_columns[$columnName]['isPivotField'] = $isPivotField;
                $this->_columns[$columnName]['relatedField'] = $attributes['references'];
                $this->_columns[$columnName]['relatedTable'] = $attributes['on'];
                $this->_columns[$columnName]['relationName'] = null;
                $this->_columns[$columnName]['relationType'] = $attributes['relationType'];
                $this->_columns[$columnName]['reverseRelationName'] = null;
                $this->_columns[$columnName]['reverseRelationType'] = null;

                $this->_foreignKeyAttributes[$columnName] = $attributes;

                $this->_foreignKeys[$columnName] = $this->_columns[$columnName];

                if (isset($additionalColumnSetting[$columnName]['relationName'])) {
                    $this->_foreignKeys[$columnName]['relationName'] = $additionalColumnSetting[$columnName]['relationName'];
                }
                if (isset($additionalColumnSetting[$columnName]['reverseRelationName'])) {
                    $this->_foreignKeys[$columnName]['reverseRelationName'] = $additionalColumnSetting[$columnName]['reverseRelationName'];
                }
                if (isset($additionalColumnSetting[$columnName]['reverseRelationType'])) {
                    $this->_foreignKeys[$columnName]['reverseRelationType'] = $additionalColumnSetting[$columnName]['reverseRelationType'];
                }
                ksort($this->_foreignKeys[$columnName]);

                continue;

            } elseif ($Command instanceof Fluent) {
                if ($attributes['name'] === 'create') {
                    continue;
                }
                /*
                    Index-Informationen werden aktuell noch nicht berücksichtigt, könnten aber insbesondere
                    für die Rules relevant werden um zum Beispiel eine unique-Rule zu erzeugen.

                */
                //self::$_collectedIndexes[$this->_attributes['tableName']][] = $attributes;
            }
        }
        return $this;

    } // _processCommands()


    /**
     *
     * @param       string $columnName
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function fieldExists(string $columnName): bool
    {
        return isset($this->_fields[$columnName]);

    } // fieldExists()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getAdditionalRelations(): array
    {
        return $this->_additionalRelations;

    } // getAdditionalRelations()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-12-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getAppends(): array
    {
        return $this->_appends;

    } // getAppends()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getBaseName(CaseStyle $CaseStyle, Number $Number)
    {
        return StringHelper::reformat($this->_baseName, $CaseStyle, $Number);

    } // getBaseName()


    /**
     *
     * @param       string $extendsFor
     *
     * @return 	    string|null
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getExtends(string $extendsFor): string|null
    {
        if (empty($this->_extends[$extendsFor])) {
            return null;
        }
        return $this->_extends[$extendsFor];

    } // getExtends()


    /**
     *
     * @param       string $tableName
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getFields(string $tableName): array
    {
        return self::$_fieldStorage[$tableName];

    } // getFields()


    /**
     *
     * @param       string $columnName
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getForeignKey(string $columnName): array|null
    {
        if (!isset($this->_foreignKeys[$columnName])) {
            return null;
        }
        return $this->_foreignKeys[$columnName];

    } // getForeignKey()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getForeignKeys(): array
    {
        return $this->_foreignKeys;

    } // getForeignKeys()


    /**
     *
     * @return 	    null|string
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getIndexQuery(string $server): null|string
    {
        $server = strtolower($server);

        return $this->_indexQueries[$server] ?? null;

    } // getIndexQuery()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getJsonApiServers(): array
    {
        return $this->_jsonApiServers;

    } // getJsonApiServers()


    /**
     *
     * @return 	    ModelType
     *
     * @version     2.0.0 / 2025-03-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getModelType(): ModelType
    {
        return $this->_ModelType;

    } // getModelType()


    /**
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2025-03-11
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPivotAppendFields(): array|null
    {
        return $this->_pivotAppendFields;

    } // getPivotAppendFields()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPivotTables(): array
    {
        return $this->_pivotTables;

    } // getPivotTables()


    /**
     *
     * @return 	    Package
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getPackage(): Package
    {
        return $this->_Package;

    } // getPackage()


    /**
     *
     * @param       string $pivotFieldName
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-05-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRelatedPivotField(string $pivotFieldName): string
    {
        $pivotFieldTableName = $this->_pivotFields[$pivotFieldName];

        $relatedTableName = $this->_pivotTables[$pivotFieldTableName];

        return $relatedFieldName = $this->_pivotFieldTables[$relatedTableName];

    } // getRelatedPivotField()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRouting(): array
    {
        return $this->_routing;

    } // getRouting()


    /**
     *
     * @param       string $traitsFor
     *
     * @return 	    array|null
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getTraits(string $traitsFor): array|null
    {
        if (empty($this->_traits[$traitsFor])) {
            return null;
        }
        return $this->_traits[$traitsFor];

    } // getTraits()


    /**
     *
     * @param       string $usesFor
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getUses(string $usesFor): array|null
    {
        if (empty($this->_uses[$usesFor])) {
            return null;
        }
        return $this->_uses[$usesFor];

    } // getUses()


    /**
     *
     * @param       Generator|string $generator
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasExclude(Generator|string $generator): bool
    {
        if ($generator instanceof Generator) {
            $generator = $generator->value;
        }
        return isset($this->_excludes[$generator]);

    } // hasExclude()


    /**
     *
     * @param       string $name
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasField(string $name): bool
    {
        return isset($this->_fields[$name]);

    } // hasField()


    /**
     * Liefert die Information, ob der aktuelle Type zu einem Package gehört
     *
     * @return 	    bool
     *
     * @version     1.1.0 / 2025-03-10
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function isForPackage(): bool
    {
        return $this->_Package->isForProject() === false;

    } // isForPackage()


    /**
     *
     * @param       Blueprint $Blueprint
     *
     * @param       array $additionalSettings
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function update(Blueprint $Blueprint, array $additionalSettings = []): self
    {
        $this->_processColumns($Blueprint->getColumns());
        $this->_processCommands($Blueprint->getCommands(), $additionalSettings['columns'] ?? null);

        $this->_processAdditionalSettings($additionalSettings, forUpdate: true);

        $this->_createFieldObjects($additionalSettings['columns'] ?? null);

        return $this;

    } // update()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function writeCacheFile(): self
    {
        $cachePath = CodeGenerator::getCachePath();

        ksort($this->_fields);

        unset($this->_columns);

        file_put_contents($cachePath.'/'.$this->_attributes['tableName'].'.php', '<?php'."\n\n".'return '.var_export($this, true).';'."\n");

        return $this;

    } // writeCacheFile()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class Type {}
