<?php
/**
 * Base Generator Class
 *
 * @version     3.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-base/laravel-code-generator
 * @subpackage  Generators
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanBase\CodeGenerator\Generators\CodeGenerator;


use BplanBase\CodeGenerator\DTOs\ConsoleMessageDTO;
use BplanBase\CodeGenerator\Elements\Package;
use BplanBase\CodeGenerator\Enums\GeneratorMode;
use BplanBase\CodeGenerator\Enums\PackageType;
use BplanBase\CodeGenerator\Generators\CodeGenerator;
use BplanBase\CodeGenerator\DTOs\GeneratedDTO;


/**
 * Base Code Generator Class
 *
 * @version     11.1.0 / 2025-11-23
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
abstract class BaseGenerator
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string CONTEXT_BUILD
     */
    const CONTEXT_BUILD      = 'build';


    /**
     * @var     string CONTEXT_PRODUCTIVE
     */
    const CONTEXT_PRODUCTIVE = 'productive';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Der Name der obersten Verzeichnisebene nach dem BasePath (z.B. "app").
     *
     * @var     string $_baseDir
     */
    protected $_baseDir;


    /**
     * @var     bool $_buildOnlyMode
     */
    protected bool $_buildOnlyMode = false;


    /**
     * @var     array $_defaultReplacements
     */
    private $_defaultReplacements;


    /**
     * @var     array $_errors
     */
    protected $_errors = [];


    /**
     * @var     string $_fileHash
     */
    private $_fileHash;


    /**
     * Der relative Pfad zum Zielverzeichnis, ausgehend vom Wurzelverzeichnis des Projekts
     *
     * Der Wert dieser Variablen muss in den abgeleiteten Generator-Klassen gesetzt werden.
     *
     * @var     string $_filePath
     */
    protected $_filePath;


    /**
     * @var     null|string $_generatedDate
     */
    private $_generatedDate;


    /**
     * @var     null|string $_generatedTimestamp
     */
    private $_generatedTimestamp;


    /**
     * @var     GeneratorMode $_GeneratorMode
     */
    protected $_GeneratorMode = GeneratorMode::Default;


    /**
     * @var     string $_namespace
     */
    protected $_namespace;


    /**
     * @var     null|Package $_Package
     */
    protected Package $_Package;


    /**
     * Enthält bei Typen aus externen Paketen den Paketnamen, ist bei Projekt-Typen NULL
     *
     * @var     null|string $_packageName
     */
    protected $_packageName;


    /**
     * Enthält bei Typen aus externen Paketen den Paketnamen, ist bei Projekt-Typen NULL
     *
     * @var     PackageType $_PackageType
     */
    protected $_PackageType;


    /**
     * @var     array $_paths
     */
    protected $_paths;


    /**
     * @var     array $_technicalFieldNames
     */
    protected $_technicalFieldNames = [
        'active',
        'created_at',
        'deleted_at',
        'updated_at',
    ];


    /**
     * @var     array $_traits
     */
    private $_traits = [];


    /**
     * @var     array $_uses
     */
    private $_uses = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_generator
     */
    protected static $_generator;


    /**
     * @var         array $_resultCounters
     * @version     2.0.0 / 2025-11-23
     */
    private static $_resultCounters = [
        'created' => 0,
        'replaced' => 0,
        'protected' => 0,
        'skipped' => 0,
        'unchanged' => 0,
    ];


    /**
     * @var         string $_startTime
     */
    private static $_startTime;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       array|string
     *              string  'Illuminate\\Database\\Eloquent\\SoftDeletes' oder auch
     *                      'Illuminate\\Database\\Eloquent\\SoftDeletes as MySoftDelete'
     *
     *              array   [
     *                          'Illuminate\\Database\\Eloquent\\SoftDeletes',
     *                              oder auch
     *                          'Illuminate\\Database\\Eloquent\\SoftDeletes as MySoftDelete',
     *                          ...
     *                      ]
     *                      oder auch
     *                      [
     *                          'SoftDeletes' => 'Illuminate\\Database\\Eloquent\\SoftDeletes',
     *                              oder auch
     *                          'MySoftDelete' => 'Illuminate\\Database\\Eloquent\\SoftDeletes as MySoftDelete',
     *                      ]
     *
     * @return 	    $this
     *
     * @version     1.0.1 / 2025-05-30
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _addTrait(array|string $trait): self
    {
        if (is_array($trait)) {
            foreach ($trait as $key => $item) {
                if (is_numeric($key)) {
                    $this->_addTrait($item);

                    continue;
                }
                $this->_traits[$key] = $key;
                $this->_addUse($item);
            }
            return $this;
        }
        if (stripos($trait, ' as ') === false) {
            $name = basename(str_replace('\\', '/', $trait));

        } else {
            list($nul, $name) = preg_split("/ as /i", $trait);
        }
        $this->_traits[$name] = trim($name);
        $this->_addUse($trait);

        return $this;

    } // _addTrait()


    /**
     *
     * @param       array|string $use
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _addUse(array|string $use): self
    {
        if (is_array($use)) {
            foreach ($use as $class) {
                $this->_addUse($class);
            }
            return $this;
        }
        $this->_uses[$use] = $use;

        return $this;

    } // _addUse()


    /**
     *
     * @param       Package $Package
     *
     * @param       string $file
     *
     * @param       null|string & $ignoreReason
     *
     * @param       bool $replaceOriginalFile
     *              Um eine Originaldatei zu ersetzen kann der Wert dieses Parameters
     *              auf TRUE gesetzt werden.
     *
     * @return      bool Liefert TRUE, wenn die Datei geschrieben werden darf.
     *
     * @version     4.1.0 / 2025-11-23
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _checkFileWriteStatus(Package $Package, string $context, string $file, null|string & $ignoreReason = null, bool $replaceOriginalFile = false): bool
    {
        /*
        **  Vorgelagerte Prüfungen durchführen, um zu entscheiden ob die aktuelle Datei geschrieben
        **  oder auch nicht geschrieben werden soll (unabhängig vom Dateiinhalt beziehungsweise
        **  CodeGenerator-Tag). */
        if (!file_exists($file) || ($context === self::CONTEXT_BUILD && config('code-generator.force-create-build-files') === true)) {
            /*
            **  Wenn die Zieldatei nicht existiert, dann kann sie in jedem Fall
            **  geschrieben werden. */
            return true;
        }
        if ($Package->isForProject() === false && $Package->isDevPackage() === false) {
            /*
            **  Schreibvorgang unterbinden, wenn es sich nicht um eine Projektdatei (sondern um eine
            **  Package-Datei) handelt, das Package aber nicht als Entwicklungspaket installiert ist. */
            $ignoreReason = 'no-dev-package';

            return false;
        }
        /*
        **  Zieldatei öffnen und die erste Zeile auslesen. */
        $line = fgets(fopen($file, 'r'));

        if (strpos($line, CodeGenerator::CODE_GENERATOR_TAG) !== false) {
            /*
            **  Wenn es ein CodeGenerator-Tag gibt, dann wird der FileHash ermittelt
            **  und mit dem aktuellen FileHash verglichen. */
            list($tag, $hash, $nul) = explode('Δ', $line, 3);

            $hash = trim($hash);

            if ($hash !== $this->_fileHash) {
                /*
                **  Bei abweichendem FileHash wird die Datei geschrieben.
                **  In diesem Fall werden immer das aktuelle Datum und die aktuelle
                **  Uhrzeit verwendet. */
                return true;
            }
            if ($context === self::CONTEXT_BUILD) {
                /*
                **  Generated-Daten aus der Build-Datei ermitteln. */
                $BuildGenerated = $this->_determineFileGenerated($file);
                /*
                **  Versuchen die Generated-Daten aus der Productive-Datei zu ermitteln. */
                $ProductiveGenerated = $this->_initGenerated($file);

                if ($ProductiveGenerated?->timestamp !== null && $ProductiveGenerated?->timestamp !== $BuildGenerated->timestamp) {
                    /*
                    **  Wenn Daten in der Productive-Datei gefunden werden und diese nicht
                    **  mit den Daten der Build-Datei übereinstimmen, dann wird die Build-
                    **  Datei mit den Produktiv-Daten neu geschrieben. */
                    $this->_generatedDate = $ProductiveGenerated->date;
                    $this->_generatedTimestamp = $ProductiveGenerated->timestamp;

                    return true;
                }
            }
            // return true;
            $ignoreReason = 'unchanged';

            return false;
        }
        if ($replaceOriginalFile === true) {
            /*
            **  Wenn Originaldateien ersetzt werden sollen, dann fungiert das @generated-Tag
            **  als Schreibschutz.
            **  Ist es nicht vorhanden, dann bedeutet das, dass es sich um die Originaldatei
            **  handelt, die ersetzt werden soll. Existiert das @generated-Tag jedoch, dann
            **  ist die vorhandene Datei geschützt ist und darf überschrieben werden.
            **
            **  Sollte zusätzlich noch ein CodeGenerator-Tag vorhanden sein, dann kommt es gar
            **  nicht zu diesem Fall, weil die Datei bereits weiter oben verarbeitet wurde.*/
            $ProductiveGenerated = $this->_initGenerated($file);

            if ($ProductiveGenerated->timestamp === null) {
                return true;
            }
        }
        $ignoreReason = 'protected';

        return false;

    } // _checkFileWriteStatus()


    /**
     * Ermittelt Generated-Daten aus der übergebenen Datei
     *
     * @param       string $file
     *
     * @return 	    GeneratedDTO
     *
     * @version     1.1.0 / 2025-05-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _determineFileGenerated(string $file)
    {
        $date = null;
        $timestamp = null;

        $fp = fopen($file, 'r');
        /*
        **  Die ersten Zeilen der Datei abarbeiten und nach einem @generator-Tag suchen. Wenn
        **  ein entsprechendes Tag gefunden wird, dann wird die Zeile zerlegt um den kompletten
        **  Zeitstempel und das Datum zu ermitteln. */
        for ($i = 0; $i < 10; $i++) {
            $line = fgets($fp);

            if (strpos($line, '@generated') !== false) {
                list($nul, $tmp) = explode('@generated', $line);

                $tmp = trim($tmp);

                if (!empty($tmp)) {
                    list($date) = explode(' ', $tmp);

                    $date = trim($date);
                    $timestamp = $tmp;
                }
                /*
                **  Abbruch der Schleife, sobald Daten gefunden und verarbeitet wurden. */
                break;
            }
        }
        fclose($fp);

        return new GeneratedDTO($timestamp, $date);

    } // _determineFileGenerated()


    /**
     * Prepares the console output
     *
     * @param       boolean $fileWriteStatus
     *
     * @param       string $context
     *
     * @param       string $file
     *
     * @param       string $ignoreReason
     *
     * @return      ConsoleMessageDTO
     *
     * @version     1.2.0 / 2025-11-23
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getConsoleMessage(bool $fileWriteStatus, string $context, string $file, string|null $ignoreReason): ConsoleMessageDTO
    {
        $color = null;
        $message = null;

        if ($fileWriteStatus === false) {
            /*
            **  Die Zieldatei ist geschützt oder unverändert und wird nicht überschrieben. */
            if ($context === self::CONTEXT_PRODUCTIVE) {
                $color = null;

                if ($ignoreReason === 'protected') {
                    $color = CodeGenerator::getOutputColor('protecting');
                    $message = 'Ignoring protected file';

                    self::$_resultCounters['protected']++;

                } elseif ($ignoreReason === 'no-dev-package') {
                    $color = CodeGenerator::getOutputColor('no-dev-package');
                    $message = 'Skipping no dev file';

                    self::$_resultCounters['unchanged']++;

                } else {
                    $message = 'Skipping unchanged file';

                    self::$_resultCounters['unchanged']++;
                }
            } else {
                /*
                **  Ausgaben für die build-Dateien werden nur ausgegeben, wenn der
                **  BuildOnly-Modus aktiviert ist. */
                if ($this->_buildOnlyMode === true) {
                    if ($ignoreReason === 'protected') {
                        $color = CodeGenerator::getOutputColor('protecting');
                        $message = 'Ignoring protected build file';

                        self::$_resultCounters['protected']++;

                    } else {
                        $message = 'Skipping unchanged build file';

                        self::$_resultCounters['unchanged']++;
                    }
                }
            }
        } else {
            if ($context === self::CONTEXT_PRODUCTIVE) {
                if (file_exists($file)) {
                    /*
                    **  Der globale BuildOnly-Modus hat nur Auswirkungen auf bestehende 
                    **  Productive-Dateien deren Inhalt eigentlich verändert werden würde. */
                    if (config('code-generator.build-only-mode') === true) {
                        /*
                        **  Mit aktiviertem BuildOnly-Modus werden Dateien nicht neu 
                        **  geschrieben, selbst wenn die Inhalte verändert sind. */
                        $color = CodeGenerator::getOutputColor('skipped');
                        $message = 'Skipping changed file';

                        self::$_resultCounters['skipped']++;

                    } else {
                        $color = CodeGenerator::getOutputColor('replacing');
                        $message = 'Replacing file';

                        self::$_resultCounters['replaced']++;
                    }
                } else {
                    $color = CodeGenerator::getOutputColor('creating');
                    $message = 'Creating new file';

                    self::$_resultCounters['created']++;
                }
            } else {
                /*
                **  Ausgaben für die build-Dateien werden nur ausgegeben, wenn der
                **  BuildOnly-Modus aktiviert ist. */
                if ($this->_buildOnlyMode === true) {
                    if (file_exists($file)) {
                        $color = CodeGenerator::getOutputColor('replacing');
                        $message = 'Replacing build file';

                        self::$_resultCounters['replaced']++;

                    } else {
                        $color = CodeGenerator::getOutputColor('creating');
                        $message = 'Creating new build file';

                        self::$_resultCounters['created']++;
                    }
                }
            }
        }
        return new ConsoleMessageDTO($color, $message);

    } // _getConsoleMessage()


    /**
     *
     * @param       Package $Package
     *
     * @param       string $typeName
     *
     * @return 	    bool
     *
     * @version     1.0.0 / 2025-07-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _hasDerivedModel(Package $Package, string $typeName): bool
    {
        $baseDir = $Package->getBaseDir();
        $basePath = $Package->getBasePath();

        return file_exists($basePath.'/'.$baseDir.'/Models/'.$typeName.'.php');

    } // _hasDerivedModel()


    /**
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2024-11-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _init(): self
    {
        if (self::$_startTime === null) {
            self::$_startTime = date('Y-m-d H.i.s');
        }
        return $this;

    } // _init()


    /**
     *
     * @return 	    $this
     *
     * @version     2.2.0 / 2025-11-19
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _initDefaultReplacements(Package $Package): self
    {
        $authors = $Package->getAuthors();
        $copyright = $Package->getCopyright();
        $devStart = $Package->getDevStart();
        $currentYear = (int) date('Y');

        if ($devStart === $currentYear) {
            $copyright = $devStart.' '.$copyright;

        } elseif ($devStart === ($currentYear - 1)) {
            $copyright = $devStart.', '.$currentYear.' '.$copyright;

        } else {
            $copyright = $devStart.' - '.$currentYear.' '.$copyright;
        }
        $this->_defaultReplacements = [
            'authors' => $authors,
            'code-generator-tag' => CodeGenerator::CODE_GENERATOR_TAG
                .' V'.CodeGenerator::CODE_GENERATOR_VERSION
                .' Δ {{ hash }} Δ DYNAMICALLY GENERATED FILE'
                .' - Do not modify this line! Remove it completely to protect the file from changes.',
            'copyright' => $copyright,
            'doc-package' => $Package->getName(),
        ];
        return $this;

    } // _initDefaultReplacements()


    /**
     *
     * @param       string $fileContents
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _initFileHash(string $fileContents)
    {
        $this->_fileHash = md5($fileContents);

        return $this;

    } // _initFileHash()


    /**
     * Initialisiert die Objektvariablen $_generatedDate und $_generatedTimestamp mit Werten aus der Productive-Datei
     *
     * @return 	    GeneratedDTO|null
     *
     * @version     1.1.0 / 2025-11-23
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _initGenerated(string $file): GeneratedDTO|null
    {
        $this->_generatedDate = null;
        $this->_generatedTimestamp = null;

        $GeneratedDTO = null;

        $productiveFile = str_replace($this->_paths[self::CONTEXT_BUILD], $this->_paths[self::CONTEXT_PRODUCTIVE], $file);

        if (file_exists($productiveFile)) {
            $GeneratedDTO = $this->_determineFileGenerated($productiveFile);

            if ($GeneratedDTO->timestamp !== null) {
                $this->_generatedDate = $GeneratedDTO->date;
                $this->_generatedTimestamp = $GeneratedDTO->timestamp;
            }
        }
        return $GeneratedDTO;

    } // _initGenerated()


    /**
     *
     * @return 	    $this
     *
     * @version     1.2.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _initTraits(): self
    {
        $traits = $this->_Type->getTraits(static::$_generator);

        if ($traits !== null) {
            foreach ($traits as $name => $class) {
                if (is_numeric($name)) {
                    $this->_addTrait($class);

                    continue;
                }
                $this->_addTrait([$name => $class]);
            }
        }
        return $this;

    } // _initTraits()


    /**
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareTraitReplacement(): string
    {
        $replacement = '';

        if (!empty($this->_traits)) {
            ksort($this->_traits);
            /*
            **  Ersetzungs-String für die Use-Statements zusammenstellen. */
            $padding = str_repeat(' ', 4);

            foreach ($this->_traits as $trait) {
                $replacement .= "\n".$padding.'use '.$trait.';';
            }
            $replacement .= "\n\n";
        }
        return $replacement;

    } // _prepareTraitReplacement()


    /**
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2024-10-22
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareUseReplacement(): string
    {
        $replacement = '';

        if (!empty($this->_uses)) {
            ksort($this->_uses);
            /*
            **  Ersetzungs-String für die Traits zusammenstellen. */
            foreach ($this->_uses as $use) {
                $replacement .= "\n".'use '.$use.';';
            }
            $replacement .= "\n\n";
        }
        return $replacement;

    } // _prepareUseReplacement()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-04-10
     * @history     BaseGenerator::_initUses(), 1.2.0 / 2025-03-04
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _prepareUses(): self
    {
        $uses = $this->_Type->getUses(static::$_generator);

        if ($uses !== null) {
            foreach ($uses as $class) {
                $this->_addUse($class);
            }
        }
        return $this;

    } // _prepareUses()


    /**
     *
     * @param       string $use
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-04-16
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _removeUse(string $use): self
    {
        unset($this->_uses[$use]);

        return $this;

    } // _removeUse()


    /**
     *
     * @param       array $authors
     *
     * @param       string $fileContents
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-04-05
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _replaceAuthorPlaceholder(array $authors, string $fileContents): string
    {
        for ($i = 0; $i < 2; $i++) {
            $placeholder = 'author-'.$i;

            $j = 0;
            $padding = str_repeat(' ', 4 * $i);
            $replacement = '';

            foreach ($authors as $author) {
                if ($j === 0) {
                    $replacement .= $author;
                } else {
                    $replacement .= "\n".$padding.' * @author      '.$author;
                }
                $j++;
            }
            if ($i === 0) {
                $fileContents = self::replacePlaceholder('author', $replacement, $fileContents);
            }
            $fileContents = self::replacePlaceholder($placeholder, $replacement, $fileContents);
        }
        return $fileContents;

    } // _replaceAuthorPlaceholder()


    /**
     * Führt Standard-Textersetzungen durch
     *
     * @param       Package $Package
     *
     * @param       string $fileContents
     *
     * @param       array $additionalReplacements
     *
     * @return 	    string
     *
     * @version     3.2.0 / 2025-04-05
     * @history     BaseGenerator::_makeDefaultReplacements(), 1.1.0 / 2024-10-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _replaceDefaults(Package $Package, string $fileContents, array $additionalReplacements): string
    {
        if ($this->_defaultReplacements === null) {
            $this->_initDefaultReplacements($Package);
        }
        $fileContents = $this->_replaceAuthorPlaceholder($this->_defaultReplacements['authors'], $fileContents);

        foreach ($this->_defaultReplacements as $placeholder => $replacement) {
            if ($placeholder === 'authors') {
                continue;
            }
            if (isset($additionalReplacements[$placeholder])) {
                $replacement = $additionalReplacements[$placeholder];

                unset($additionalReplacements[$placeholder]);
            }
            $fileContents = self::replacePlaceholder($placeholder, $replacement, $fileContents);
        }
        foreach ($additionalReplacements as $placeholder => $replacement) {
            $fileContents = self::replacePlaceholder($placeholder, $replacement, $fileContents);
        }
        $this->_initFileHash($fileContents);

        return $fileContents;

    } // _replaceDefaults()


    /**
     *
     * @param       string $fileContents
     *
     * @return 	    string
     *
     * @version     1.1.0 / 2025-05-20
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _replaceFinal(string $fileContents): string
    {
        /*
        **  Nachfolgende Ersetzungen dürfen erst ausgeführt werden wenn der FileHash ermittelt wurde, weil sie
        **  bei jeder Erzeugung anders sind und den Hash verändern würden. */
        $fileContents = self::replacePlaceholder('date', $this->_generatedDate ?? date('Y-m-d'), $fileContents);
        $fileContents = self::replacePlaceholder('generated', $this->_generatedTimestamp ?? date('Y-m-d H:i:s'), $fileContents);
        $fileContents = self::replacePlaceholder('hash', $this->_fileHash, $fileContents);

        $this->_generatedDate = null;
        $this->_generatedTimestamp = null;

        return $fileContents;

    } // _replaceFinal()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-02-12
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _resetUses(): self
    {
        $this->_uses = [];

        return $this;

    } // _resetUses()


    /**
     *
     * @param       Package $Package
     *
     * @param       string $fileName
     *
     * @param       string $fileContents
     *
     * @param       bool $replaceOriginalFile
     *
     * @return 	    $this
     *
     * @version     5.4.0 / 2025-11-23
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _writeFileContents(Package $Package, string $fileName, string $fileContents, bool $replaceOriginalFile = false): self
    {
        $this->_buildOnlyMode = (bool) (config('code-generator.test-mode') ?? $this->_buildOnlyMode);

        $packageName = $Package->getName();
        /*
        **  Dieser Methodenaufruf muss vor der Überprüfung der Datei mit _checkFileWriteStatus()
        **  erfolgen, weil in der Methode der FileHash gebildet wird. */
        $fileContents = $this->_replaceDefaults($Package, $fileContents, [
            'doc-package' => $packageName
        ]);
        $subPath = $this->_baseDir;

        if ($this->_filePath !== null) {
            $subPath .= '/'.$this->_filePath;
        }
        $outputSubPath = $subPath;

        $this->_paths = [
            self::CONTEXT_BUILD      => CodeGenerator::getBuildPath($Package, $subPath),
            self::CONTEXT_PRODUCTIVE => CodeGenerator::getBasePath($Package, $subPath),
        ];
        $generatingPaths = [
            self::CONTEXT_BUILD => $this->_paths[self::CONTEXT_BUILD],
        ];
        if ($this->_buildOnlyMode === false) {
            $generatingPaths[self::CONTEXT_PRODUCTIVE] = $this->_paths[self::CONTEXT_PRODUCTIVE];
        }
        if ($Package->isForProject() === false) {
            $outputSubPath = '['.$packageName.']/'.$outputSubPath;
        }
        foreach ($generatingPaths as $context => $path) {
            $file = $path.'/'.$fileName;

            $ignoreReason = null;
            /*
            **  Prüfen ob die Datei neu erzeugt werden muss und auch ob sie überhaupt
            **  überschrieben werden darf. */
            if ($this->_checkFileWriteStatus($Package, $context, $file, $ignoreReason, $replaceOriginalFile) === false) {
                $ConsoleMessage = $this->_getConsoleMessage(false, $context, $file, $ignoreReason);

                if ($ConsoleMessage->message !== null) {
                    CodeGenerator::output($ConsoleMessage->message, bullet: true, color: $ConsoleMessage->color, file: $outputSubPath.'/'.$fileName);
                }
                continue;
            }
            if (!file_exists($path)) {
                mkdir($path, 0777, true);
            }
            $ConsoleMessage = $this->_getConsoleMessage(true, $context, $file, $ignoreReason);

            if ($ConsoleMessage->message !== null) {
                CodeGenerator::output($ConsoleMessage->message, bullet: true, color: $ConsoleMessage->color, file: $outputSubPath.'/'.$fileName);
            }
            /*
            **  Wenn der globale BuildOnly-Modus aktiv ist, dann werden bestehende Dateien nicht
            **  überschrieben, selbst wenn geänderte Inhalte verfügbar sind. */
            if (file_exists($file) && config('code-generator.build-only-mode') === true) {
                continue;
            }
            /*
            **  Backup der Datei erstellen. */
            $backupPath = CodeGenerator::getBackupPath(self::$_startTime.'/'.$subPath);

            if (!file_exists($backupPath)) {
                mkdir($backupPath, 0777, true);
            }
            if (file_exists($file)) {
                copy($file, $backupPath.'/'.$fileName);
            }
            $fileContents = $this->_replaceFinal($fileContents);
            /*
            **  Datei schreiben. */
            file_put_contents($file, $fileContents);
        }
        return $this;

    } // _writeFileContents()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       null|string $group
     *
     * @return 	    array|int
     *
     * @version     1.0.1 / 2025-11-23
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getResultCounters(null|string $group = null): array|int|null
    {
        if ($group === null) {
            return self::$_resultCounters;
        }
        if (!isset(self::$_resultCounters[$group])) {
            return null;
        }
        return self::$_resultCounters[$group];

    } // getResultCounters()


    /**
     * Ersetzt Platzhalter durch die übergebene Zeichenkette
     *
     * @param       string $placeholder
     *
     * @param       string $replace
     *
     * @param       string $subject
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2024-10-06
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function replacePlaceholder(string $placeholder, string $replace, string $subject): string
    {
        $placeholder = [
            '{{'.$placeholder.'}}',
            '{{ '.$placeholder.' }}',
        ];
        return str_replace($placeholder, $replace, $subject);

    } // replacePlaceholder()


} // abstract class BaseGenerator {}
