# Filler Service

A JSON-RPC 2.0 connector for the _Filler_ executable.

## Installation

As the package is not in a public repository, the repository must be registered in the first installation step. To avoid having to register a separate repository for each private package, selected packages are made available via a central repository:

`composer config repositories.bplan/composer-packages composer https://satis.bplan.solutions`

The package can then be installed:

`composer require bplan-components/filler-service`

## Configuration

A few global settings can only be made in the .env file.

```toml
FILLER_BINARY_PATH = "/path/to/filler.exe"
```
Defines the path to the Filler executable. Defaults to `/.server/bin/filler/filler.exe`.

```toml
FILLER_ROUTE_PREFIX = terminal
```
Defines a route context for the api route. The default value is `terminal`, which results in the route `/api/terminal/filler`.

### Publishing the configuration file

The configuration file must be published with the following command:

`php artisan vendor:publish --provider="BplanComponents\FillerService\PackageServiceProvider" --tag=config`

This copies the file `filler-service.php` into the configuration directory of the project, which is required to define the settings for the layouts.
Example:

```php

return [
    'ini-files' => [
        'visitor-pass' => 'visitor-pass.ini',
    ],
    'settings' => [
        'visitor-pass' => [
            'date-time-format' => 'd.m.Y H:i',
            'timezone' => 'Europe/Berlin',
            'timezone-offset' => 0,
        ],
    ],
    'types' => \[
        'visitor-pass' => \App\Filler\Preparers\VisitorPassPreparer::class,
    ],
];

```
The keys in all sections (here `visitor-pass`) must be identical and must correspond to the type designation specified in the request. It must also be used in the _Filler's_ ini file as the final directory name of the target path (e.g. `/path-to-project/storage/app/public/filler/visitor-pass`).

The names of the ini files for the _Filler_ are specified in the `ini-files` section. The ini file must be located in the directory defined in the configuration file `filler.ini` under `InitialPath` (default value is `.\InitialCustomer`).

In `settings`, any values required for the preparation of the data can be specified. The use of this section is optional.
Settings that are defined here are available in the "Preparer" class in the object variable `$this->_settings`.

The relevant FillerPreparer class must be specified under `types`. Each FillerPreparer class should extend the class `BplanComponents\FillerService\Preparers\BasePreparer` and must implement the interface `BplanComponents\FillerService\Contracts\DataPreparerInterface`.


## Usage

Example request:

```json

{
  "id": 99,
  "jsonrpc": "2.0",
  "method": "filler_create",
  "params": {
    "data": {
        "appointmentId": 2,
        "terminalId": 1
    },
    "type": "visitor-pass"
  }
}

```
`params.data` contains the request data that is passed on to the `prepare` method of the FillerPreparer class in the `$params` parameter. The value of `type` must correspond to a key record in the configuration file.

Example class:

```php
<?php

declare(strict_types=1);

namespace App\Filler\Preparers;

use App\Repositories\VisitAppointmentRepository;
use App\Repositories\VisitEmployeeRepository;
use App\Repositories\VisitVisitorRepository;
use BplanComponents\FillerService\Contracts\DataPreparerInterface;
use BplanComponents\FillerService\DTOs\PreparedDataDTO;
use BplanComponents\FillerService\Preparers\BasePreparer;
use Carbon\Carbon;

/**
 * Filler Preparer Class
 */
class VisitorPassPreparer extends BasePreparer implements DataPreparerInterface
{

    /**
     *
     * @param       array $params
     *
     * @return      PreparedDataDTO
     */
    public function prepare(array $params): PreparedDataDTO
    {
        // Determination and compilation of the data for the Filler.
        $Appointment = VisitAppointmentRepository::getById($params['appointmentId']);

        $Employee = null;

        if ($Appointment->main_employee_id !== null) {
            $Employee = VisitEmployeeRepository::getById($Appointment->main_employee_id);
        }
        $Visitor = VisitVisitorRepository::getById($Appointment->main_visitor_id);

        $checkIn = Carbon::now()->setTimezone($this->_settings['timezone'])
            ->addHours($this->_settings['timezone-offset'])
            ->format($this->_settings['date-time-format']);

        $validFrom = $Appointment->valid_from->setTimezone($this->_settings['timezone'])
            ->addHours($this->_settings['timezone-offset'])->format($this->_settings['date-time-format']);

        $validUntil = $Appointment->valid_until->setTimezone($this->_settings['timezone'])
            ->addHours($this->_settings['timezone-offset'])->format($this->_settings['date-time-format']);

        $jsonData = [
            'visitors' => [
                [
                    'id' => $Appointment->id,
                    'checkIn' => $checkIn,
                    'company' => $Visitor->company,
                    'firstName' => $Visitor->first_name,
                    'fullName' => $Visitor->getFullName(reverted: true),
                    'lastName' => $Visitor->last_name,
                    'licensePlate' => $Appointment->license_plate,
                    'hostFirstName' => ($Employee === null ? '' : $Employee->first_name),
                    'hostFullName' => ($Employee === null ? '' : $Employee->getFullName()),
                    'hostLastName' => ($Employee === null ? '' : $Employee->last_name),
                    'token' => $Appointment->token,
                    'validFrom' => $validFrom,
                    'validUntil' => $validUntil,
                ],
            ]
        ];
        // Here comes the important part, the transfer of the data to the FillerService.
        $this->_Dto->jsonData = $jsonData;
        $this->_Dto->jsonFile = $this->_writeQueueFile($Appointment->token.'.json', $jsonData);

        return $this->_Dto;

    } // prepare()

} // class VisitorPassPreparer extends BasePreparer implements DataPreparerInterface {}

```

### Generate Preparer Classes

This package provides a custom Artisan command to generate "Preparer" classes, typically used for preparing or transforming data within your application.

#### Command

```bash
php artisan make:filler-preparer <Name> [--force]
```

#### Behavior

* Creates a class inside `app/Filler/Preparers`.
* The class and filename can be given in studly case (`StaffBadge`) or slug case (`staff-badge`).
* The class and filename will always end with `Preparer`, even if the user omits it.
   Example: `php artisan make:preparer StaffBadge` → `app/Filler/Preparers/StaffBadgePreparer.php`
* If the file already exists, it will **not** be overwritten unless `--force` is used.


## Dependencies
* "php": "*"
* "sajya/server": "*"
