<?php
/**
 * DTO Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-components/filler-service
 * @subpackage  FillerPreparers
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

declare(strict_types=1);

namespace BplanComponents\FillerService\Preparers;

use BplanComponents\FillerService\DTOs\PreparedDataDTO;
use BplanComponents\FillerService\Services\FillerService;
use Illuminate\Filesystem\LocalFilesystemAdapter;
use Illuminate\Support\Facades\Storage;
use RuntimeException;


/**
 * DTO Class
 *
 * @version     1.0.0 / 2025-05-07
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
abstract class BasePreparer
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_dirName
     */
    protected string $_dirName = FillerService::FILLER_DIR_NAME;


    /**
     * @var     PreparedDataDTO $_Dto
     */
    protected PreparedDataDTO $_Dto;


    /**
     * @var     LocalFilesystemAdapter $_Storage
     */
    protected LocalFilesystemAdapter $_Storage;


    /**
     * @var     array|null $_settings
     */
    protected array|null $_settings;


    /**
     * @var     string $_subPath
     */
    protected string $_subPath;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $type
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(string $type)
    {
        $this->_init($type);

    } // __construct()


    /**
     *
     * @param       string $type
     *
     * @throws      RuntimeException
     *
     * @return      $this
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _init(string $type): self
    {
        $iniFileName = config('filler-service.ini-files.'.$type);

        $this->_settings = config('filler-service.settings.'.$type);
        $this->_Dto = new PreparedDataDTO($type, $iniFileName);
        $this->_Storage = Storage::disk('local');
        $this->_subPath = 'queue/'.$this->_dirName.'/'.$type;

        $targetPath = $this->_Storage->path($this->_subPath);

        if ($this->_Storage->exists($this->_subPath) === false) {
            if (mkdir($targetPath, 0777, true) === false) {
                throw new RuntimeException('Could not create target directory ['.$targetPath.']');
            }
        } elseif (is_writable($targetPath) === false) {
            throw new RuntimeException('Target directory  ['.$targetPath.'] is not writable');
        }
        return $this;

    } // _init()


    /**
     *
     * @param       string $fileName
     *
     * @param       array $fileData
     *
     * @throws      RuntimeException
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    protected function _writeQueueFile(string $fileName, array $fileData): string
    {
        if ($this->_Storage->put($this->_subPath.'/'.$fileName, json_encode($fileData, JSON_PRETTY_PRINT)) === false) {
            throw new RuntimeException('Could not create file ['.$this->_subPath.'/'.$fileName.']');
        }
        return $this->_Storage->path($this->_subPath.'/'.$fileName);

    } // _writeQueueFile()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // abstract class BasePreparer {}
