<?php
/**
 * Service Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-components/filler-service
 * @subpackage  Services
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 bplan-solutions GmbH & Co. KG <https://www.bplan-solutions.de/>
 * /Δ\
 */

declare(strict_types=1);

namespace BplanComponents\FillerService\Services;

use BplanComponents\FillerService\Dispatchers\DataPreparerDispatcher;
use BplanComponents\FillerService\DTOs\PreparedDataDTO;
use BplanComponents\FillerService\Enums\ImageOutputFormat;
use BplanComponents\FillerService\Enums\JsonDataType;
use BplanComponents\FillerService\Enums\OutputFileIndexPosition;
use BplanModules\VisitorManagement\Repositories\VisitAppointmentRepository;
use BplanModules\VisitorManagement\Repositories\VisitEmployeeRepository;
use BplanModules\VisitorManagement\Repositories\VisitVisitorRepository;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use RuntimeException;
use Sajya\Server\Annotations\Param;
use Sajya\Server\Annotations\Result;
use Sajya\Server\Http\Request;
use Sajya\Server\Procedure;


/**
 * Service Class
 *
 * @version     1.0.0 / 2025-05-07
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class FillerService
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    const FILLER_DIR_NAME = 'filler';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     string $_binaryPath
     */
    private string $_binaryPath;


    /**
     * @var     string $_command
     */
    private string $_command;


    /**
     * @var     string $_dirName
     */
    private string $_dirName = self::FILLER_DIR_NAME;


    /**
     * @var     string $_imageOutputFormat
     */
    private string $_imageOutputFormat;


    /**
     * @var     string $_jsonDataToken
     */
    private string $_jsonDataToken;


    /**
     * @var     JsonDataType $_JsonDataType
     */
    private JsonDataType $_JsonDataType;


    /**
     * @var     string $_iniFileName
     */
    private string $_iniFileName;


    /**
     * @var     string $_outputFileExtension
     */
    private string $_outputFileExtension;


    /**
     * @var     null|string $_outputFileName
     */
    private null|string $_outputFileName;


    /**
     * @var     null|string $_outputFileNameIndex
     */
    private null|string $_outputFileNameIndex;


    /**
     * @var     OutputFileIndexPosition $_OutputFileNameIndexPosition
     */
    private OutputFileIndexPosition $_OutputFileNameIndexPosition;


    /**
     * @var     string $_outputPath
     */
    private string $_outputPath;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       array $dataSet
     *
     * @param       int|null $key
     *
     * @return 	    string
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _buildOutputFileName(array $dataSet, int|null $key = null): string
    {
        $fileName = '';

        if ($this->_outputFileName !== null) {
            $fileName = $this->_outputFileName;
        }
        if ($this->_outputFileNameIndex !== null) {
            if ($this->_JsonDataType === JsonDataType::Array) {
                $postfix = '';
                $prefix = '';

                if ($this->_outputFileNameIndex === 'x' || $this->_outputFileNameIndex === '#') {
                    if ($this->_OutputFileNameIndexPosition === OutputFileIndexPosition::Before) {
                        $prefix = $key.'_';
                    } else {
                        $postfix = '_'.$key;
                    }
                } else {
                    /*
                    **  Das funktioniert aktuell nur mit Angaben wie "visitors[].id". */
                    list($nul, $key) = explode('.', $this->_outputFileNameIndex);

                    if ($this->_OutputFileNameIndexPosition === OutputFileIndexPosition::Before) {
                        $prefix = $dataSet[$key].'_';
                    } else {
                        $postfix = '_'.$dataSet[$key];
                    }
                }
            } else {
                $postfix = '';
                $prefix = '';

                $key = $dataSet[$this->_outputFileNameIndex];

                if ($this->_OutputFileNameIndexPosition === OutputFileIndexPosition::Before) {
                    $prefix = $key.'_';
                } else {
                    $postfix = '_'.$key;
                }
            }
            $fileName = $prefix.$fileName.$postfix;
        }
        $fileName .= '.'.$this->_outputFileExtension;

        return $fileName;

    } // _buildOutputFileName()


    /**
     *
     * @param       PreparedDataDTO $PreparedData
     *
     * @throws      RuntimeException
     *
     * @return      $this
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(PreparedDataDTO $PreparedData): self
    {
        $this->_iniFileName = $PreparedData->iniFileName;

        $Storage = Storage::disk('local');
        /*
        **  Alle Verzeichnisnamen und Pfade sind für die Speicherung der Queue-Datei (Json). Lediglich der
        **  Verzeichnisname in "$dirName" wird auch für die Ausgabedateien des Layouters verwendet und muss
        **  in der ini-Datei genutzt werden. */
        $subPath = 'queue/'.$this->_dirName;

        $targetPath = $Storage->path($subPath);

        if ($Storage->exists($subPath) === false) {
            if (mkdir($targetPath, 0777, true) === false) {
                throw new RuntimeException('Could not create target directory ['.$subPath.']');
            }
        } elseif (is_writable($targetPath) === false) {
            throw new RuntimeException('Target directory ['.$subPath.'] is not writable');
        }
        $configBinary = config('filler-service-globals.binary');

        if($configBinary === null) {
            throw new RuntimeException('No binary configured');

        } elseif ($configBinary === false || file_exists($configBinary) === false) {
            throw new RuntimeException('Could not find binary ['.$configBinary.']');

        } elseif (is_executable($configBinary) === false) {
            throw new RuntimeException('['.$configBinary.'] is not executable');
        }
        $pathInfo = pathinfo($configBinary);

        $binaryFile = $pathInfo['basename'];
        $binaryFileName = $pathInfo['filename'];
        $this->_binaryPath = $pathInfo['dirname'];

        $this->_processIniFiles($this->_binaryPath, $binaryFileName);

        $this->_command = '"'.$binaryFile.'" '.$PreparedData->iniFileName.' "'.$PreparedData->jsonFile.'"';

        return $this;

    } // _init()


    /**
     *
     * @return      void
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _generate(): void
    {
        $cwd = getcwd();
        /*
        **  Ins Verzeichnis des Fillers wechseln. */
        chdir($this->_binaryPath);

        ob_start(); {
            $lastOutputLine = system($this->_command, $resultCode);

        } ob_end_clean();

        chdir($cwd);

        if ($resultCode !== false) {
            switch ($resultCode) {
                case 1:
                    throw new RuntimeException('Error while creating: error code ['.$resultCode.'], '.$lastOutputLine);
                    break;

                case 0:
                    if(!empty($lastOutputLine)) {
                        throw new RuntimeException('Error while creating: error code ['.$resultCode.'], '.$lastOutputLine);
                    }
                    // no break

                default:
                    // Success
            }
        }
        try {
            unlink($this->_queueFile);

        } catch (\Throwable $th) {
            //
        }
    } // _generate()


    /**
     * Liest die ini-Datei des Fillers und ermittelt Informationen
     *
     * @param       string $binaryPath
     *
     * @param       string $binaryFileName
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processIniFiles(string $binaryPath, string $binaryFileName): self
    {
        $binaryIniFile = $binaryPath.'/'.$binaryFileName.'.ini';

        $iniFile = realpath($binaryIniFile);

        if ($iniFile === false || file_exists($iniFile) === false) {
            throw new RuntimeException('Could not find ini file ['.$binaryIniFile.']');
        }
        $iniContents = parse_ini_file($iniFile, true);

        $customerIniPath = realpath($binaryPath.'/'.$iniContents['GLOBAL']['InitialPath']);

        $this->_processCustomerIniFile($customerIniPath, $this->_iniFileName);

        return $this;

    } // _processIniFiles()


    /**
     * Liest die ini-Datei des Customers und ermittelt Informationen
     *
     * @param       string $customerIniPath
     *
     * @param       string $iniFileName
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _processCustomerIniFile(string $customerIniPath, string $iniFileName): self
    {
        $customerIniFile = $customerIniPath.'/'.$iniFileName;

        $iniFile = realpath($customerIniFile);

        if ($iniFile === false || file_exists($iniFile) === false) {
            throw new RuntimeException('Could not find ini file ['.$customerIniFile.']');
        }
        $iniContents = parse_ini_file($iniFile, true);
        // todo     Hier ist noch zu berücksichtigen, dass es mehrere LayoutSections geben kann.
        $layoutSection = $iniContents['GLOBAL']['LayoutSection'];
        $outputFilenameIndexPosition = $iniContents[$layoutSection]['OutputFilenameIndexPosition'] ?? OutputFileIndexPosition::After->value;

        $this->_jsonDataToken = $iniContents[$layoutSection]['Token'];
        $this->_JsonDataType = JsonDataType::getByValue($iniContents[$layoutSection]['Type']);
        $this->_outputFileName = $iniContents[$layoutSection]['OutputFilename'] ?? null;
        $this->_outputFileNameIndex = $iniContents[$layoutSection]['OutputFilenameIndex'] ?? null;
        $this->_OutputFileNameIndexPosition = OutputFileIndexPosition::getByValue($outputFilenameIndexPosition) ?? OutputFileIndexPosition::After;
        // todo     Der OutputPath wird aktuell noch nicht verwendet.
        $this->_outputPath = $iniContents[$layoutSection]['OutputPath'];
        // todo     Hier kann auch ein kommaseparierter String mit mehreren Typen vorkommen.
        $this->_imageOutputFormat = $iniContents[$layoutSection]['GrafikOutputFormat'];

        $this->_outputFileExtension = ImageOutputFormat::{$this->_imageOutputFormat}->value;

        return $this;

    } // _processCustomerIniFile()


    /**
     *
     * @param       PreparedDataDTO $PreparedData
     *
     * @return      array|string
     *
     * @version     1.0.0 / 2025-05-07
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function handle(PreparedDataDTO $PreparedData): array|string
    {
        $this->_init($PreparedData);
        $this->_generate();

        $storageConfig = Storage::disk('public')->getConfig();

        if ($this->_JsonDataType === JsonDataType::Array) {
            $result = [];

            foreach ($PreparedData->jsonData[$this->_jsonDataToken] as $key => $dataSet) {
                $outputFileName = $this->_buildOutputFileName($dataSet, $key);

                $result[] =  $storageConfig['url'].'/'.$this->_dirName.'/'.$PreparedData->layoutType.'/'.$outputFileName;
            }
        } else {
            $result = [
                $outputFileName = $this->_buildOutputFileName($PreparedData->jsonData)
            ];
        }
        return $result;

    } // handle()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class FillerService {}
