<?php
/**
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-components/ics-file
 * @subpackage  ICS
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024, 2025 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanComponents\ICS\ICS;


use DateTime;


/**
 *
 * @version     1.1.0 / 2025-01-13
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class Event
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string DATE_FORMAT
     */
    protected const DATE_FORMAT = 'Y-m-d';


    /**
     * @var         string DATETIME_FORMAT
     */
    protected const DATETIME_FORMAT = 'Ymd\THis\Z';


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         array $_attendees
     */
    private $_attendees = [];


    /**
     * @var         array $_availableProperties
     */
    private array $_availableProperties = [
        'categories',
        'description',
        'dtend',
        'dtstart',
        'location',
        'priority',
        'sequence',
        'summary',
        'uid',
        'url',
    ];


    /**
     * @var         array $_organizer
     */
    private $_organizer;


    /**
     *
     */
    protected array $_properties = [];


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       array $properties
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(array $properties = [])
    {
        $this->setProperty($properties);

    } // __construct()


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _buildPropertyStrings(): array
    {
        $propertyStrings = [];

        foreach ($this->_getPreparedProperties() as $property => $value) {
            $propertyStrings[] = $property.':'.$value;
        }
        return $propertyStrings;

    } // _buildPropertyStrings()


    /**
     *
     * @param       string $string
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _escapeString(string $string): string
    {
        return preg_replace('/([\,;])/', '\\\$1', $string);

    } // _escapeString()


    /**
     *
     * @param       string $timestamp
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _formatTimestamp(string $timestamp): string
    {
        $day_light_start = strtotime('last sunday of '.date('Y').'-'.config('ics.daylight-saving-start-month'));
        $day_light_end = strtotime('last sunday of '.date('Y').'-'.config('ics.daylight-saving-end-month'));

        $DateTime = new DateTime($timestamp);

        if (config('ics.daylight-saving') === true
          && $DateTime->format(self::DATE_FORMAT) >= date(self::DATE_FORMAT, $day_light_start)
          && $DateTime->format(self::DATE_FORMAT) <= date(self::DATE_FORMAT, $day_light_end)) {
            $DateTime->modify('-'.config('ics.daylight-saving-offset'));
        }
        return $DateTime->format(self::DATETIME_FORMAT);

    } // _formatTimestamp()


    /**
     *
     * @param       array $attendee
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getAttendeeString(array $attendee): string
    {
        return 'ATTENDEE'
            .';CN='.$attendee['name']
            .';CUTYPE=INDIVIDUAL'
            .';PARTSTAT='.($attendee['accepted'] === true ? 'ACCEPTED' : 'NEEDS-ACTION')
            .';ROLE='.($attendee['required'] === true ? 'REQ-PARTICIPANT' : 'NON-PARTICIPANT')
            .';RSVP=TRUE:mailto:'.$attendee['email'];

    } // _getAttendeeString()


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getPreparedProperties(): array
    {
        $properties = [];

        foreach ($this->_properties as $property => $value) {
            $properties[strtoupper($property)] = $value;
        }
        $properties['DTSTAMP'] = $this->_formatTimestamp('now');
        $properties['UID'] = $properties['UID'] ?? uniqid();

        return $properties;

    } // _getPreparedProperties()


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _getPropertyStrings(): array
    {
        $propertyStrings = $this->_buildPropertyStrings();

        if (!empty($this->_organizer)) {
            $propertyStrings[] = $this->_getOrganizerString();
        }
        if (!empty($this->_attendees)) {
            foreach ($this->_attendees as $attendee) {
                $propertyStrings[] = $this->_getAttendeeString($attendee);
            }
        }
        return $propertyStrings;

    } // _getPropertyStrings()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function _getOrganizerString(): string
    {
        if ($this->_organizer === null) {
            return '';
        }
        return 'ORGANIZER'
            .';CN='.$this->_organizer['name']
            .':MAILTO:'.$this->_organizer['email'];

    } // _getOrganizerString()


    /**
     *
     * @param       string $property
     *
     * @param       null|string $value
     *
     * @return      string
     *
     * @version     2.0.0 / 2025-01-13
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _sanitizeValue(string $property, null|string $value): null|string
    {
        switch ($property) {
            case 'dtend':
            case 'dtstamp':
            case 'dtstart':
                $value = $this->_formatTimestamp($value);
                break;

            default:
                if ($value !== null) {
                    $value = $this->_escapeString($value);
                }
        }
        return $value;

    } // _sanitizeValue()


    /**
     *
     * @param       string $name
     *
     * @param       string $email
     *
     * @param       bool $accepted
     *
     * @param       bool $required
     *
     * @return      $this
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function addAttendee(string $name, string $email, bool $accepted = false, bool $required = true): self
    {
        $this->_attendees[] = [
            'accepted' => $accepted,
            'email' => $email,
            'name' => $name,
            'required' => $required,
        ];
        return $this;

    } // addAttendee()


    /**
     *
     * @param       string $name
     *
     * @param       string $email
     *
     * @return      $this
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setOrganizer(string $name, string $email): self
    {
        $this->_organizer = [
            'email' => $email,
            'name' => $name,
        ];
        return $this;

    } // setOrganizer()


    /**
     *
     * @param       array|string $property
     *
     * @param       null|string $value
     *
     * @return      $this
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setProperty(array|string $property, null|string $value = null): self
    {
        if (is_array($property)) {
            foreach ($property as $name => $value) {
                $this->setProperty(strtolower($name), $value);
            }
        } else {
            if (in_array($property, $this->_availableProperties)) {
                $this->_properties[$property] = $this->_sanitizeValue($property, $value);
            }
        }
        return $this;

    } // setProperty()


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2024-12-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function toString(): string
    {
        $string = 'BEGIN:VEVENT'
            ."\r\n".implode("\r\n", $this->_getPropertyStrings())
            ."\r\n".'END:VEVENT';

        return $string;

    } // toString()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class Event {}
