<?php
/**
 * Procedure Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-vega-rpc
 * @subpackage  Procedures
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

declare(strict_types=1);

namespace Bplan\LaravelVegaRpc\Http\Procedures;


use Bplan\LaravelVegaRpc\Enums\CheckResultCode;
use Sajya\Server\Annotations\Param;
use Sajya\Server\Annotations\Result;
use Sajya\Server\Http\Request;
use Sajya\Server\Procedure;


use Bplan\LaravelVegaRpc\Enums\CheckResultStatus;
use Bplan\LaravelVegaRpc\Models\VegaLicensePlate;
use Bplan\LaravelVegaRpc\Repositories\VegaLicensePlateRepository;
use Bplan\LaravelVegaRpc\Services\VegaMatchLogService;


/**
 * Procedure Class
 *
 * @version     2.0.0 / 2024-09-28
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class MatchProcedure extends Procedure
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * The name of the procedure that is used for referencing.
     *
     * @var     string
     */
    public static string $name = 'match';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       string $licensePlate
     *
     * @param       bool $accessGranted
     *
     * @param       array|null|string $result
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2024-09-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _prepareCheckResult(bool $accessGranted, string $licensePlate, array|null|string $result): array
    {
        if ($result === null) {
            $result = ($accessGranted === true)
                ? VegaLicensePlateRepository::prepareCheckResult(CheckResultStatus::OK, $licensePlate)
                : VegaLicensePlateRepository::prepareCheckResult(CheckResultStatus::Forbidden, $licensePlate);

        } elseif (is_array($result)) {
            if ($accessGranted === true) {
                if (empty($result['code'])) {
                    $result['code'] = CheckResultCode::OK->value;
                }
                if (empty($result['status'])) {
                    $result['status'] = CheckResultStatus::OK->name;
                }
                if (empty($result['message'])) {
                    $result['message'] = __('vega-rpc::vega-rpc.check-result.message.OK', ['plate' => $licensePlate]);
                }
            } else {
                if (empty($result['code'])) {
                    $result['code'] = CheckResultCode::Forbidden->value;
                } else {
                    $result['code'] = $result['code'];
                }
                if (empty($result['status'])) {
                    $statuses = array_combine(array_column(CheckResultCode::cases(), 'value'), array_column(CheckResultCode::cases(), 'name'));
                    $result['status'] = $statuses[$result['code']];
                }
                if (empty($result['message'])) {
                    $result['message'] = __('vega-rpc::vega-rpc.check-result.message.'.CheckResultStatus::{$result['status']}->name, ['plate' => $licensePlate]);
                }
            }
        } else {
            $message = $result;

            $result = ($accessGranted === true)
                ? VegaLicensePlateRepository::prepareCheckResult(CheckResultStatus::OK, $licensePlate)
                : VegaLicensePlateRepository::prepareCheckResult(CheckResultStatus::Forbidden, $licensePlate);

            $result['message'] = $message;
        }
        return $result;

    } // _prepareCheckResult()


    /**
     * Execute the procedure.
     *
     * @Param(name="class",   value="required|string")
     * @Param(name="country", value="required|string")
     * @Param(name="date",    value="required|string")
     * @Param(name="imageid", value="required|string")
     * @Param(name="plate",   value="required|string")
     * @Param(name="time",    value="required|string")
     *
     * @Result(name="description", value="nullable|string")
     * @Result(name="match",       value="required|boolean")
     * @Result(name="match_data",  value="nullable|string")
     *
     * @param       Request $Request
     *
     * @return      array
     *
     * @version     2.0.0 / 2024-09-28
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function list(Request $Request): array
    {
        $requestId = $Request->getId();
        $requestParams = $Request->getParams()->toArray();

        $licensePlate = $requestParams['plate'];

        $matchData = '';

        foreach ($requestParams as $key => $value) {
            $matchData .= '<'.$key.'>'.$value.'</'.$key.'>';
        }
        $result = null;

        if ($licensePlate === 'NOPLATE”' || $licensePlate === 'NOTREAD') {
            $accessGranted = false;

            $result = VegaLicensePlateRepository::prepareCheckResult(CheckResultStatus::BadRequest, $licensePlate);

        } else {
            $repositoryClass = VegaLicensePlate::determineRepositoryClass();
            $permissionMethod = $repositoryClass.'::checkPermission';

            $accessGranted = $permissionMethod($licensePlate, $result);
        }
        $result = $this->_prepareCheckResult($accessGranted, $licensePlate, $result);

        $logData = $requestParams;

        $timeParts = explode('-', $logData['time']);
        $microSeconds = array_pop($timeParts);
        $logData['time'] = implode(':', $timeParts).'.'.$microSeconds;

        $LogService = new VegaMatchLogService();

        $LogService->create(
            array_merge([
                'access_granted' => $accessGranted,
                'request_id' => $requestId,
                'result_code' => $result['code'],
                'result_status' => $result['status'],
                'rpc_params' => $requestParams,
            ],
            $logData)
        );
        return [
            'description' => $result['message'],
            'match' => $accessGranted,
            'match_data' => $matchData,
        ];
    } // list()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


} // class MatchProcedure extends Procedure {}
