<?php
/**
 * Repository Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan/laravel-vega-rpc
 * @subpackage  Repositories
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2024 Wassilios Meletiadis <http://www.bplan-solutions.de/>
 * /Δ\
 */

namespace Bplan\LaravelVegaRpc\Repositories;

use Bplan\LaravelVegaRpc\Enums\CheckResultCode;
use Bplan\LaravelVegaRpc\Enums\CheckResultStatus;
use Bplan\LaravelVegaRpc\Models\VegaLicensePlate;


/**
 * Repository Class
 *
 * @version     1.1.0 / 2024-11-09
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class VegaLicensePlateRepository implements VegaLicensePlateRepositoryInterface
{

/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Überprüft die Zufahrtsberechtigung für ein Kennzeichen
     *
     * @param       string $licensePlate
     *              Das zu überprüfende Kennzeichen.
     *
     * @param       array $requestParams
     *              Alle Request-Daten.
     *
     * @param       array|string & $result
     *              Falls der Parameter $result übergeben wurde, wird er mit einem Text gefüllt,
     *              der Details zum Ergebnis beinhaltet.
     *
     * @return      bool
     *
     * @version     2.0.0 / 2024-11-09
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function checkPermission(string $licensePlate, array $requestParams, array|string & $result = null): bool
    {
        if ($licensePlate === 'NOPLATE' || $licensePlate === 'NOTREAD') {
            $result = self::prepareCheckResult(CheckResultStatus::BadRequest, $licensePlate);

            return false;
        }
        $LicensePlate = VegaLicensePlate::where('license_plate', '=', $licensePlate)
            ->first();

        if ($LicensePlate === null) {
            $result = self::prepareCheckResult(CheckResultStatus::NotFound, $licensePlate);

            return false;

        } elseif ($LicensePlate->passage_permitted == false) {
            $result = self::prepareCheckResult(CheckResultStatus::Forbidden, $licensePlate);

            return false;
        }
        $result = self::prepareCheckResult(CheckResultStatus::OK, $licensePlate);

        return true;

    } // checkPermission()


    /**
     *
     * @param       CheckResultStatus $Status
     *
     * @param       string $licensePlate
     *
     * @return 	    array
     *
     * @version     1.0.1 / 2024-11-09
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function prepareCheckResult(CheckResultStatus $Status, string $licensePlate): array
    {
        return [
            'code' => CheckResultCode::{$Status->name}->value,
            'message' => __('vega-rpc::vega-rpc.check-result.message.'.$Status->name, ['plate' => $licensePlate]),
            'status' => $Status->name,
        ];
    } // prepareCheckResult()


} // class VegaLicensePlateRepository implements VegaLicensePlateRepositoryInterface {}
