<?php
/**
 * Connector Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-components/web-io-connect
 * @subpackage  Connectors
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 Wassilios Meletiadis <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanComponents\WebIoConnect\Connectors;


use BplanComponents\WebIoConnect\Api\Credentials;
use BplanComponents\WebIoConnect\Devices\BaseDevice;
use BplanComponents\WebIoConnect\Enums\ResponseFormat;
use BplanComponents\WebIoConnect\Enums\DeviceType;
use BplanComponents\WebIoConnect\Enums\State;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;


/**
 * Connector Class
 *
 * @version     1.0.0 / 2025-04-26
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class WebIoConnector
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         string $_authPass
     */
    protected string $_authPass;


    /**
     * @var         string $_authUser
     */
    protected string $_authUser;


    /**
     * @var         string $_baseUrl
     */
    protected string $_baseUrl;


    /**
     * @var         string $_Device
     */
    protected BaseDevice $_Device;


    /**
     * @var         DeviceType $_DeviceType
     */
    protected DeviceType $_DeviceType;


    /**
     * @var         string $_responseFormat
     */
    protected string $_responseFormat;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     */
    private static string $_fakerRouteBase = 'web-io-faker';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       DeviceType $DeviceType
     *
     * @param       string $baseUrl
     *
     * @param       ResponseFormat $ResponseFormat
     *              Aktuell wird nur das ResponseFormat "Json" unterstützt.
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(DeviceType $DeviceType, string $baseUrl, $ResponseFormat = ResponseFormat::Json)
    {
        /*
        **  Zum Beispiel "http://webio.example.com:8080" */
        $this->_DeviceType = $DeviceType;
        $this->_responseFormat = $ResponseFormat->value;
        $this->_baseUrl = $baseUrl.'/rest/'.$this->_responseFormat;

        $this->_init();

    } // __construct()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(): self
    {
        if (strtolower(config('app.env')) !== 'production') {
            $this->_baseUrl = request()->getSchemeAndHttpHost()
                .'/api/'.self::$_fakerRouteBase
                .'/rest/'.$this->_responseFormat
                .'/'.$this->_DeviceType->value;
        }
        $Credentials = new Credentials();

        if ($Credentials->hasCredentials() === true) {
            $this->_authPass = $Credentials->getPassword();
            $this->_authUser = $Credentials->getUserName();
        }
        $response = $this->readProcessImage('GET');

        $class = $this->_DeviceType->getClass();

        $this->_Device = new $class($response);

        return $this;

    } // _init()


    /**
     *
     * @param       string $method
     *
     * @param       string $route
     *
     * @param       string $payload
     *
     * @return      Response
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _sendRequest(string $method, null|string $route = null, null|string $payload = null): Response
    {
        $url = $this->_baseUrl;

        if ($route !== null) {
            $url .= '/'.$route;
        }
        $Request = Http::timeout(5);

        if ($this->_authPass !== null) {
            $Request = $Request->withDigestAuth($this->_authUser, $this->_authPass);
        }
        if ($method === 'POST') {
            return $Request->withBody($payload)->post($url)->json();
        }
        return $Request->get($url)->json();

    } // _sendRequest()


    /**
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getCounters(): Collection
    {
        return $this->_Device->getCounters();

    } // getCounters()


    /**
     *
     * @return 	    BaseDevice|null
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getDevice(): BaseDevice|null
    {
        return $this->_Device;

    } // getDevice()


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getInfo(): array
    {
        return $this->_Device->getInfo();

    } // getInfo()


    /**
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getInputs(): Collection
    {
        return $this->_Device->getInputs();

    } // getInputs()


    /**
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getOutputs(): Collection
    {
        return $this->_Device->getOutputs();

    } // getOutputs()


    /**
     * Liefert den State eines Inputs
     *
     * @param       int $output
     *
     * @param       bool $returnBool
     *
     * @return      bool|State|null
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readCounterState(int $counter, bool $returnBool = false): bool|State|null
    {
        if ($this->_Device->hasCounter() === false) {
            return null;
        }
        $route = 'iostate/counter/'.$input;

        $result = $this->_sendRequest('GET', $route);

        $bool = (bool) $result['iostate']['counter'][0]['state'];

        if ($returnBool === true) {
            return $bool;
        }
        return ($bool === true) ? State::On : State::Off;

    } // readCounterState()


    /**
     * Liefert den State eines Inputs
     *
     * @param       int $output
     *
     * @param       bool $returnBool
     *
     * @return      bool|State|null
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readInputState(int $input, bool $returnBool = false): bool|State|null
    {
        if ($this->_Device->hasInput() === false) {
            return null;
        }
        $route = 'iostate/input/'.$input;

        $result = $this->_sendRequest('GET', $route);

        $bool = (bool) $result['iostate']['input'][0]['state'];

        if ($returnBool === true) {
            return $bool;
        }
        return ($bool === true) ? State::On : State::Off;

    } // readInputState()


    /**
     * Liest das vollständige Prozessabbild aus
     *
     * @return      array|null
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readProcessImage(): array|null
    {
        return $this->_sendRequest('GET');

    } // readProcessImage()


    /**
     * Liefert den State eines Outputs
     *
     * @param       int $output
     *
     * @param       bool $returnBool
     *
     * @return      bool|State|null
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readOutputState(int $output, bool $returnBool = false): bool|State|null
    {
        if ($this->_Device->hasOutput() === false) {
            return null;
        }
        $route = 'iostate/output/'.$output;

        $result = $this->_sendRequest('GET', $route);

        $bool = (bool) $result['iostate']['output'][0]['state'];

        if ($returnBool === true) {
            return $bool;
        }
        return ($bool === true) ? State::On : State::Off;

    } // readOutputState()


    /**
     *
     * @param       int $input
     *
     * @return      Response
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function resetCounterState(int $counter): Response
    {
        $route = 'iostate/counterclear/'.$counter;

        return $this->_sendRequest('POST', $route);

    } // resetCounterState()


    /**
     *
     * @param       int $output
     *
     * @param       bool|State $state
     *
     * @return      Response|null
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function sendOutputState(int $output, bool|State $state): Response|null
    {
        if ($this->_Device->hasOutput() === false) {
            return null;
        }
        $route = 'iostate/output/'.$output;

        if ($state instanceof State) {
            $payload = 'Set='.$state->value;
        } else {
            $payload = 'Set='.($state === true ? 'ON' : 'OFF');
        }
        return $this->_sendRequest('POST', $route, $payload);

    } // sendOutputState()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getFakerRouteBase(): string
    {
        return self::$_fakerRouteBase;

    } // getFakerRouteBase()


} // class WebIoConnector {}
