<?php
/**
 * Faker Controller Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-components/web-io-connect
 * @subpackage  Connectors
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 Wassilios Meletiadis <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanComponents\WebIoConnect\Fakers;


use BplanComponents\WebIoConnect\Enums\DeviceType;
use BplanComponents\WebIoConnect\Enums\State;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;


/**
 * Faker Controller Class
 *
 * @version     1.0.0 / 2025-04-26
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class WebIoFaker
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     DeviceType $_DeviceType
     */
    private DeviceType $_DeviceType;


    /**
     * @var     array $_ioState
     */
    private array $_ioState;


    /**
     * @var     Request $_Request
     */
    private Request $_Request;


    /**
     * @var     string $_storageFile
     */
    private string $_storageFile;


    /**
     * @var     string $_storagePath
     */
    private string $_storagePath;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * Wenn $_devMode TRUE ist, dann wird kein spezieller Relais-Typ verwendet und
     * es wird dafür gesorgt, dass immer "counter", "input" und "output" in den
     * Testdaten verfügbar sind.
     *
     * @var     bool $_devMode
     */
    private static bool $_devMode = false;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param 	    DeviceType $DeviceType
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(DeviceType $DeviceType)
    {
        $this->_DeviceType = $DeviceType;

        $this->_init();

    } // __construct()


    /**
     *
     * @return 	    void
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __destruct()
    {
        $contents = '<?'.'php'
            ."\n\n"
            .'return'
            .' '.var_export($this->_ioState, true)
            .';'
            ."\n";

        file_put_contents($this->_storageFile, $contents);

    } // __destruct()


    /**
     *
     * @return 	    array
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _buildIoState(): array
    {
        $deviceClass = $this->_DeviceType->getClass();

        $ioState = [];

        if (($inputCount = $deviceClass::getInputCount()) > 0 || self::$_devMode === true) {
            if (self::$_devMode === true && $inputCount === 0) {
                $inputCount = $deviceClass::getOutputCount();
            }
            $ioState = [
                'counter' => [],
                'input' => [],
            ];
            for ($i = 0; $i < $inputCount; $i++) {
                $ioState['counter'][] = [
                    'number' => $i,
                    'state' => rand(0, 100),
                ];
                $ioState['input'][] = [
                    'number' => $i,
                    'state' => rand(0, 1),
                ];
            }
        }
        if (($outputCount = $deviceClass::getOutputCount()) > 0 || self::$_devMode === true) {
            if (self::$_devMode === true && $outputCount === 0) {
                $outputCount = $deviceClass::getInputCount();
            }
            $ioState['output'] = [];

            for ($i = 0; $i < $outputCount; $i++) {
                $ioState['output'][] = [
                    'number' => $i,
                    'state' => rand(0, 1), // Random ON/OFF
                ];
            }
        }
        return $ioState;

    } // _buildIoState()


    /**
     *
     * @return 	    $this
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(): self
    {
        $this->_Request = request();

        $this->_storagePath = storage_path('app/tmp/WebIoFaker');

        if (self::$_devMode === true) {
            $this->_storageFile = $this->_storagePath.'/iostate.full.php';
        } else {
            $this->_storageFile = $this->_storagePath.'/iostate.'.$this->_DeviceType->value.'.php';
        }
        if (!file_exists($this->_storagePath)) {
            mkdir($this->_storagePath, 0777, true);
        }
        if (file_exists($this->_storageFile)) {
            $this->_ioState = require $this->_storageFile;

        } else {
            $this->_ioState = $this->_buildIoState();
        }
        return $this;

    } // _init()


    /**
     * Setzt einen Counter zurück
     *
     * @param       int $counter
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function clearCounter(int $counter): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'counter.'.$counter) === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        Arr::set($this->_ioState, 'counter.'.$counter.'.state', 0);

        return $this->getCounter($counter);

    } // clearCounter()


    /**
     * Liefert die Daten eines einzelnen Counters
     *
     * @param       int $counter
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getCounter(int $counter): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'counter.'.$counter) === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        return [
            'iostate' => [
                'counter' => [
                    'number' => $counter,
                    'state' => Arr::get($this->_ioState, 'counter.'.$counter.'.state')
                ]
            ]
        ];
    } // getCounter()


    /**
     * Liefert die Daten aller Counters
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getCounters(): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'counter') === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        return [
            'iostate' => [
                'counter' => Arr::get($this->_ioState, 'counter')
            ]
        ];
    } // getCounters()


    /**
     * Liefert die Daten eines einzelnen Inputs
     *
     * @param       int $input
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getInput(int $input): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'input.'.$input) === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        return [
            'iostate' => [
                'output' => [
                    'number' => $input,
                    'state' => Arr::get($this->_ioState, 'input.'.$input.'.state')
                ]
            ]
        ];
    } // getInput()


    /**
     * Liefert die Daten aller Inputs
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getInputs(): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'input') === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        return [
            'iostate' => [
                'input' => Arr::get($this->_ioState, 'input')
            ]
        ];
    } // getInputs()


    /**
     * Liefert die Daten eines einzelnen Outputs
     *
     * @param       int $output
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getOutput(int $output): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'output.'.$output) === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        return [
            'iostate' => [
                'output' => [
                    'number' => $output,
                    'state' => Arr::get($this->_ioState, 'output.'.$output.'.state')
                ]
            ]
        ];
    } // getOutput()


    /**
     * Liefert die Daten aller Outputs
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getOutputs(): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'output') === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        return [
            'iostate' => [
                'output' => Arr::get($this->_ioState, 'output')
            ]
        ];
    } // getOutputs()


    /**
     * Liefert ein vollständiges Prozessabbild
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getProcessImage(): array
    {
        return [
            'info' => [
                'request' => $this->_Request->getRequestUri(),
                'time' => Carbon::now()->format('Y-m-d,H:i:s'),
                'ip' => $this->_Request->ip(),
                'devicename' => $this->_DeviceType->name,
            ],
            'iostate' => $this->_ioState,
            'system' => [
                'time' => Carbon::now()->format('Y-m-d H:i:s'),
                'diagnosis' => [
                    [
                        'time' => Carbon::now()->format('d.m.Y H:i:s'),
                        'msg' => 'Gerätestatus: OK',
                    ],
                ],
                'diagarchive' => [
                    [
                        'time' => Carbon::now()->format('d.m.Y H:i:s'),
                        'msg' => 'Gerätestatus: OK',
                    ],
                ],
            ],
        ];
    } // getProcessImage()


    /**
     * Setzt den Status für ein Output
     *
     * @param       int $output
     *
     * @return      array|JsonResponse
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function setOutput(int $output, $state): array|JsonResponse
    {
        if (Arr::has($this->_ioState, 'output.'.$output) === false) {
            return response()->json(['message' => 'Not Found!'], 404);
        }
        Arr::set($this->_ioState, 'output.'.$output.'.state', $state === State::On->value ? 1 : 0);

        return $this->getOutput($output);

    } // setOutput()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function isDevMode(): bool
    {
        return self::$_devMode;

    } // isDevMode()


} // class WebIoFaker {}
