<?php
/**
 * Connector Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-components/web-io-connect
 * @subpackage  Connectors
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 Wassilios Meletiadis <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanComponents\WebIoConnect\Connectors;

use BplanBase\Globals\Enums\LogLevel;
use BplanBase\Globals\Services\LogService;
use BplanComponents\WebIoConnect\Api\Credentials;
use BplanComponents\WebIoConnect\Devices\BaseDevice;
use BplanComponents\WebIoConnect\Enums\ResponseFormat;
use BplanComponents\WebIoConnect\Enums\DeviceType;
use BplanComponents\WebIoConnect\Enums\State;
use Exception;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;


/**
 * Connector Class
 *
 * @version     2.1.0 / 2025-07-01
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
class WebIoConnector
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         null|string $_authPass
     */
    protected null|string $_authPass = null;


    /**
     * @var         string $_authUser
     */
    protected string $_authUser;


    /**
     * @var         string $_baseUrl
     */
    protected string $_baseUrl;


    /**
     * @var         BaseDevice $_Device
     */
    protected BaseDevice $_Device;


    /**
     * @var         DeviceType $_DeviceType
     */
    protected DeviceType $_DeviceType;


    /**
     * @var         LogService $_LogService
     */
    protected LogService $_LogService;


    /**
     * @var         string $_responseFormat
     */
    protected string $_responseFormat;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     */
    private static string $_fakerRouteBase = 'web-io-faker';


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       DeviceType $DeviceType
     *
     * @param       string $baseUrl
     *
     * @param       ResponseFormat $ResponseFormat
     *              Aktuell wird nur das ResponseFormat "Json" unterstützt.
     *
     * @version     1.2.0 / 2025-07-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(DeviceType $DeviceType, string $baseUrl, $ResponseFormat = ResponseFormat::Json)
    {
        /*
        **  Zum Beispiel "http://webio.example.com:8080" */
        $this->_baseUrl = $baseUrl;
        $this->_DeviceType = $DeviceType;
        $this->_responseFormat = $ResponseFormat->value;

        $this->_LogService = new LogService();

        $this->_init();

    } // __construct()


    /**
     *
     * @return 	    $this
     *
     * @version     1.1.0 / 2025-04-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _init(): self
    {
        if (strtolower(config('app.env')) !== 'production' && config('web-io-connect.force-real-api') === false) {
            $this->_baseUrl = request()->getSchemeAndHttpHost()
                .'/api/'.self::$_fakerRouteBase
                .'/'.$this->_DeviceType->value;
        }
        $Credentials = new Credentials();

        if ($Credentials->hasCredentials() === true) {
            $this->_authPass = $Credentials->getPassword();
            $this->_authUser = $Credentials->getUserName();
        }
        $response = $this->readProcessImage('GET');

        $class = $this->_DeviceType->getClass();

        $this->_Device = new $class($response);

        return $this;

    } // _init()


    /**
     *
     * @param       string $method
     *
     * @param       string $route
     *
     * @param       string $payload
     *
     * @throws      Exception
     *
     * @return      array
     *
     * @version     1.1.0 / 2025-07-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    private function _sendRequest(string $method, null|string $route = null, null|string $payload = null): array
    {
        $url = $this->_baseUrl.'/rest/'.$this->_responseFormat;

        if ($route !== null) {
            $url .= '/'.$route;
        }
        $Request = Http::timeout(5);

        if ($this->_authPass !== null) {
            $Request = $Request->withDigestAuth($this->_authUser, $this->_authPass);
        }
        if ($method === 'POST') {
            return $Request->withBody($payload)->post($url)->json();
        }
        $response = $Request->get($url)->json();

        if ($response === null) {
            throw new Exception('Something went wrong when trying to contact the API (using authentication: '.($this->_authPass === null ? 'no' : 'yes').').');
        }
        $this->_LogService->log(LogLevel::Info, 'Sent request ['.$method.' '.$url.'] to device ['.$this->_DeviceType->value.'].');

        return $response;

    } // _sendRequest()


    /**
     * Liefert den State eines Inputs
     *
     * @param       int $output
     *
     * @param       bool $returnBool
     *
     * @return      bool|State|null
     *
     * @version     1.0.1 / 2025-04-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readCounterValue(int $counter, bool $returnBool = false): bool|State|null
    {
        if ($this->_Device->hasCounter($counter) === false) {
            return null;
        }
        $route = 'iostate/counter/'.$counter;

        $result = $this->_sendRequest('GET', $route);

        $bool = (bool) $result['iostate']['counter']['state'];

        if ($returnBool === true) {
            return $bool;
        }
        return ($bool === true) ? State::On : State::Off;

    } // readCounterValue()


    /**
     * Liefert den State eines Inputs
     *
     * @param       int $output
     *
     * @param       bool $returnBool
     *
     * @return      bool|State|null
     *
     * @version     1.0.1 / 2025-04-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readInputState(int $input, bool $returnBool = false): bool|State|null
    {
        if ($this->_Device->hasInput($input) === false) {
            return null;
        }
        $route = 'iostate/input/'.$input;

        $result = $this->_sendRequest('GET', $route);

        $bool = (bool) $result['iostate']['input']['state'];

        if ($returnBool === true) {
            return $bool;
        }
        return ($bool === true) ? State::On : State::Off;

    } // readInputState()


    /**
     * Liest das vollständige Prozessabbild aus
     *
     * @return      array|null
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readProcessImage(): array|null
    {
        return $this->_sendRequest('GET');

    } // readProcessImage()


    /**
     * Liefert den State eines Outputs
     *
     * @param       int $output
     *
     * @param       bool $returnBool
     *
     * @return      bool|State|null
     *
     * @version     1.0.1 / 2025-04-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function readOutputState(int $output, bool $returnBool = false): bool|State|null
    {
        if ($this->_Device->hasOutput($output) === false) {
            return null;
        }
        $route = 'iostate/output/'.$output;

        $result = $this->_sendRequest('GET', $route);

        $bool = (bool) $result['iostate']['output']['state'];

        if ($returnBool === true) {
            return $bool;
        }
        return ($bool === true) ? State::On : State::Off;

    } // readOutputState()


    /**
     *
     * @param       int $input
     *
     * @return      Response
     *
     * @version     1.0.1 / 2025-07-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function resetCounter(int $counter): array
    {
        $route = 'iostate/counterclear/'.$counter;

        return $this->_sendRequest('POST', $route);

    } // resetCounter()


    /**
     *
     * @param       int $output
     *
     * @param       bool|State $state
     *
     * @return      array|null
     *
     * @version     1.0.1 / 2025-07-01
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function sendOutputState(int $output, bool|State $state): array|null
    {
        if ($this->_Device->hasOutput() === false) {
            return null;
        }
        $route = 'iostate/output/'.$output;

        if ($state instanceof State) {
            $payload = 'Set='.$state->value;
        } else {
            $payload = 'Set='.($state === true ? 'ON' : 'OFF');
        }
        return $this->_sendRequest('POST', $route, $payload);

    } // sendOutputState()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return      string
     *
     * @version     1.0.0 / 2025-
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getFakerRouteBase(): string
    {
        return self::$_fakerRouteBase;

    } // getFakerRouteBase()


} // class WebIoConnector {}
