<?php
/**
 * Base Device Class
 *
 * @version     1.0.$Revision:$
 * @version     SVN: $Id:$
 * @package     bplan-components/web-io-connect
 * @subpackage  Devices
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 * @copyright   Copyright (C) 2025 Wassilios Meletiadis <https://www.bplan-solutions.de/>
 * /Δ\
 */

namespace BplanComponents\WebIoConnect\Devices;


use BplanComponents\WebIoConnect\Enums\ContactType;
use BplanComponents\WebIoConnect\Devices\Io\Counter;
use BplanComponents\WebIoConnect\Devices\Io\Input;
use BplanComponents\WebIoConnect\Devices\Io\Output;
use Illuminate\Support\Collection;


/**
 * Base Device Class
 *
 * @version     1.1.0 / 2025-04-27
 * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
 */
abstract class BaseDevice
{


/* +++ TRAITS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ CLASS CONSTANTS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++ */


/* +++ OBJECT MEMBERS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var         Collection $_Counters
     */
    protected Collection $_Counters;


    /**
     * @var         array $_info
     */
    protected array $_info;


    /**
     * @var         Collection $_Inputs
     */
    protected Collection $_Inputs;


    /**
     * @var         Collection $_Outputs
     */
    protected Collection $_Outputs;


    /**
     * @var         array $_system
     */
    protected array $_system;


/* +++ CLASS MEMBERS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     * @var     array $_coContacts
     */
    protected static array $_coContacts = [];


    /**
     * @var         int $_inputCount
     */
    protected static int $_inputCount = 0;


    /**
     * @var     array $_noContacts
     */
    protected static array $_noContacts = [];


    /**
     * @var         int $_outputCount
     */
    protected static int $_outputCount = 0;


/* +++ OBJECT METHODS +++ +++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @param       array|null $response
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function __construct(array|null $response)
    {
        $this->_info = $response['info'] ?? [];
        $this->_system = $response['system'] ?? [];

        $this->_Counters = collect($response['iostate']['counter'] ?? [])->map(fn($counter) => new Counter($counter['number'], $counter['state']));
        $this->_Inputs = collect($response['iostate']['input'] ?? [])->map(fn($input) => new Input($input['number'], $input['state']));
        $this->_Outputs = collect($response['iostate']['output'] ?? [])->map(fn($output) => new Output($output['number'], $output['state']));

    } // __construct()


    /**
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getCounters(): Collection
    {
        return $this->_Counters;

    } // getCounters()


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getInfo(): array
    {
        return $this->_info;
    }


    /**
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getInputs(): Collection
    {
        return $this->_Inputs;

    } // getInputs()


    /**
     *
     * @return      Collection
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getOutputs(): Collection
    {
        return $this->_Outputs;

    } // getOutputs()


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getRelays(): array
    {
        /*
        **  Relais können entweder NO oder CO sein, hier wird für jedes Relais
        **  die passende Klasse zurückgegeben. */
        return [
            ContactType::ChangeOver->value => $this->getOutputs()->filter(fn($Output) => $Output instanceof Output && in_array($Output->getNumber(), static::$_coContacts))->values(),
            ContactType::NormallyOpen->value => $this->getOutputs()->filter(fn($Output) => $Output instanceof Output && in_array($Output->getNumber(), static::$_noContacts))->values(),
        ];
    } // getRelays()


    /**
     *
     * @return      array
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function getSystem(): array
    {
        return $this->_system;

    } // getSystem()


    /**
     *
     * @return      bool
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasCounter(int|null $counter = null): bool
    {
        return $this->hasInput($counter);

    } // hasCounter()


    /**
     *
     * @return      bool
     *
     * @version     1.0.1 / 2025-04-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasInput(int|null $input = null): bool
    {
        if ($input > (static::$_inputCount - 1)) {
            return false;
        }
        return static::$_inputCount > 0;

    } // hasInput()


    /**
     *
     * @return      bool
     *
     * @version     1.0.1 / 2025-04-27
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public function hasOutput(int|null $output = null): bool
    {
        if ($output > (static::$_outputCount - 1)) {
            return false;
        }
        return static::$_outputCount > 0;

    } // hasOutput()


/* +++ CLASS METHODS +++ ++++++++++++++++++++++++++++++++++++++++++++++++++++ */


    /**
     *
     * @return      int
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getInputCount(): int
    {
        return static::$_inputCount;

    } // getInputCount()


    /**
     *
     * @return      int
     *
     * @version     1.0.0 / 2025-04-26
     * @author      Wassilios Meletiadis <wassilios.meletiadis@bplan-solutions.de>
     */
    public static function getOutputCount(): int
    {
        return static::$_outputCount;

    } // getOutputCount()


} // abstract class BaseDevice {}
